/*=============================================================================
    The Production rule

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

======================================================================================*/
#ifndef SPIRIT_RULE_HPP
#define SPIRIT_RULE_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/parser.hpp"
#ifdef SPIRIT_DEBUG
#include <string>
#endif

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

namespace impl {

    template <typename IteratorT> class abstract_parser;
}

///////////////////////////////////////////////////////////////////////////////
struct none_t;  //  Forward declaration

///////////////////////////////////////////////////////////////////////////////
//
//  rule class
//
//      The rule is a polymorphic variable that acts as a named place-
//      holder capturing the behavior of an EBNF expression assigned to
//      it. Naming an EBNF expression allows it to be referenced later
//      on by another EBNF expression. Rules may be forward declared.
//      This enables rules to be recursive. A rule may even reference
//      itself. The limitation is that direct or indirect left recursion
//      is not allowed.
//
//      The rule class models EBNF's production rule. Example:
//
//          Rule<>   a_rule = *(a | b) & +(c | d | e);
//
//      The type and functionality of the right-hand expression, which
//      may be arbitrarily complex, is encoded in the rule named a_rule.
//      a_rule may now be referenced.
//
//      Rules may be declared before it is defined to allow cyclic
//      structures typically found in EBNF declarations. An undefined
//      rule matches nothing. Multiple definitions of a rule are taken
//      as alternatives. Example, the definition:
//
//          r = a;
//          r = b;
//
//      behaves as:
//
//          r = a | b;
//
//      When a rule is invoked by an EBNF expression, the rule is held
//      by the expression by reference. It is the responsibility of the
//      client to ensure that the referenced rule stays in scope and
//      does not get destructed while it is being referenced.
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT>
class rule : public parser<rule<IteratorT> > {

public:

    typedef IteratorT   iterator_type;

    rule();
    ~rule();
    rule(rule const& other);

    rule&
    operator = (rule const& other);

    match
    parse(IteratorT& first, IteratorT const& last) const;

    template <typename ParserT>
    rule(ParserT const& parser);

    template <typename ParserT>
    rule&
    operator=(ParserT const& parser);

#ifdef SPIRIT_DEBUG
    void
    debug_name(char const* name);
#endif

private:

    impl::abstract_parser<IteratorT>* meta;

#ifdef SPIRIT_DEBUG
    std::string name;
#endif
};

///////////////////////////////////////////////////////////////////////////////
//
//  skipper class
//
//      Utility class that facilitates the skipping of characters.
//      The single member function 'skip' enters a loop incrementing
//      the 'current' iterator until the supplied skip rule fails
//      to match.
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT = char const*>
class skipper {

public:

    template <typename ParserT>
    skipper(ParserT const& skip_rule, IteratorT const& last);
    skipper(IteratorT const& last);

    void
    skip(IteratorT& current) const;

private:

    IteratorT           last;
    rule<IteratorT>     skip_rule;
};

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
