/*=============================================================================
    Semantic actions

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_ACTION_HPP
#define SPIRIT_ACTION_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/composite.hpp"
#include "boost/spirit/parser.hpp"
#include "boost/spirit/spirit_fwd.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  action class
//
//      Links a parser with a user defined semantic action. The semantic
//      action may be a function or a functor. A function should be compatible
//      with the interface:
//
//          void f(IteratorT first, IteratorT last);
//
//      Where first points to the current input and last points to one after
//      the end of the input (same as STL algorithms).
//
//      A functor should have a member operator() with a compatible signature
//      as above. Iterators pointing to the matching portion of the input are
//      passed into the function/functor.
//
//      This is the default class that parsers (see parser::operator[]
//      <in parser.hpp>) use when dealing with the construct:
//
//          p[f]
//
//      where p is a parser and f is a function or functor. Sometimes,
//      when appropriate, some parsers overload the parser's operator[]
//      (see char_parser in <primitives.hpp> and xx_parser_gen(s) in
//      <numerics.hpp>).
//
///////////////////////////////////////////////////////////////////////////////
template <typename ParserT, typename ActionT>
class action
:   public unary<ParserT>,
    public parser<action<ParserT, ActionT> > {

public:

    action(ParserT const& parser, ActionT const& actor);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;

private:

    ActionT  actor;
};

///////////////////////////////////////////////////////////////////////////////
//
//  reference_wrapper class
//
//      Wraps a reference to a variable of type T. This class also does
//      the dual purpose of acting like a functor that is compatible
//      with the action class above, that expects an interface:
//
//          void operator()(IteratorT begin, IteratorT end);
//
//      When this operator is invoked, the reference_wrapper expects that
//      the referenced object is a container of some sort (e.g. std::string,
//      std::vector) that may be constructed given two iterators, and has
//      a member function swap, that swaps the contents of two containers.
//
//      The reference_wrapper handles the construct:
//
//          p[ref(var)]
//
//      where p is a parser and var is a variable that will hold the parsed
//      result.
//
//          ref(var)
//
//      is a generator that creates a reference_wrapper.
//
//      There is also an alternative:
//
//          void operator()(T val);
//
//      provided. This may be used by other action classes (apart from
//      the one above) that expect this interface (see char_action in
//      <primitives.hpp> and numeric_action in <numerics.hpp>).
//
///////////////////////////////////////////////////////////////////////////////
template <typename T>
class reference_wrapper {

public:

    typedef T element_type;

    explicit
    reference_wrapper(T& ref);

    operator T&() const;

    void
    operator()(T const& val) const;

    template <typename IteratorT>
    void
    operator()(IteratorT const& begin, IteratorT const& end) const;

private:

    T&  ref;
};

///////////////////////////////////////////////////////////////////////////////
//
//  ref generator (Generate a reference_wrapper object)
//
///////////////////////////////////////////////////////////////////////////////
template <class T>
reference_wrapper<T> const
ref(T& t);

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
