/*=============================================================================
    Parser directives

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

======================================================================================*/
#ifndef SPIRIT_DIRECTIVES_HPP
#define SPIRIT_DIRECTIVES_HPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/composite.hpp"
#include "boost/spirit/parser.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////

template <typename CharT>           class chlit;
template <typename StringT>         class strlit;
template <typename CharT>           class cstring;
template <typename A, typename B>   struct alternative;

///////////////////////////////////////////////////////////////////////////////
//
//  contiguous class
//
//      This class handles the directive:
//
//          lexeme[p]
//
//      where p is a parser. Its function parse() strips the scanner, which
//      skips white spaces, off its iterator before delegating the actual
//      parsing to its subject. The lexeme_parser_gen with its single instance
//      lexeme is a generator that creates a contiguous object given a parser.
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
struct contiguous
:   public unary<S>,
    public parser<contiguous<S> > {

    contiguous(S const& a);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

//////////////////////////////////
struct lexeme_parser_gen {

    template <typename S>
    contiguous<S>
    operator[](parser<S> const& subject) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return contiguous<S>(subject.derived());
    }
};

const lexeme_parser_gen lexeme = lexeme_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  inhibit_case class
//
//      This class handles the directive;
//
//      nocase[p]
//
//      where p is a parser. Its function parse() wraps the iterator in
//      a nocase_iterator that converts characters from the input to
//      lowercase (see nocase_iterator in <iterators.hpp>) before delegating
//      the actual parsing to its subject. The inhibit_case_parser_gen is
//      a generator that creates an inhibit_case object given a parser.
//      Overloads are also provided for literal strings and characters.
//      Examples:
//
//          nocase['x']
//          nocase["begin"].
//
//      It is important to note that only the input is converted to lower
//      case. The parser will fail if it expects any upper case characters.
//      from the input. Example: nocase['X'] will never succeed because
//      it expects an upper case 'X' that the nocase_iterator will never
//      produce.
//
///////////////////////////////////////////////////////////////////////////////
template <typename S>
struct inhibit_case
:   public unary<S>,
    public parser<inhibit_case<S> > {

    inhibit_case(S const& a);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

//////////////////////////////////
struct inhibit_case_parser_gen {

    template <typename S>
    inhibit_case<S>
    operator[](parser<S> const& subject) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return inhibit_case<S>(subject.derived());
    }

    inhibit_case<chlit<char> >
    operator[](char ch) const;

    inhibit_case<chlit<wchar_t> >
    operator[](wchar_t ch) const;

    inhibit_case<strlit<cstring<char> > >
    operator[](char const* str) const;

    inhibit_case<strlit<cstring<wchar_t> > >
    operator[](wchar_t const* str) const;
};

const inhibit_case_parser_gen nocase = inhibit_case_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  longest_alternative class
//
//      This class handles the directive:
//
//          longest[a | b | c | ... ]
//
//      where a, b and c are parsers. All alternatives (which does short-
//      circuit evaluation) are converted to longest_alternative(s) where
//      the alternative that matches the longest part of the input (maximal
//      munch) wins.
//
//      The class to_longest_alternative and its specializations extract
//      all the alternative(s) and converts these to longest_alternative(s).
//      The class longest_parser_gen with its sole instance longest is a
//      generator that creates a longest_alternative object given a parser
//      composite comprising of alternatives.
//
///////////////////////////////////////////////////////////////////////////////

template <typename A, typename B>
struct longest_alternative
:   public binary<A, B>,
    public parser<longest_alternative<A, B> > {

    longest_alternative(A const& a, B const& b);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

//////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename T>
    struct to_longest_alternative;

    //////////////////////////////////
    template <typename A, typename B>
    struct to_longest_alternative<alternative<A, B> >;
}

//////////////////////////////////
struct longest_parser_gen {

    template <typename A, typename B>
    typename impl::to_longest_alternative<alternative<A, B> >::return_t
    operator[](alternative<A, B> const& alt) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return impl::to_longest_alternative<alternative<A, B> >::convert(alt);
    }
};

const longest_parser_gen longest = longest_parser_gen();

///////////////////////////////////////////////////////////////////////////////
//
//  shortest_alternative class
//
//      This class handles the directive:
//
//          shortest[a | b | c | ... ]
//
//      where a, b and c are parsers. All alternatives (which does short-
//      circuit evaluation) are converted to shortest_alternative(s) where
//      the alternative that matches the shortest part of the input (minimal
//      munch) wins.
//
//      The class to_shortest_alternative and its specializations extract
//      all the alternative(s) and converts these to shortest_alternative(s).
//      The class shortest_parser_gen with its sole instance shortest is a
//      generator that creates a shortest_alternative object given a parser
//      composite comprising of alternatives.
//
///////////////////////////////////////////////////////////////////////////////
template <typename A, typename B>
struct shortest_alternative
:   public binary<A, B>,
    public parser<shortest_alternative<A, B> > {

    shortest_alternative(A const& a, B const& b);

    template <typename IteratorT>
    match
    parse(IteratorT& first, IteratorT const& last) const;
};

//////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename T>
    struct to_shortest_alternative;

    //////////////////////////////////
    template <typename A, typename B>
    struct to_shortest_alternative<alternative<A, B> >;

} // namespace impl

//////////////////////////////////
struct shortest_parser_gen {

    template <typename A, typename B>
    typename impl::to_shortest_alternative<alternative<A, B> >::return_t
    operator[](alternative<A, B> const& alt) const
    {
        //  Borland 5.5 reports an internal compiler
        //  error if this is not defined here.
        return impl::to_shortest_alternative<alternative<A, B> >::convert(alt);
    }
};

const shortest_parser_gen shortest = shortest_parser_gen();

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
