/*=============================================================================
    Character set implementation

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_CHSET_IPP
#define SPIRIT_CHSET_IPP

///////////////////////////////////////////////////////////////////////////////

#include <cassert>
#include <vector>
#include <algorithm>
#include "boost/limits.hpp"
#include "boost/spirit/chset.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

namespace impl {

///////////////////////////////////////////////////////////////////////////////
struct chset_converter;

///////////////////////////////////////////////////////////////////////////////
//
//  range class
//
//      Implements a closed range of values. This class is used in
//      the implementation of the range_run class.
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
struct range {
                    range(CharT first, CharT last);

    bool            is_valid() const;
    bool            includes(CharT v) const;
    bool            includes(range const& r) const;
    bool            is_adjacent(range const& r) const;
    void            merge(range const& r);

    bool            operator<(range const& r) const;
    bool            operator<(CharT v) const;

    CharT first, last;
};

///////////////////////////////////////////////////////////////////////////////
//
//  range_run
//
//      An implementation of a sparse bit (boolean) set. The set uses a
//      sorted vector of disjoint ranges. This class implements the bare
//      minimum essentials from which the full range of set operators
//      can be implemented. The set is constructed from ranges.
//      Internally, adjacent or overlapping ranges are coalesced.
//
//      range_runs are very space-economical in situations where there
//      are lots of ranges and a few individual disjoint values.
//      Searching is O(log n) where n is the number of ranges.
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
class range_run {

public:

    typedef typename std::vector<range<CharT> >::iterator iterator;
    typedef typename std::vector<range<CharT> >::const_iterator const_iterator;

    bool            test(CharT v) const;
    void            set(range<CharT> const& r);
    void            clear(range<CharT> const& r);
    void            clear();

    const_iterator  begin() const;
    const_iterator  end() const;

private:

    void            merge(iterator iter, range<CharT> const& r);

    friend class chset_converter;
    std::vector<range<CharT> > run;
};

///////////////////////////////////////////////////////////////////////////////
//
//  range class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline range<CharT>::range(CharT first_, CharT last_)
: first(first_), last(last_) {}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::is_valid() const
{
    return first <= last;
}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::includes(range const& r) const
{
    return (first <= r.first) && (last >= r.last);
}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::includes(CharT v) const
{
    return (first <= v) && (last >= v);
}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::is_adjacent(range const& r) const
{
    CharT   decr_first = first == std::numeric_limits<CharT>::min() ? first : first-1;
    CharT   incr_last = last == std::numeric_limits<CharT>::max() ? last : last+1;

    return ((decr_first <= r.first) && (incr_last >= r.first))
        || ((decr_first <= r.last) && (incr_last >= r.last));
}

//////////////////////////////////
template <typename CharT>
inline void
range<CharT>::merge(range const& r)
{
    first = std::min(first, r.first);
    last = std::max(last, r.last);
}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::operator < (range const& r) const
{
    return first < r.first;
}

//////////////////////////////////
template <typename CharT>
inline bool
range<CharT>::operator < (CharT v) const
{
    return first < v;
}

///////////////////////////////////////////////////////////////////////////////
//
//  range_run class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline bool
range_run<CharT>::test(CharT v) const
{
    if (!run.empty())
    {
        const_iterator iter = std::lower_bound(run.begin(), run.end(), 
            range<CharT>(v,v));
        if (iter != run.end() && iter->includes(v))
            return true;
        if (iter != run.begin())
            return (--iter)->includes(v);
    }
    return false;
}

//////////////////////////////////
template <typename CharT>
void
range_run<CharT>::merge(iterator iter, range<CharT> const& r)
{
    iter->merge(r);
    iterator i = iter + 1;

    while (i != run.end() && iter->is_adjacent(*i))
        iter->merge(*i++);

    run.erase(iter+1, i);
}

//////////////////////////////////
template <typename CharT>
void
range_run<CharT>::set(range<CharT> const& r)
{
    assert(r.is_valid());
    if (!run.empty())
    {
        iterator iter = std::lower_bound(run.begin(), run.end(), r);
        if (iter != run.end() && iter->includes(r) ||
            ((iter != run.begin()) && (iter - 1)->includes(r)))
            return;

        if (iter != run.begin() && (iter - 1)->is_adjacent(r))
            merge(--iter, r);

        else if (iter != run.end() && iter->is_adjacent(r))
            merge(iter, r);

        else
            run.insert(iter, r);
    }
    else
    {
        run.push_back(r);
    }
}

//////////////////////////////////
template <typename CharT>
void
range_run<CharT>::clear(range<CharT> const& r)
{
    assert(r.is_valid());
    if (!run.empty())
    {
        iterator iter = std::lower_bound(run.begin(), run.end(), r);
        iterator left_iter;

        if ((iter != run.begin()) && (left_iter = (iter - 1))->includes(r.first))
            if (left_iter->last > r.last)
            {
                int save_last = left_iter->last;
                left_iter->last = r.first-1;
                run.insert(iter, range<CharT>(r.last+1, save_last));
                return;
            }
            else
            {
                left_iter->last = r.first-1;
            }

        iterator i = iter;
        while (i != run.end() && r.includes(*i))
            i++;
        if (i != run.end() && i->includes(r.last))
            i->first = r.last+1;
        run.erase(iter, i);
    }
}

//////////////////////////////////
template <typename CharT>
void
range_run<CharT>::clear()
{
    run.clear();
}

//////////////////////////////////
template <typename CharT>
inline typename range_run<CharT>::const_iterator
range_run<CharT>::begin() const
{
    return run.begin();
}

//////////////////////////////////
template <typename CharT>
inline typename range_run<CharT>::const_iterator
range_run<CharT>::end() const
{
    return run.end();
}

///////////////////////////////////////////////////////////////////////////////
//
//  utility functions
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline range<CharT> const&
full_range()
{
    static range<CharT> full(std::numeric_limits<CharT>::min(),
        std::numeric_limits<CharT>::max());
    return full;
}

///////////////////////////////////////////////////////////////////////////////
}   //  namespace spirit::impl

///////////////////////////////////////////////////////////////////////////////
//
//  chset::impl class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
class chset<CharT>::rep {

public:
    typedef typename std::vector<impl::range<CharT> >::iterator iterator;
    typedef typename std::vector<impl::range<CharT> >::const_iterator const_iterator;

                        rep();
                        rep(rep const& arg);

    bool                test(CharT v) const;
    void                set(impl::range<CharT> const& r);
    void                clear(impl::range<CharT> const& r);

    const_iterator      begin() const;
    const_iterator      end() const;

    static rep*         ref(rep* ptr);
    static void         deref(rep* ptr);
    static void         detach(rep*& ptr);
    static void         detach_clear(rep*& ptr);

#ifndef __GNUC__
private:    //  I can't get G++ to make impl::chset_converter a friend
#endif      //  of this class. Complains that rep_of is private.

    rep&                operator=(rep const& rhs);  //  No definition

    static rep*&        rep_of(chset<CharT>& set);
    static rep const*   rep_of(chset<CharT> const& set);

#ifndef __GNUC__
    friend chset<CharT> operator|<>(chset<CharT> const&, chset<CharT> const&);
    friend chset<CharT> operator-<>(chset<CharT> const&, chset<CharT> const&);
    friend class impl::chset_converter;
#endif

    unsigned uc;
    impl::range_run<CharT> rr;
};

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::rep::rep()
:   uc(1)
{
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::rep::rep(rep const& arg)
:   uc(1)
,   rr(arg.rr)
{
}

//////////////////////////////////
template <typename CharT>
inline bool
chset<CharT>::rep::test(CharT v) const
{
    return rr.test(v);
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::rep::set(impl::range<CharT> const& r)
{
    rr.set(r);
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::rep::clear(impl::range<CharT> const& r)
{
    rr.clear(r);
}

//////////////////////////////////
template <typename CharT>
inline typename chset<CharT>::rep::const_iterator
chset<CharT>::rep::begin() const
{
    return rr.begin();
}

//////////////////////////////////
template <typename CharT>
inline typename chset<CharT>::rep::const_iterator
chset<CharT>::rep::end() const
{
    return rr.end();
}

//////////////////////////////////
template <typename CharT>
inline typename chset<CharT>::rep*
chset<CharT>::rep::ref(rep* ptr)
{
    ptr->uc++;
    return ptr;
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::rep::deref(rep* ptr)
{
    if (--ptr->uc == 0)
        delete ptr;
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::rep::detach(rep*& ptr)
{
    if (ptr->uc > 1)
    {
        rep* t = new rep(*ptr);
        --ptr->uc;
        ptr = t;
    }
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::rep::detach_clear(rep*& ptr)
{
    if (ptr->uc > 1)
    {
        rep* t = new rep;
        --ptr->uc;
        ptr = t;
    }
    else
    {
        ptr->rr.clear();
    }
}

//////////////////////////////////
template <typename CharT>
inline typename chset<CharT>::rep*&
chset<CharT>::rep::rep_of(chset<CharT>& set)
{
    return set.ptr;
}

//////////////////////////////////
template <typename CharT>
inline typename chset<CharT>::rep const*
chset<CharT>::rep::rep_of(chset<CharT> const& set)
{
    return set.ptr;
}

///////////////////////////////////////////////////////////////////////////////
//
//  chset class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset()
:   ptr(new rep)
{
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(chset const& arg)
:   char_parser<chset<CharT> >()
,   ptr(rep::ref(arg.ptr))
{
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(CharT arg)
:   ptr(new rep)
{
    ptr->set(impl::range<CharT>(arg, arg));
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(anychar_)
:   ptr(new rep)
{
    ptr->set(impl::full_range<CharT>());
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(nothing_)
:   ptr(new rep)
{
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(chlit<CharT> const& arg)
:   ptr(new rep)
{
    ptr->set(impl::range<CharT>(arg.ch, arg.ch));
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::chset(range<CharT> const& arg)
:   ptr(new rep)
{
    ptr->set(impl::range<CharT>(arg.first, arg.last));
}

//////////////////////////////////
template <typename CharT>
template <typename CharTB>
chset<CharT>::chset(CharTB const* definition)
:   ptr(new rep)
{
    CharTB ch = *definition++;
    while (ch)
    {
        CharTB next = *definition++;
        if (next == '-')
        {
            next = *definition++;
            if (next == 0)
            {
                ptr->set(impl::range<CharT>(ch, ch));
                ptr->set(impl::range<CharT>('-', '-'));
                break;
            }
            ptr->set(impl::range<CharT>(ch, next));
        }
        else
        {
            ptr->set(impl::range<CharT>(ch, ch));
        }
        ch = next;
    }
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>::~chset()
{
    rep::deref(ptr);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(chset const& rhs)
{
    if (this != &rhs)
    {
        rep* tp = rep::ref(rhs.ptr);
        rep::deref(ptr);
        ptr = tp;
    }
    return *this;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(CharT rhs)
{
    rep::detach_clear(ptr);
    ptr->set(impl::range<CharT>(rhs, rhs));
    return *this;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(anychar_ rhs)
{
    rep::detach_clear(ptr);
    ptr->set(impl::full_range<CharT>());
    return *this;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(nothing_ rhs)
{
    rep::detach_clear(ptr);
    return *this;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(chlit<CharT> const& rhs)
{
    rep::detach_clear(ptr);
    ptr->set(impl::range<CharT>(rhs.ch, rhs.ch));
    return *this;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>&
chset<CharT>::operator=(range<CharT> const& rhs)
{
    rep::detach_clear(ptr);
    ptr->set(impl::range<CharT>(rhs.first, rhs.last));
    return *this;
}

////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::set(range<CharT> const& arg)
{
    rep::detach(ptr);
    ptr->set(impl::range<CharT>(arg.first, arg.last));
}

//////////////////////////////////
template <typename CharT>
inline void
chset<CharT>::clear(range<CharT> const& arg)
{
    rep::detach(ptr);
    ptr->clear(impl::range<CharT>(arg.first, arg.last));
}

////////////////////////////////
template <typename CharT>
inline bool
chset<CharT>::test(CharT ch) const
{
    return ptr->test(ch);
}

///////////////////////////////////////////////////////////////////////////////
namespace impl {

    struct chset_converter {

        template <typename CharTA, typename CharTB>
        static void
        convert(chset<CharTA>& dest, chset<CharTB> const& src)
        {
            typedef typename std::vector<impl::range<CharTA> > vector_a;
            typedef typename std::vector<impl::range<CharTB> > const vector_b;
            typedef typename chset<CharTA>::rep rep_a;
            typedef typename chset<CharTB>::rep rep_b;
            typedef typename impl::range<CharTA> range;

            rep_b const*    sp = rep_b::rep_of(src);
            vector_a&       ss = rep_a::rep_of(dest)->rr.run;

            for (typename vector_b::const_iterator iter = sp->begin();
                iter != sp->end(); ++iter)
                ss.push_back(range(iter->first, iter->last));
        }
    };
}

////////////////////////////////
template <typename CharT>
template <typename CharTB>
inline chset<CharT>::chset(chset<CharTB> const& arg)
:   ptr(new rep)
{
    impl::chset_converter::convert(*this, arg);
}

////////////////////////////////
template <typename CharT>
template <typename CharTB>
inline chset<CharT>&
chset<CharT>::operator=(chset<CharTB> const& rhs)
{
    rep::detach_clear(ptr);
    impl::chset_converter::convert(*this, rhs);
    return *this;
}

///////////////////////////////////////////////////////////////////////////////
//
//  chset free operators implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const& a, chset<CharT> const& b)
{
    typedef typename chset<CharT>::rep      rep;
    typedef typename rep::const_iterator    citerator;

    chset<CharT> a_(a);
    rep*&        ap = rep::rep_of(a_);
    rep const*   bp = rep::rep_of(b);

    rep::detach(ap);
    for (citerator iter = bp->begin(); iter != bp->end(); ++iter)
        ap->set(*iter);
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const& a, chset<CharT> const& b)
{
    typedef typename chset<CharT>::rep      rep;
    typedef typename rep::const_iterator    citerator;

    chset<CharT> a_(a);
    rep*&        ap = rep::rep_of(a_);
    rep const*   bp = rep::rep_of(b);

    rep::detach(ap);
    for (citerator iter = bp->begin(); iter != bp->end(); ++iter)
        ap->clear(*iter);
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator~(chset<CharT> const& a)
{
    return chset<CharT>(anychar) - a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, chset<CharT> const& b)
{
    return a - ~b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, chset<CharT> const& b)
{
    return a - b | b - a;
}

///////////////////////////////////////////////////////////////////////////////
//
//  range <--> chset free operators implementation
//
///////////////////////////////////////////////////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename CharT>
    inline CharT
    decr(CharT v)
    {
        return v == std::numeric_limits<CharT>::min() ? v : v-1;
    }

    //////////////////////////////////
    template <typename CharT>
    inline CharT
    incr(CharT v)
    {
        return v == std::numeric_limits<CharT>::max() ? v : v+1;
    }
}

template <typename CharT>
inline chset<CharT>
operator~(range<CharT> const& a)
{
    chset<CharT> a_;
    a_.set(range<CharT>(std::numeric_limits<CharT>::min(), impl::decr(a.first)));
    a_.set(range<CharT>(impl::incr(a.last), std::numeric_limits<CharT>::max()));
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const& a, range<CharT> const& b)
{
    chset<CharT> a_(a);
    a_.set(b);
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, range<CharT> const& b)
{
    chset<CharT> a_(a);
    a_.clear(range<CharT>(std::numeric_limits<CharT>::min(), impl::decr(b.first)));
    a_.clear(range<CharT>(impl::incr(b.last), std::numeric_limits<CharT>::max()));
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const& a, range<CharT> const& b)
{
    chset<CharT> a_(a);
    a_.clear(b);
    return a_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, range<CharT> const& b)
{
    return a ^ chset<CharT>(b);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(range<CharT> const& a, chset<CharT> const& b)
{
    chset<CharT> b_(b);
    b_.set(a);
    return b_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(range<CharT> const& a, chset<CharT> const& b)
{
    chset<CharT> b_(b);
    b_.clear(range<CharT>(std::numeric_limits<CharT>::min(), impl::decr(a.first)));
    b_.clear(range<CharT>(impl::incr(a.last), std::numeric_limits<CharT>::max()));
    return b_;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(range<CharT> const& a, chset<CharT> const& b)
{
    return chset<CharT>(a) - b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(range<CharT> const& a, chset<CharT> const& b)
{
    return chset<CharT>(a) ^ b;
}

///////////////////////////////////////////////////////////////////////////////
//
//  literal primitives <--> chset free operators implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const& a, CharT b)
{
    return a | range<CharT>(b, b);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, CharT b)
{
    return a & range<CharT>(b, b);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const& a, CharT b)
{
    return a - range<CharT>(b, b);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, CharT b)
{
    return a ^ range<CharT>(b, b);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(CharT a, chset<CharT> const& b)
{
    return range<CharT>(a, a) | b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(CharT a, chset<CharT> const& b)
{
    return range<CharT>(a, a) & b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(CharT a, chset<CharT> const& b)
{
    return range<CharT>(a, a) - b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(CharT a, chset<CharT> const& b)
{
    return range<CharT>(a, a) ^ b;
}

///////////////////////////////////////////////////////////////////////////////
//
//  chlit <--> chset free operators implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator~(chlit<CharT> const& a)
{
    return ~range<CharT>(a.ch, a.ch);
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const& a, chlit<CharT> const& b)
{
    return a | b.ch;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, chlit<CharT> const& b)
{
    return a & b.ch;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const& a, chlit<CharT> const& b)
{
    return a - b.ch;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, chlit<CharT> const& b)
{
    return a ^ b.ch;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chlit<CharT> const& a, chset<CharT> const& b)
{
    return a.ch | b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chlit<CharT> const& a, chset<CharT> const& b)
{
    return a.ch & b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chlit<CharT> const& a, chset<CharT> const& b)
{
    return a.ch - b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chlit<CharT> const& a, chset<CharT> const& b)
{
    return a.ch ^ b;
}

///////////////////////////////////////////////////////////////////////////////
//
//  anychar_ <--> chset free operators
//
//      Where a is chset and b is a anychar_, and vice-versa, implements:
//
//          a | b, a & b, a - b, a ^ b
//
//      Where a is a chlit, implements:
//
//          ~a
//
///////////////////////////////////////////////////////////////////////////////
namespace impl {

    template <typename CharT>
    inline spirit::range<CharT> const&
    full()
    {
        static spirit::range<CharT> full_(std::numeric_limits<CharT>::min(),
            std::numeric_limits<CharT>::min());
        return full_;
    }

    template <typename CharT>
    inline spirit::range<CharT> const&
    empty()
    {
        static spirit::range<CharT> empty_;
        return empty_;
    }
}

//////////////////////////////////
inline nothing_
operator~(anychar_)
{
    return nothing;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const&, anychar_)
{
    return chset<CharT>(impl::full<CharT>());
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, anychar_)
{
    return a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const&, anychar_)
{
    return chset<CharT>();
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, anychar_)
{
    return ~a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(anychar_, chset<CharT> const& b)
{
    return chset<CharT>(impl::full<CharT>());
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(anychar_, chset<CharT> const& b)
{
    return b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(anychar_, chset<CharT> const& b)
{
    return ~b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(anychar_, chset<CharT> const& b)
{
    return ~b;
}

///////////////////////////////////////////////////////////////////////////////
//
//  nothing_ <--> chset free operators implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(chset<CharT> const& a, nothing_)
{
    return a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(chset<CharT> const& a, nothing_)
{
    return impl::empty<CharT>();
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(chset<CharT> const& a, nothing_)
{
    return a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(chset<CharT> const& a, nothing_)
{
    return a;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator|(nothing_, chset<CharT> const& b)
{
    return b;
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator&(nothing_, chset<CharT> const& b)
{
    return impl::empty<CharT>();
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator-(nothing_, chset<CharT> const& b)
{
    return impl::empty<CharT>();
}

//////////////////////////////////
template <typename CharT>
inline chset<CharT>
operator^(nothing_, chset<CharT> const& b)
{
    return b;
}

///////////////////////////////////////////////////////////////////////////////
}   //  namespace spirit

#endif
