/*=============================================================================
    Parser exceptions

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_EXCEPTIONS_IPP
#define SPIRIT_EXCEPTIONS_IPP

///////////////////////////////////////////////////////////////////////////////

#include "boost/spirit/exceptions.hpp"
#include "boost/spirit/primitives.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  parser_error_base class implementation
//
///////////////////////////////////////////////////////////////////////////////
inline parser_error_base::parser_error_base()
{
}

//////////////////////////////////
inline parser_error_base::~parser_error_base()
{
}

//////////////////////////////////
inline parser_error_base::parser_error_base(parser_error_base const&)
{
}

//////////////////////////////////
inline parser_error_base& parser_error_base::operator=(parser_error_base const&)
{
    return *this;
}

///////////////////////////////////////////////////////////////////////////////
//
//  parser_error class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename IteratorT>
inline parser_error<ErrorDescrT, IteratorT>::parser_error(
    IteratorT   where,
    ErrorDescrT what)
:
    iter(where),
    info(what)
{
}

//////////////////////////////////
template <typename ErrorDescrT, typename IteratorT>
inline IteratorT
parser_error<ErrorDescrT, IteratorT>::where() const
{
    return iter;
}

//////////////////////////////////
template <typename ErrorDescrT, typename IteratorT>
inline ErrorDescrT
parser_error<ErrorDescrT, IteratorT>::what() const
{
    return info;
}

//////////////////////////////////
template <typename ErrorDescrT, typename IteratorT>
void
throw_(IteratorT where, ErrorDescrT what)
{
    throw parser_error<ErrorDescrT, IteratorT>(where, what);
}

///////////////////////////////////////////////////////////////////////////////
//
//  assertive_parser class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename ParserT>
inline assertive_parser<ErrorDescrT, ParserT>::assertive_parser(
    ParserT const&  parser,
    ErrorDescrT           what)
:
    unary<ParserT>(parser),
    info(what)
{
}

//////////////////////////////////
template <typename ErrorDescrT, typename ParserT>
template <typename IteratorT>
inline match
assertive_parser<ErrorDescrT, ParserT>::parse(
    IteratorT&          first,
    IteratorT const&    last) const
{
    match hit = this->subject().parse(first, last);
    if (!hit)
        throw parser_error<ErrorDescrT, IteratorT>(first, info);
    return hit;
}

///////////////////////////////////////////////////////////////////////////////
//
//  assertion class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT>
inline assertion<ErrorDescrT>::assertion(ErrorDescrT what)
:   info(what) {}

///////////////////////////////////////////////////////////////////////////////
//
//  fallback_parser class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename ErrorDescrT, typename ParserT, typename HandlerT>
inline fallback_parser<ErrorDescrT, ParserT, HandlerT>::fallback_parser(
    ParserT const&      parser,
    HandlerT const&     handler_)
:
    unary<ParserT>(parser),
    handler(handler_)
{
}

#ifdef __BORLANDC__
////////////////////////////////
//
//  Borland does not like calling the
//  subject directly in the try block.
//  Removing the #ifdef __BORLANDC__
//  code makes Borland complain that the
//  first and last arguments cannot be
//  found in the catch block .
//
//  Weird!
//
////////////////////////////////
namespace impl {

    template <typename ParserT, typename IteratorT>
    match
    subject_parse(
        ParserT const&      subject,
        IteratorT&          first,
        IteratorT const&    last);
}
#endif

////////////////////////////////
template <typename ErrorDescrT, typename ParserT, typename HandlerT>
template <typename IteratorT>
inline match
fallback_parser<ErrorDescrT, ParserT, HandlerT>::parse(
    IteratorT&          first,
    IteratorT const&    last) const
{
    try {
#ifndef __BORLANDC__
        return this->subject().parse(first, last);
#else
        return impl::subject_parse(subject(), first, last);
#endif
    }

    catch (parser_error<ErrorDescrT, IteratorT> error) {
        return handler(first, last, error.where(), error.what());
    }
}

#ifdef __BORLANDC__
////////////////////////////////
namespace impl {

    template <typename ParserT, typename IteratorT>
    match
    subject_parse(
        ParserT const&      subject,
        IteratorT&          first,
        IteratorT const&    last)
    {
        return subject.parse(first, last);
    }
}
#endif

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
