/*=============================================================================
    Iterators

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_ITERATORS_IPP
#define SPIRIT_ITERATORS_IPP

///////////////////////////////////////////////////////////////////////////////

#include <cctype>
#include "boost/spirit/iterators.hpp"

///////////////////////////////////////////////////////////////////////////////

namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  strip_scanner class and specializations
//
///////////////////////////////////////////////////////////////////////////////
namespace impl {

    //////////////////////////////////
    template <typename T>
    inline T
    strip_scanner<T>::get(T ptr)
    {
        return ptr;
    }

    //////////////////////////////////
    template <typename IteratorT, typename SkipT>
    inline IteratorT
    strip_scanner<scanner<IteratorT, SkipT> >::
    get(scanner<IteratorT, SkipT> const& scanner)
    {
        return scanner.iterator();
    }

    //////////////////////////////////
    template <typename IteratorT, typename SkipT>
    inline nocase_iterator<IteratorT>
    strip_scanner<nocase_iterator<scanner<IteratorT, SkipT> > >::
    get(nocase_iterator<scanner<IteratorT, SkipT> > const& nocase_iter)
    {
        return strip_scanner<scanner<IteratorT, SkipT> >::
            get(nocase_iter.iterator());
    }

} // namespace impl

///////////////////////////////////////////////////////////////////////////////
//
//  scanner class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT, typename SkipT>
inline scanner<IteratorT, SkipT>::scanner(
    IteratorT const&    iter_,
    SkipT const*        skip_)
:
    skip(skip_),
    iter(iter_),
    dirty(true)
{
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline scanner<IteratorT, SkipT>&
scanner<IteratorT, SkipT>::operator = (IteratorT const& iter_)
{
    dirty = true;
    iter = iter_;
    return *this;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline void scanner<IteratorT, SkipT>::next() const
{
    if (dirty)
    {
        dirty = false;
        skip->skip(iter);
    }
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline IteratorT
scanner<IteratorT, SkipT>::get() const
{
    next();
    return iter;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline IteratorT
scanner<IteratorT, SkipT>::iterator() const
{
    next();
    dirty = true;
    return iter;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline typename scanner<IteratorT, SkipT>::value_type
scanner<IteratorT, SkipT>::operator*() const
{
    return *get();
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline typename scanner<IteratorT, SkipT>::pointer
scanner<IteratorT, SkipT>::operator->() const
{
    return get();
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline scanner<IteratorT, SkipT>&
scanner<IteratorT, SkipT>::operator++()
{
    ++iter;
    dirty = true;
    return *this;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline scanner<IteratorT, SkipT>
scanner<IteratorT, SkipT>::operator++(int)
{
    scanner sc(*this);
    ++iter;
    dirty = true;
    return sc;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline bool
scanner<IteratorT, SkipT>::operator==(scanner const& other) const
{
    next();
    other.next();
    return iter == other.iter;
}

//////////////////////////////////
template <typename IteratorT, typename SkipT>
inline bool
scanner<IteratorT, SkipT>::operator!=(scanner const& other) const
{
    return !(*this == other);
}

#ifdef tolower
#undef tolower
#endif

///////////////////////////////////////////////////////////////////////////////
//
//  nocase_iterator class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename IteratorT>
inline nocase_iterator<IteratorT>::nocase_iterator(IteratorT const& iter_)
:   iter(iter_),
    next(std::tolower(*iter))
{
}

//////////////////////////////////
template <typename IteratorT>
template <typename IteratorTB>
inline nocase_iterator<IteratorT>&
nocase_iterator<IteratorT>::operator=(nocase_iterator<IteratorTB> const& iter_)
{
    iter = iter_.iterator();
    return *this;
}

//////////////////////////////////
template <typename IteratorT>
inline nocase_iterator<IteratorT>&
nocase_iterator<IteratorT>::operator=(IteratorT const& iter_)
{
    iter = iter_;
    return *this;
}

//////////////////////////////////
template <typename IteratorT>
inline IteratorT
nocase_iterator<IteratorT>::iterator() const
{
    return iter;
}

//////////////////////////////////
template <typename IteratorT>
inline typename nocase_iterator<IteratorT>::reference
nocase_iterator<IteratorT>::operator*()
{
    return next;
}

//////////////////////////////////
template <typename IteratorT>
inline typename nocase_iterator<IteratorT>::pointer
nocase_iterator<IteratorT>::operator->() const
{
    return &next;
}

//////////////////////////////////
template <typename IteratorT>
inline nocase_iterator<IteratorT>&
nocase_iterator<IteratorT>::operator++()
{
    next = std::tolower(*(++iter));
    return *this;
}

//////////////////////////////////
template <typename IteratorT>
inline nocase_iterator<IteratorT>
nocase_iterator<IteratorT>::operator++(int)
{
    nocase_iterator t(*this);
    next = std::tolower(*(++iter));
    return t;
}

//////////////////////////////////
template <typename IteratorT>
inline bool
nocase_iterator<IteratorT>::operator==(nocase_iterator const& other) const
{
    return iter == other.iter;
}

//////////////////////////////////
template <typename IteratorT>
inline bool
nocase_iterator<IteratorT>::operator!=(nocase_iterator const& other) const
{
    return !(*this == other);
}

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif
