/*=============================================================================
    Parser Loops

    Spirit V1.2
    Copyright (c) 2001, Joel de Guzman

    This software is provided 'as-is', without any express or implied
    warranty. In no event will the copyright holder be held liable for
    any damages arising from the use of this software.

    Permission is granted to anyone to use this software for any purpose,
    including commercial applications, and to alter it and redistribute
    it freely, subject to the following restrictions:

    1.  The origin of this software must not be misrepresented; you must
        not claim that you wrote the original software. If you use this
        software in a product, an acknowledgment in the product documentation
        would be appreciated but is not required.

    2.  Altered source versions must be plainly marked as such, and must
        not be misrepresented as being the original software.

    3.  This notice may not be removed or altered from any source
        distribution.

    Acknowledgements:

        Special thanks to Dan Nuffer, John (EBo) David, Chris Uzdavinis,
        and Doug Gregor. These people are most instrumental in steering
        Spirit in the right direction.

        Special thanks also to people who have contributed to the code base
        and sample code, ported Spirit to various platforms and compilers,
        gave suggestions, reported and provided bug fixes. Alexander
        Hirner, Andy Elvey, Bogdan Kushnir, Brett Calcott, Bruce Florman,
        Changzhe Han, Colin McPhail, Hakki Dogusan, Jan Bares, Joseph
        Smith, Martijn W. van der Lee, Raghavendra Satish, Remi Delcos, Tom
        Spilman, Vladimir Prus, W. Scott Dillman, David A. Greene, Bob
        Bailey, Hartmut Kaiser.

        Finally special thanks also to people who gave feedback and
        valuable comments, particularly members of Spirit's Source Forge
        mailing list and boost.org.

    URL: http://spirit.sourceforge.net/

=============================================================================*/
#ifndef SPIRIT_LOOPS_IPP
#define SPIRIT_LOOPS_IPP

///////////////////////////////////////////////////////////////////////////////

#include <cassert>
#include "boost/spirit/loops.hpp"

///////////////////////////////////////////////////////////////////////////////
namespace spirit {

///////////////////////////////////////////////////////////////////////////////
//
//  fixed_loop class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename ExactT>
inline fixed_loop<S, ExactT>::fixed_loop(
    S const&        subject_,
    ExactT const&   exact_)
:
    unary<S>(subject_),
    exact(exact_)
{
}

//////////////////////////////////
template <typename S, typename ExactT>
template <typename IteratorT>
inline match
fixed_loop<S, ExactT>::parse(IteratorT& first, IteratorT const& last) const
{
    unsigned    n = exact;
    match       hit(0);

    for (unsigned i = 0; i < n; ++i)
    {
        match next = this->subject().parse(first, last);
        if (!next)
            return match();
        hit += next;
    }

    return hit;
}

///////////////////////////////////////////////////////////////////////////////
//
//  finite_loop class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename MinT, typename MaxT>
inline finite_loop<S, MinT, MaxT>::finite_loop(
    S const&    subject_,
    MinT const& min_,
    MaxT const& max_)
:
    unary<S>(subject_),
    min(min_),
    max(max_)
 {
 }

//////////////////////////////////
template <typename S, typename MinT, typename MaxT>
template <typename IteratorT>
inline match
finite_loop<S, MinT, MaxT>::parse(IteratorT& first, IteratorT const& last) const
{
    unsigned    n1 = min;
    unsigned    n2 = max;
    assert(n1 <= n2);

    IteratorT   s = first;
    match       hit(0);

    for (unsigned i = 0; i < n2; ++i)
    {
        match next = this->subject().parse(s, last);
        if (!next)
            if (i >= n1)
            {
                break;
            }
            else
            {
                first = s;
                return match();
            }
        first = s;
        hit += next;
    }

    return hit;
}

///////////////////////////////////////////////////////////////////////////////
//
//  infinite_loop class implementation
//
///////////////////////////////////////////////////////////////////////////////
template <typename S, typename MinT>
inline infinite_loop<S, MinT>::infinite_loop(
    S const&    subject_,
    MinT        const& min_,
    more_t)
:
    unary<S>(subject_),
    min(min_)
{
}

//////////////////////////////////
template <typename S, typename MinT>
template <typename IteratorT>
inline match
infinite_loop<S, MinT>::parse(IteratorT& first, IteratorT const& last) const
{
    unsigned    n = min;
    match       hit(0);

    for (unsigned i = 0;; ++i)
    {
        IteratorT   s = first;
        match next = this->subject().parse(s, last);
        if (!next)
            if (i >= n)
                break;
            else
                return match();
        hit += next;
        first = s;
    }

    return hit;
}

///////////////////////////////////////////////////////////////////////////////
}   //  namespace Spirit

#endif

