
/* Terminality - a portable terminal handling library
 * Copyright (C) 1998-2002, Emil Mikulic.
 * This is LGPL - look at COPYING.LIB
 */

/* Linked list code
 * Written by Emil Mikulic, 1999.
 * Overhauled by Michal Safranek, 2001.
 *
 * $Id: list.h,v 1.8 2002/07/26 01:39:40 darkmoon Exp $
 */

#ifndef __LIST_H__
#define __LIST_H__

#include "xmem.h"
#include <stdio.h>
#include <string.h>

/*
 * Short note about complexity:
 * This list is primarily intended to hold medium count of items and provide
 * fast access to them. Thus using list_delete() results in performance
 * disaster (for large lists) because it uses memmove. The lower element
 * you remove the larger block of memory must be moved.
 * (it moves n+1 to count items [* 4 bytes (sizeof ptr)])
 *
 * This means one thing: using this list as 'engine' for 'stack' is
 * not-so-bad idea but using it for 'queue' is really _bad_ idea.
 * ==========================================================================
 * Even worse: when using list_delete the memory used by list is not
 * reallocated to make the list smaller and thus making list with 1M items
 * and then using list_delete won't free *any* memory. I won't guarantee
 * this will change in future ... feel free to hack this in if you want to.
 */


#ifdef __cplusplus
extern "C" {
#endif



#define STARTING_POOL 5

/* list */
typedef struct {
	/* Don't change these or you'll get into serious troubles */
	void **elements;
	int count;
	int allocated;
	/* Expecting many list_add()s? Increase alloc_by to make insertion more
	 * effective. Warning: large values results in memory exhaustion!
	 * count == alloc_by + 1 elements results in
	 * xmalloc(2 * alloc_by * sizeof(void *)); ! Be careful.
	 */
	int alloc_by;
} tn_list;



/* create new list element */
tn_list *tn_list_new(void *x);

/* add an element on to a list */
int tn_list_add(tn_list *dest, void *elem);

/* get the number of items in the list */
int tn_list_size(tn_list *x);

/* free the whole list */
void tn_list_free(tn_list *x);

/* get the nth item (data) in a list */
void *tn_list_getdata(tn_list *x, int n);

/* delete the nth element of list */
int tn_list_delete(tn_list *x, int n);

/* replace nth element in a list */
void tn_list_replace(tn_list *dest, int n, void *elem);

/* free space used by list */
void tn_list_kill(tn_list *x);

#ifdef __cplusplus
}
#endif

#endif

