
/* Terminality - a portable terminal handling library
 * Copyright (C) 1998-2002, Emil Mikulic.
 * This is LGPL - look at COPYING.LIB
 */

/* $Id: tn.h,v 1.26 2002/07/27 07:40:37 darkmoon Exp $ */

/* Project:     Terminality
 * File:	tn.h
 * Author:      Emil Mikulic
 * Description: Provide a general interface to the Terminality library
 */

#ifndef _tn_h_
#define _tn_h_

/* Version-specifics */
#define TN_VERSION "2.1"
#define TN_BUILD 21

#include "colors.h"

/* Platform-dependent includes */
#ifdef __unix__
# include "tn_h_nc.h"
#endif
#ifdef _WIN32
# include "tn_h_w32.h"
#endif

/* Safeguard */
#ifndef TN_H_IMPLEMENTED
# error Unrecognised platform
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* Colors */
typedef enum {
	Default = -1, /* Not a real color, needed for listing */
	Black = 0,
	Blue = 1,
	Green = 2,
	Cyan = 3,
	Red = 4,
	Magenta = 5,
	Brown = 6,
	LightGray = 7,
	/* Bright [foreground] */
	DarkGray = 8,
	LightBlue = 9,
	LightGreen = 10,
	LightCyan = 11,
	LightRed = 12,
	LightMagenta = 13,
	Yellow = 14,
	White = 15 
} color;

#define LightGrey LightGray
#define DarkGrey DarkGray
#define LightBrown Yellow

#define colour color
#define COLOUR COLOR

/* Cursor type */
typedef enum {
    none,	/* hidden */
    line,	/* thin */
    rect	/* fat */
} cursor;



/* Functions */
void initcons(void);			/* Initialise console */
void donecons(void);			/* Done with console */
void clrscr(void);			/* Clear screen */
void textcolor(color c);		/* Set foreground (text) color */
void textbackground(color c);		/* Set background color */
void setcolor(color f, color b);	/* Set color to f, b */
color getbgcolor(void);			/* Get bg color */
color getfgcolor(void);			/* Get fg color */
key readkey(void);			/* Get keypress */
int keypressed(void);			/* Key pressed? */
#ifndef __DJGPP__
void delay(int ms);			/* Delay for <ms> milliseconds */
#endif
void gotoxy(int x, int y);		/* Move cursor to x,y */
void writech(chtype c);			/* Write character */
int update_set(int want);		/* Enable / disable screen updates */
void update(void);			/* Update screen */
int has_color(void);			/* Do we have colors? */
void clreol(void); 			/* Clear to end of line */
int wherey(void);			/* Return cursor Y position */
int wherex(void);			/* Return cursor X position */
int beep(void);				/* Audible beep/bell/alarm thing */
int printw(const char *format, ...);	/* printf to screen */

/* No-color terminal functions */
void highvideo(void);			/* Bright text */
void lowvideo(void);			/* Dim text */
void normvideo(void);			/* Normal text */
void reversevideo(void);		/* Reverse/standout text */

/* Cursor functions */
cursor get_cursor(void);		/* Gets cursor */
void set_cursor(cursor c);		/* Sets cursor */

/* Color scheme functions: */
color get_scheme_color(int name);	/* Returns color by constant */
void set_scheme(color *scheme);		/* Sets color scheme */
void default_scheme(void);		/* Sets default color scheme */



/* Compatibility with older Terminality releases */
#define fgcolor textcolor
#define bgcolor textbackground

/* COLOR attribute array */
extern int COLOR[16][8];

/* Global cursor */
extern cursor glb_cursor;

/* The following special characters must be supported or emulated:
 *	HLINE	- Horizontal line
 *	VLINE	- Vertical line
 *	CROSS	- Cross/plus
 *	C_UL	- Upper left corner
 *	C_UR	- Upper right
 *	C_LL	- Lower left
 *	C_LR	- Lower right
 *	BLOCK_1	- 25%-full block
 *	BLOCK_2	- 50%-full block
 *	BLOCK_3 - 75%-full block
 *	BLOCK_4	- Solid block
 *	TEE_L	- Tee pointing left
 *	TEE_R	- Tee pointing right
 *	TEE_U	- Tee pointing up
 *	TEE_D	- Tee pointing down
 *	ARROW_L	- Arrow pointing left
 *	ARROW_R	- Arrow pointing right
 *	ARROW_U	- Arrow pointing up
 *	ARROW_D	- Arrow pointing down
 *
 *
 * The macro _c(x) should be provided. It is used to create a
 * chtype value from an integer specifying a regular ASCII
 * character code. i.e. BLOCK_4 = _c(219)
 *
 *
 * The following keys must be provided:
 *	KEY_ENTER
 *	KEY_TAB
 *	KEY_BACKSPACE
 *	KEY_DEL
 *	KEY_UP
 *	KEY_DOWN
 *	KEY_LEFT
 *	KEY_RIGHT
 *	KEY_HOME
 *	KEY_END
 *	KEY_PGUP
 *	KEY_PGDOWN
 */

/* To obtain the size of the terminal, Terminality backends must
   provide CON_COLS and CON_ROWS as macros or variables. */



/* Types */
typedef enum {
	Unknown = 0,
	Checkbox = 1,
	Radio = 2,
	Textbox = 3,
	Textarea = 4,
	Button = 5,
	Menu = 6,
	Hmenu = 7,
	Listing = 8,
	Form = 9,
	Custom = 10, /* Custom type for drawing graphics that are persistant
					even when redraw is called */
	Textview = 11,
} element_type;



#ifdef __cplusplus
}
#endif

#endif

