
/* Terminality - a portable terminal handling library
 * Copyright (C) 1998-2002, Emil Mikulic.
 * This is LGPL - look at COPYING.LIB
 */

/* Project:     Terminality
 * File:	tn_c_nc.h
 * Description: Provides the ncurses implementation
 */

#define TN_C_IMPLEMENTED

const char tn_nc_rcsid[] =
"$Id: tn_c_nc.h,v 1.1 2002/07/26 08:18:14 darkmoon Exp $";



/* Current (set) colour variables */
color fg_c, bg_c;



/* Initialize console */
void initcons(void)
{
	/* Initialize console */
	initscr();
	raw();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	initcolor();
	set_cursor(line); /* want none? */
	default_keys();
}



/* Clean up console */
void donecons(void)
{
	/* Set normal cursor */
	set_cursor(line);
	/* Delete all keyhandlers */
	delete_keys();
	/* Delete register of all elements */
	register_free();
	/* Denitialize console */
	endwin();
}



/* Do we have color */
int has_color(void)
{
	return has_colors();
}



/* Init colors */
int initcolor (void)
{
	int fg, bg;
	int tmp_a;
	static int convtabl[8] = {
		COLOR_BLACK,
		COLOR_BLUE,
		COLOR_GREEN,
		COLOR_CYAN,
		COLOR_RED,
		COLOR_MAGENTA,
		COLOR_YELLOW,
		COLOR_WHITE
		};

	/* Try to initialize colors */
	if (!has_color ())
		return FALSE;
	else {
		/* Start up colors */
		start_color ();

		/* Initialize color pairs */
		for (bg = 0; bg < COLORS; bg++)
		for (fg = 0; fg < COLORS; fg++)
		{
			init_pair (bg * COLORS + fg, fg, bg);
		}

		/* Initialise COLORS array */
		for (bg = 0; bg < 8; bg++)
		for (fg = 0; fg < 16; fg++)
		{
			/* Get color */
			tmp_a = COLOR_PAIR(convtabl[bg]*COLORS +
				convtabl[fg % 8]);

			/* Make it BOLD if it's a bright color */
			if (fg>7) tmp_a |= A_BOLD;

			/* Store into array */
			COLOR[fg][bg] = tmp_a;
		}

		init_pair(COLOR_BLACK*COLORS+COLOR_WHITE,
			COLOR_BLACK, COLOR_BLACK);
		COLOR[DarkGray][Black] = COLOR_PAIR(COLOR_BLACK*COLORS+
			COLOR_WHITE) | A_BOLD;
		COLOR[LightGray][Black] = A_NORMAL;
		COLOR[White][Black] = A_BOLD;

		/* some things you just can't do */
		COLOR[Black][Black] = A_NORMAL;
		COLOR[LightGray][LightGray] = COLOR[Black][LightGray] | A_INVIS;

		/* Current (set) colors */
		fg_c = LightGray;
		bg_c = Black;

		/* Return */
		return TRUE;
	}
}



/* Clear screen */
void clrscr(void)
{
	werase(stdscr);
}



/* Set color */
void setcolor(color f, color b)
{
	assert(f != Default && b != Default);

	/* Set current */
	fg_c = f;
	bg_c = b;

	/* Set color */
	attrset(COLOR[fg_c][bg_c%8]|(bg_c>7?A_BLINK:0));
}



/* Get bg color */
color getbgcolor(void)
{
	return bg_c;
}



/* Get fg color */
color getfgcolor(void)
{
	return fg_c;
}



/* Set foreground (text) color */
void fgcolor(color c)
{
	setcolor(c, bg_c);
}



/* Set background color */
void bgcolor(color c)
{
	setcolor(fg_c, c);
}



/* ncurses handles printw() */



/* Readkey - read key from keyboard */
key readkey(void)
{
	return getch();
}



#ifndef __DJGPP__
/* Delay */
void delay (int ms)
{
	napms(ms);
}
#endif



/* gotoxy */
void gotoxy(int x, int y)
{
	move(y-1, x-1);
}



/* writech */
void writech(chtype c)
{
	addch(c);
}



/* update */
void update(void)
{
	if (update_enabled) refresh();
}



/* Check for keypress */
int keypressed (void)
{
	key k;  

	nodelay(stdscr, TRUE);
	k = getch();
	nodelay(stdscr, FALSE);

	if (k == ERR)
		return 0;
	else {
		ungetch(k);
		return 1;
	}
}



/* Bright text */
void highvideo(void)
{
	attrset(A_BOLD);
}



/* Dim text */
void lowvideo(void)
{
	attrset(A_DIM);
}



/* Normal text */
void normvideo(void)
{
	attrset(A_NORMAL);
}



/* Standout text */
void reversevideo(void)
{
	attrset(A_STANDOUT);
}



/* Get x cursor position */
int wherex(void)
{
	int x, y;
	getyx(stdscr, y, x);

	return x+1;
}



/* Get y cursor position */
int wherey(void)
{
	int x, y;
	getyx(stdscr, y, x);

	return y+1;
}



/* Sets cursor */
void set_cursor(cursor c)
{
	curs_set(c);
	glb_cursor = c;
}

