"""
web2ldapcnf/hosts.py - Host-related options
(c) by Michael Stroeder <michael@stroeder.com>
"""

# Leave this alone
import os,web2ldapcnf

# List of all LDAP hosts to use. These hosts will appear in the
# default select list of the login form.
# A list containing only one host results in a normal input field
# with the host set as default.
ldap_host_list = [
  'localhost',
  'localhost:1389',
  'localhost:2389',
  'localhost:3389',
  'localhost:4389',
  'localhost:5389',
  'ldap.openldap.org',
  'www.nldap.com',
  'ldap.surfnet.nl',
  'ldap.rediris.es',
  'ldap.uninett.no',
  'www.trustcenter.de',
  'db.debian.org',
  'ldap.ust.hk',
  'ldap.itd.umich.edu',
  'ldap.baylor.edu',
  'directory.uga.edu',
  'ldap.nameflow.net',
  'memberdir.netscape.com',
  'directory.verisign.com',
  'os.directory.dfn.de',
]

########################################################################
# LDAP host(s) with their defaults can be pre-defined as dictionary
# ldap_def = {'host:port':{paramdictionary}}
########################################################################


class Web2LDAPConfig:
  """
  Base class for a web2ldap host-/backend configuration section.
  
  Leave untouched!!!
  """
  def __init__(self,**kwargs):
    self.__dict__.update(kwargs)
  def get(self,name,default=None):
    self.__dict__.get(name,default)

ldap_def = {

  # Overall default values.
  # There's most times no need to adjust the values in this section
  # since they can be overridden by host specific sections (see below).
  '_': Web2LDAPConfig(
    # Template for mapping client certs to bind DN
    binddn=r'mail=%(Email)s',
    # Search filter template for smart login
    binddnsearch=r'(|(cn=%s)(mail=%s@*)(uid=%s))',
    # Timeout for synchronous LDAP operations (seconds)
    timeout=60,
    # Use StartTLS on connect. It's recommended saying 0 as default
    # since not many servers can handle an unknown extended control properly
    starttls=0,
    # Maximum amount of time in seconds a result is valid in the cache
    cache_timeout=0,
    # Maximum cache size in bytes (per session!)
    cache_maxmem=0,
    # regular expression of subject DNs in client certs
    sec_valid_dn=r'/.*',
    # regular expression of subject DNs in issuer certs
    sec_valid_idn=r'/.*',
    sec_reqlevel=0,
    # Never show these attributes
#    hiddenattrs=['userPassword'],
    # vCard template files
    vcard_template={
      # 'object class':'pathname of vCard template file'
      'person':os.path.join(web2ldapcnf.templates_dir,'vcard_person.txt'),
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'vcard_person.txt')
    },
    # HTML template files for printing table entries
    print_template={
      # 'object class':'pathname of printable HTML template file'
      'person':os.path.join(web2ldapcnf.templates_dir,'print_person.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'print_organization.html'),
      'organizationalUnit':os.path.join(web2ldapcnf.templates_dir,'print_organizationalUnit.html'),
    },
    # HTML template file for Basic Search form
    searchform_template= os.path.join(web2ldapcnf.templates_dir,'searchform_Base.html'),
    # Attributes which should be present in attribute select list of advanced search form
    search_attrs=['cn','mail','o','ou','telephoneNumber','homePhone','l','description'],
    # There are some situations where this client just wants to get the
    # attributes of an entry and not the data itself for saving bandwidth.
    # However some LDAP hosts (e.g. Notes Domino 4.61) have problems with
    # such an attribute-only request, they won't return any matches for a search.
    # If you experience this problems (no matching entry) set this to 0.
    search_attrsonly=1,
    # HTML template strings used to display entries in the table
    # of search results
    search_tdtemplate= {
#      'inetOrgPerson':r'%(cn)s &lt;%(mail)s&gt; Tel.: %(telephoneNumber)s or %(homePhoneNumber)s',
      'organizationalUnit':r'Org. Unit &quot;%(ou)s&quot; Tel.: %(telephoneNumber)s',
      'rfc822MailGroup':r'Mailing list %(cn)s &lt;%(mail)s&gt;, see %(labeledurl)s',
      'account':r'Account <strong>%(uid)s</strong>',
    },
    # Default for number of results shown per page
    search_resultsperpage=20,
    # HTML template file used for displaying entries of specific object class
    read_template={
      # 'object class':'pathname of HTML template file'
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'organizationalPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'read_organization.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'read_posixAccount.html'),
    },
    # HTML template file used for displaying operational attributes
    # in a footer when displaying a single entry
    read_operationalattrstemplate=os.path.join(web2ldapcnf.templates_dir,'read_operationalattrs.html'),
    # Number of columns for print output
    print_cols=4,
    # List of tuples (description,object class list) for displaying
    # buttons for quick-choosing object classes of new entries
    addform_oc_list=[
      ('Person','cn',['person','organizationalPerson','inetOrgPerson']),
      ('Company','o',['organization','labeledURIObject']),
      ('Posix User','uid',['posixAccount']),
      ('Org. Unit','ou',['organizationalUnit']),
    ],
  ),

  'localhost': Web2LDAPConfig(
    description='My poorly configured LDAP host',
  ),

  'localhost:1389': Web2LDAPConfig(
    description='My local OpenLDAP 2.0.x test server',
    starttls=2,
  ),

  # This meant as an more complex example section
  # Adjust the settings to reflect your local LDAP installation
  'ldap://localhost/o=michaels': Web2LDAPConfig(
    description='Private address book of Michael Strder',
    ssl_valid_dn=r'^/S=Stroeder/G=Michael/CN=Michael Stroeder/Email=michael@stroeder.com$',
    ssl_valid_idn=r'^/C=ZA/ST=Western Cape/L=Durbanville/O=Thawte/OU=Certificate Services/CN=Personal Freemail RSA 1999\.9\.16$',
#    'ssl_minlevel=1,
    search_tablistattrs=['cn','mail','homephone','telephonenumber','mobile'],
    dit= {
      # 'DN':(
      #   'DN component attribute of subordinate entries',
      #   [list of objectClasses],
      #   immutable
      # )
      'o=Michaels':('ou',['organizationalUnit'],1),
      'ou=Subscribed mailing lists,o=Michaels':('cn',['rfc822MailGroup'],0),
      'ou=web2ldap-users,ou=web2ldap,ou=Projects,o=Michaels':('mail',['person','organizationalPerson','inetOrgPerson'],0),
    },
    read_template={
      # 'object class':'pathname of HTML template file'
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'read_organization.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'read_posixAccount.html'),
      'bankArrangement':os.path.join(web2ldapcnf.templates_dir,'read_bankArrangement.html'),
      'musician':os.path.join(web2ldapcnf.templates_dir,'read_musician.html'),
    },
    passwd_hashtypes=['ssha']
  ),

  # Special example for bibtex database of DFN:
  # - HTML templates for searchform, read and print are used.
  # - Search table contains most significant attributes.
  'ldap://os.directory.dfn.de/cn=bibtex,dc=directory,dc=dfn,dc=de': Web2LDAPConfig(
    description='Bibtex database of DFN',
    search_tablistattrs=['cn','bibtexTitle','bibtexAuthor','labeledURI'],
    searchform_template=os.path.join(
      web2ldapcnf.templates_dir,'searchform_bibtex.html'
    ),
    read_template={
      'bibtexEntry':os.path.join(
      web2ldapcnf.templates_dir,'read_bibtex.html'
    ),
    },
    print_cols=2,
    print_template={
      # 'object class':'pathname of printable HTML template file'
      'bibtexEntry':os.path.join(
        web2ldapcnf.templates_dir,'print_bibtex.html'
      ),
    },
  ),

}

