# postfix-lib.pl
#
# postfix-module by Guillaume Cottenceau <gc@mandrakesoft.com>,
# for webmin by Jamie Cameron
#
# Copyright (c) 2000 by Mandrakesoft
#
# Permission to use, copy, modify, and distribute this software and its
# documentation under the terms of the GNU General Public License is hereby 
# granted. No representations are made about the suitability of this software 
# for any purpose. It is provided "as is" without express or implied warranty.
# See the GNU General Public License for more details.
#
#
# Functions for managing the postfix module for Webmin.
#
# Written by G. Cottenceau for MandrakeSoft <gc@mandrakesoft.com>
# This is free software under GPL license.
#

$POSTFIX_MODULE_VERSION = 5;

#
#
# -------------------------------------------------------------------------


do '../web-lib.pl';
&init_config();
%access = &get_module_acl();

sub guess_config_dir
{
    my $answ = $config{'postfix_config_file'};
    $answ =~ /(.*)\/[^\/]*/;
    return $1;
}

$config_dir = guess_config_dir();


## DOC: compared to other webmin modules, here we don't need to parse
##      the config file, because a config command is provided by
##      postfix to read and write the config parameters


# postfix_module_version()
# returns the version of the postfix module
sub postfix_module_version
{
    return $POSTFIX_MODULE_VERSION;
}

# is_postfix_running()
# returns 1 if running, 0 if stopped, calls error() if problem
sub is_postfix_running
{
    my $queuedir = get_current_value("queue_directory");
    my $processid = get_current_value("process_id_directory");

    my $pid_file = $queuedir."/".$processid."/master.pid";

    if (open(PID, $pid_file))
    {
	chop(my $pid = <PID>);
	close(PID);
	$pid =~ /([0-9]+)/;
	return kill 0, $1;
    }
    else
    {
	return 0;
    }
}


sub is_existing_parameter
{
    my $out = `$config{'postfix_config_command'} -c $config_dir $_[0] 2>&1`;
    return !($out =~ /unknown parameter/);
}


# get_current_value(parameter_name)
# returns a scalar corresponding to the value of the parameter
sub get_current_value
{
    my $out = `$config{'postfix_config_command'} -c $config_dir -h $_[0] 2>&1`;  # -h tells postconf not to output the name of the parameter
    if ($?) { &error(&text('query_get_efailed', $_[0], $out)); }
    chop($out);
    return $out;
}

# if_default_value(parameter_name)
# returns if the value is the default value
sub if_default_value
{
    my $out = `$config{'postfix_config_command'} -c $config_dir -n $_[0] 2>&1`;
    if ($?) { &error(&text('query_get_efailed', $_[0], $out)); }
    return ($out eq "");
}

# get_default_value(parameter_name)
# returns the default value of the parameter
sub get_default_value
{
    my $out = `$config{'postfix_config_command'} -c $config_dir -dh $_[0] 2>&1`;  # -h tells postconf not to output the name of the parameter
    if ($?) { &error(&text('query_get_efailed', $_[0], $out)); }
    chop($out);
    return $out;
}


# set_current_value(parameter_name, parameter_value)
# 
sub set_current_value
{
    my $value = $_[1];
#    print "--".$value."--<br>";
    if (($value eq "__DEFAULT_VALUE_IE_NOT_IN_CONFIG_FILE__") || ($value eq &get_default_value($_[0])))
    {
	# there is a special case in which there is no static default value ;
	# postfix will handle it correctly if I remove the line in `main.cf'
	my $all_lines = &read_file_lines($config{'postfix_config_file'});
	my $line_of_parameter = -1;
	my $i = 0;

	foreach (@{$all_lines})
	{
	    if (/^\s*$_[0]\s*=/)
	    {
		$line_of_parameter = $i;
	    }
	    $i++;
	}

	if ($line_of_parameter != -1)
	{
	    splice(@{$all_lines}, $line_of_parameter, 1);
	    
	}
        &flush_file_lines();
    }
    else
    {
	$value =~ s/\$/\\\$/g;     # prepend a \ in front of every $ to protect from shell substitution
	$out = system("$config{'postfix_config_command'} -c $config_dir -e $_[0]=\"$value\" 2>&1");
    }
    if ($out) { &error(&text('query_set_efailed', $_[0], $_[1], $out)."<br> $config{'postfix_config_command'} -c $config_dir -e $_[0]=\"$value\" 2>&1"); }
}

# check_postfix()
#
sub check_postfix
{
	my $out = system("$config{'postfix_control_command'} -c $config_dir check 2>&1");
	return $out;
}

# reload_postfix()
#
sub reload_postfix
{
    $access{'startstop'} || &error($text{'reload_ecannot'});
    if (is_postfix_running())
    {
	if (check_postfix()) { &error("$text{'check_error'}"); }
	my $out = system("$config{'postfix_control_command'} -c $config_dir reload 2>&1");
	if ($out) { &error($text{'reload_efailed'}); }
    }
}





# option_radios_freefield(name_of_option, length_of_free_field, [name_of_radiobutton, text_of_radiobutton]+)
# builds an option with variable number of radiobuttons and a free field
# WARNING: *FIRST* RADIO BUTTON *MUST* BE THE DEFAULT VALUE OF POSTFIX
sub option_radios_freefield
{
    my ($name, $length) = ($_[0], $_[1]);

    my $v = &get_current_value($name);
    my $key = 'opts_'.$name;

    my $check_free_field = 1;
    
    printf "<td>".&hlink("<b>$text{$key}</b>", "opt_".$name)."</td> <td %s nowrap>\n",
    $length > 20 ? "colspan=3" : "";

    # first radio button (must be default value!!)
    
    printf "<input type=radio name=$name"."_def value=\"__DEFAULT_VALUE_IE_NOT_IN_CONFIG_FILE__\" %s> $_[2]\n",
    (&if_default_value($name)) ? "checked" : "";

    $check_free_field = 0 if &if_default_value($name);
    shift;
    
    # other radio buttons
    while (defined($_[2]))
    {
	printf "<input type=radio name=$name"."_def value=\"$_[2]\" %s> $_[3]\n",
	($v eq $_[2]) ? "checked" : "";
	if ($v eq $_[2]) { $check_free_field = 0; }
	shift;
	shift;
    }

    # the free field
    printf "<input type=radio name=$name"."_def value=__USE_FREE_FIELD__ %s>\n",
    ($check_free_field == 1) ? "checked" : "";
    printf "<input name=$name size=$length value=\"%s\"> </td>\n",
    ($check_free_field == 1) ? $v : "";
}


# option_freefield(name_of_option, length_of_free_field)
# builds an option with free field
sub option_freefield
{
    my ($name, $length) = ($_[0], $_[1]);

    my $v = &get_current_value($name);
    my $key = 'opts_'.$name;
    
    printf "<td>".&hlink("<b>$text{$key}</b>", "opt_".$name)."</td> <td %s nowrap>\n",
    $length > 20 ? "colspan=3" : "";
    
    print "<input name=$name"."_def size=$length value=\"$v\"> </td>\n";
}


# option_yesno(name_of_option, [help])
# if help is provided, displays help link
sub option_yesno
{
    my $name = $_[0];
    my $v = &get_current_value($name);
    my $key = 'opts_'.$name;

    defined($_[1]) ?
	print "<td>".&hlink("<b>$text{$key}</b>", "opt_".$name)."</td> <td nowrap>\n"
    :
	print "<td><b>$text{$key}</b></td> <td nowrap>\n";
    
    printf "<input type=radio name=$name"."_def value=\"yes\" %s> $text{'yes'}\n",
    (lc($v) eq "yes") ? "checked" : "";

    printf "<input type=radio name=$name"."_def value=\"no\" %s> $text{'no'}\n",
    (lc($v) eq "no") ? "checked" : "";

    print "</td>\n";
}



############################################################################
# aliases support    [too lazy to create a aliases-lib.pl :-)]

# get_aliases_files($alias_maps) : @aliases_files
# parses its argument to extract the filenames of the aliases files
# supports multiple alias-files
sub get_aliases_files
{
    $_[0] =~ /:(\/[^,\s]*)(.*)/;
    (my $returnvalue, my $recurse) = ( $1, $2 );

    # Yes, Perl is also a functional language -> I construct a list, and no problem, lists are flattened in Perl
    return ( $returnvalue,
	     ($recurse =~ /:\/[^,\s]*/) ?
	         &get_aliases_files($recurse)
	     :
	         ()
           )
}

 
# get_aliases() : \@aliases
# construct the aliases database taken from the aliases files given in the "alias_maps" parameter
sub get_aliases
{
    if (!@aliases_cache)
    {
	my @aliases_files = &get_aliases_files(&get_current_value("alias_maps"));
	my $number = 0;
	foreach $aliases_file (@aliases_files)
	{
	    open(ALIASES, $aliases_file);
	    my $i = 0;
	    while (<ALIASES>)
	    {
		s/^#.*$//g;	# remove comments
		s/\r|\n//g;	# remove newlines
		if ((/^\s*\"([^\"]*)[^:]*:\s*([^#]*)/) ||      # names with double quotes (") are special, as seen in `man aliases(5)`
		    (/^\s*([^\s:]*)[^:]*:\s*([^#]*)/))         # other names
		{
		    $number++;
		    my %alias;
		    $alias{'name'} = $1;
		    $alias{'value'} = $2;
		    $alias{'line'} = $i;
		    $alias{'alias_file'} = $aliases_file;
		    $alias{'number'} = $number;
		    push(@aliases_cache, \%alias);
		}
		$i++;
	    }
	    close(ALIASES);
	}
    }
    return \@aliases_cache;
}


# init_new_alias() : $number
# gives a new number of alias
sub init_new_alias
{
    $aliases = &get_aliases();

    my $max_number = 0;

    foreach $trans (@{$aliases})
    {
	if ($trans->{'number'} > $max_number) { $max_number = $trans->{'number'}; }
    }
    
    return $max_number+1;
}



# save_options(%options)
#
sub save_options
{
    if (check_postfix()) { &error("$text{'check_error'}"); }

    my %options = %{$_[0]};

    foreach $key (keys %options)
    {
	if ($key =~ /_def/)
	{
	    (my $param = $key) =~ s/_def//;
	    
	    ($options{$key} eq "__USE_FREE_FIELD__") ?
		&set_current_value($param, $options{$param})
		    :
		&set_current_value($param, $options{$key});
	}
    }
}


# regenerate_aliases
#
sub regenerate_aliases
{
    $access{'aliases'} || error($text{'regenerate_ecannot'});
    if (get_current_value("alias_maps") eq "")
    {
	$out = system("$config{'postfix_newaliases_command'} 2>&1");
	if ($out) { &error(&text('regenerate_alias_efailed', $out)); }
    }
    else
    {
	foreach $map (get_maps_files(get_current_value("alias_maps")))
	{
	    $out = system("$config{'postfix_aliases_table_command'} -c $config_dir $map 2>&1");
	    if ($out) { &error(&text('regenerate_table_efailed', $map, $out)); }
	}
    }
}


# regenerate_relocated_table
#
sub regenerate_relocated_table
{
    &regenerate_any_table("relocated_maps");
}


# regenerate_virtual_table
#
sub regenerate_virtual_table
{
    &regenerate_any_table("virtual_maps");
}


# regenerate_canonical_table
#
sub regenerate_canonical_table
{
    &regenerate_any_table("canonical_maps");
    &regenerate_any_table("recipient_canonical_maps");
    &regenerate_any_table("sender_canonical_maps");
}


# regenerate_transport_table
#
sub regenerate_transport_table
{
    &regenerate_any_table("transport_maps");
}


# regenerate_any_table($parameter_where_to_find_the_table_names)
#
sub regenerate_any_table
{
    if (&get_current_value($_[0]) ne "")
    {
	foreach $map (&get_maps_files(&get_current_value($_[0])))
	{
	    $out = `$config{'postfix_lookup_table_command'} -c $config_dir $map 2>&1`;
	    if ($out) { &error(&text('regenerate_table_efailed', $map, $out)); }
	}
    }
}



############################################################################
# maps [canonical, virtual, transport] support

# get_maps_files($maps_param) : @maps_files
# parses its argument to extract the filenames of the mapping files
# supports multiple maps-files
sub get_maps_files
{
    $_[0] =~ /:(\/[^,\s]*)(.*)/;
    (my $returnvalue, my $recurse) = ( $1, $2 );

    return ( $returnvalue,
	     ($recurse =~ /:\/[^,\s]*/) ?
	         &get_maps_files($recurse)
	     :
	         ()
           )
}

 
# get_maps($maps_name) : \@maps
# construct the mappings database taken from the map files given from the parameter
sub get_maps
{
    if (!@maps_cache)
    {
	my @maps_files = &get_maps_files(&get_current_value($_[0]));
	my $number = 0;
	foreach $maps_file (@maps_files)
	{
	    open(MAPS, $maps_file);
	    my $i = 0;
	    while (<MAPS>)
	    {
		s/^#.*$//g;	# remove comments
		s/\r|\n//g;	# remove newlines
		if (/^\s*([^\s]+)\s+([^#]*)/)
		{
		    $number++;
		    my %map;
		    $map{'name'} = $1;
		    $map{'value'} = $2;
		    $map{'line'} = $i;
		    $map{'map_file'} = $maps_file;
		    $map{'number'} = $number;
		    push(@maps_cache, \%map);
		}
		$i++;
	    }
	    close(MAPS);
	}
    }
    return \@maps_cache;
}


sub generate_map_edit
{
    if (&get_current_value($_[0]) eq "")
    {
	print ("<h2>$text{'no_map2'}</h2><br>");
	print "<hr>\n";
	&footer("", $text{'index_return'});
	exit;
    }


    my $mappings = &get_maps($_[0]);

    if ($#{$mappings} ne -1)
    {
	print $_[1];
	
	print "<table width=100%> <tr><td width=50% valign=top>\n";
	
	print "<table border width=100%>\n";
	print "<tr $tb> <td><b>$text{'mapping_name'}</b></td> ",
	"<td><b>$text{'mapping_value'}</b></td> </tr>\n";

	my $split_index = int(($#{$mappings})/2);
	my $i = -1;
	
	foreach $map (@{$mappings})
	{
	    print "<tr $cb>\n";
	    print "<td><a href=\"edit_mapping.cgi?num=$map->{'number'}&map_name=$_[0]\">$map->{'name'}</a></td>\n";
	    print "<td>$map->{'value'}</td>\n</tr>\n";
	    $i++;
	    if ($i == $split_index)
	    {
		print "</table></td><td width=50% valign=top>\n";
		print "<table border width=100%>\n";
		print "<tr $tb> <td><b>$text{'mapping_name'}</b></td> ",
		"<td><b>$text{'mapping_value'}</b></td> </tr>\n";
	    }
	}
	
	print "</tr></td></table>\n";
	print "</table>\n";
    }


# new form

    print "<table cellpadding=5 width=100%><tr><td>\n";
    print "<form action=edit_mapping.cgi>\n";
    print "<input type=hidden name=\"map_name\" value=\"$_[0]\">";
    print "<input type=submit value=\"$text{'new_mapping'}\">\n";
    print "</td> <td width=\"99%\">$text{'new_mappingmsg'}\n";
    print "</td></tr></table></form>\n";

}


# init_new_mapping($maps_parameter) : $number
# gives a new number of mapping
sub init_new_mapping
{
    $maps = &get_maps($_[0]);

    my $max_number = 0;

    foreach $trans (@{$maps})
    {
	if ($trans->{'number'} > $max_number) { $max_number = $trans->{'number'}; }
    }
    
    return $max_number+1;
}

# postfix_mail_file(user)
sub postfix_mail_file
{
local @s = &postfix_mail_system();
if ($s[0] == 0) {
	return "$s[1]/$_[0]";
	}
elsif (@_ > 1) {
	return "$_[7]/$s[1]";
	}
else {
	local @u = getpwnam($_[0]);
	return "$u[7]/$s[1]";
	}
}

# postfix_mail_system()
# Returns 0 and the spool dir for sendmail style,
#         1 and the mbox filename for ~/Mailbox style
#         2 and the maildir name for ~/Maildir style
sub postfix_mail_system
{
if (!defined(@mail_system_cache)) {
	local $home_mailbox = &get_current_value("home_mailbox");
	local $mail_spool_directory =&get_current_value("mail_spool_directory");
	if ($home_mailbox) {
		@mail_system_cache = $home_mailbox =~ /^(.*)\/$/ ?
			(2, $home_mailbox) : (1, $home_mailbox);
		}
	else {
		@mail_system_cache = (0, $mail_spool_directory);
		}
	}
return wantarray ? @mail_system_cache : $mail_system_cache[0];
}

1;

