#ifndef XPARAM_REG_VECTOR_H
#define XPARAM_REG_VECTOR_H

/*	Copyright (C) 2001,2002,2003 Ronnie Maor and Michael Brand
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the copyright holders give permission
 * for additional uses of the text contained in its release of XPARAM.
 *
 * The exception is that, if you link the XPARAM library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the XPARAM library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * If you write modifications of your own for XPARAM, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */

/*
 * Defines the user's API for registering vectors
 */

#include <iosfwd>
#include <vector>
#include <memory>
#include "xp_config.h"
#include "xp_value_list.h"
#include "xpv_arg_passers.h"
#include "xpv_scheduler.h"
#include "xpv_common_regcom.h"
#include "xpv_registry.h"
#include "xpv_type.h"
#include "xpv_convweight.h"

#include "xpv_reg_class.h"
#include "xpv_reg_ctor.h"
#include "xpv_reg_output.h"
#include "xpv_reg_conversion.h"


namespace xParam_internal {

 	/////////////////////////////////////
	// Vector kinds
	/////////////////////////////////////

	template<class T>
	class ValueVector {
		public:
			typedef std::vector<T> VecType;

			typedef ConstRef<T> ArgPasser;

			static Handle<T> ref_handle(const T& obj) { 
				T* obj_ptr = const_cast<T*>(&obj);
				return Handle<T>(obj_ptr,false); 
			}

			static std::string vec_name(const std::string& scalar_name) {
				return "vector<" + scalar_name + ">";
			}
	};

	template<class T>
	class PtrVector {
		public:
			typedef std::vector<T*> VecType;

			typedef ClassPtr<T> ArgPasser;

			static Handle<T> ref_handle(T* obj_ptr) { 
				return Handle<T>(obj_ptr,false);
			}

			static std::string vec_name(const std::string& scalar_name) {
				return "vector<" + scalar_name + "*>";
			}
	};

	template<class T>
	class ConstPtrVector {
		public:
			typedef std::vector<const T*> VecType;

			typedef ClassConstPtr<T> ArgPasser;

			static Handle<T> ref_handle(const T* const_obj_ptr) { 
				T* obj_ptr = const_cast<T*>(const_obj_ptr);
				return Handle<T>(obj_ptr,false);
			}

			static std::string vec_name(const std::string& scalar_name) {
				return "vector<const " + scalar_name + "*>";
			}
	};


 	/////////////////////////////////////
	// Vector Creators
	/////////////////////////////////////

	template<class T, class VectorKind>
	class VectorFromList {
		public:
			typedef typename VectorKind::VecType VecType;
			typedef typename VectorKind::ArgPasser ArgPasser;

			static VecType* create(const ValueList& val_list) 
			{
				std::auto_ptr<VecType> vec(new VecType);

				ValueList::const_iterator i;
				for (i=val_list.begin(); i!=val_list.end(); ++i) {
					Handle<Value> converted_val = (*i)->convert_to(typeid(T));
					vec->push_back( ArgPasser::pass(converted_val) );
				}
				return vec.release();
			}
	};

	/////////////////////////////////////
	// Vector output function
	/////////////////////////////////////

	template<class T, class VectorKind>
		class VectorOutput {
		public:
			typedef typename VectorKind::VecType VecType;

			static void output(std::ostream& os, const VecType& vec) 
			{
				ValueList val_list;

				typename VecType::const_iterator i;
				for (i=vec.begin(); i!=vec.end(); ++i) {
					Handle<T> obj_ref = VectorKind::ref_handle(*i);
					val_list.push_back( make_value(obj_ref) );
				}
				val_list.output(os);
			}
		};


	/////////////////////////////////////
	// RegistrationCommand
	/////////////////////////////////////

	template<class T, class VectorKind>
	class VectorRegCommand : public CommonRegCommand {
		public:
			VectorRegCommand() : CommonRegCommand(typeid(T)) {}
			
			typedef typename VectorKind::VecType VecType;
			typedef typename VectorKind::ArgPasser ArgPasser;

			virtual void do_registration() const 
			{
				// register class
				Type& scalar_type = type_registry().type(typeid(T));
				param_class<VecType> temp(VectorKind::vec_name(scalar_type.name()));

				// remove compilar warning about unused variable
				TemplateFooler::launder((void*)&temp); 

				// register empty ctor
				param_ctor(TypeTag<VecType>());


				// register ctor from single value and number of copies
				param_ctor(TypeTag<VecType>(), ByVal<long>("n"), ArgPasser("val"));

				// register ctor from ValueList
				typedef VectorFromList<T,VectorKind> ListCreator;
				param_explicit_creator(TypeTag<VecType>(),
										TypeTag<ListCreator>(),
										ConstRef<ValueList>("list"));

				// register output
				typedef VectorOutput<T,VectorKind> OutputFunc;
				param_direct_output<VecType,OutputFunc>();

				// register special conversion source
				assert(type_registry().is_registered(typeid(VecType)));
				Type& vec_type = type_registry().type(typeid(VecType));
				vec_type.reg_conversion_source(typeid(T),CONV_LIST);
			}

			virtual std::string description() const {
				Type& scalar_type = type_registry().type(typeid(T));
				return "vector type - " + VectorKind::vec_name(scalar_type.name());
			}

	};

	/////////////////////////////////////
	// User registration API
	/////////////////////////////////////

	template<class T>	
		class param_vector {
		public:
			param_vector()
			{
				typedef ValueVector<T> VectorKind;
				Handle<RegistrationCommand> command(new VectorRegCommand<T,VectorKind>);
				registration_scheduler().add_command(command);
			}
		};

	template<class T>	
		class param_ptr_vector {
		public:
			param_ptr_vector() 
			{
				typedef PtrVector<T> VectorKind;
				Handle<RegistrationCommand> command(new VectorRegCommand<T,VectorKind>);
				registration_scheduler().add_command(command);
			}
		};

	template<class T>	
		class param_const_ptr_vector {
		public:
			param_const_ptr_vector()
			{
				typedef ConstPtrVector<T> VectorKind;
				Handle<RegistrationCommand> command(new VectorRegCommand<T,VectorKind>);
				registration_scheduler().add_command(command);
			}
		};

} // namespace xParam_internal

#endif // XPARAM_REG_VECTOR_H
