/* RawFeatureEdit.java
 *
 * created: Tue Jan  5 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1998,1999,2000  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/RawFeatureEdit.java,v 1.7 2000/01/24 17:04:20 kmr Exp $
 **/

package diana.components;

import diana.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.Label;

/**
 *  Components of this class are used to edit a Feature in raw embl form.
 *
 *  @author Kim Rutherford
 *  @version $Id: RawFeatureEdit.java,v 1.7 2000/01/24 17:04:20 kmr Exp $
 **/

public class RawFeatureEdit
    implements EntryChangeListener, FeatureChangeListener,
               java.awt.event.TextListener{
  /**
   *  Create a new RawFeatureEdit object from the given Feature.
   **/
  public RawFeatureEdit (Feature edit_feature) {
//    System.out.println ("RawFeatureEdit constructor 1");

    this.edit_feature = edit_feature;
    this.edit_entry = edit_feature.getEntry ();
    
    file_viewer = new FileViewer ("Artemis Raw Feature Edit: " +
                                  edit_feature.getIDString ());

    file_viewer.add (error_text, "North");

    readFeature ();

    file_viewer.text_area.setEditable (true); // XXX
    file_viewer.text_area.addTextListener (this);

    edit_feature.getEntry ().addEntryChangeListener (this);
    edit_feature.addFeatureChangeListener (this);

    file_viewer.addWindowListener (new WindowAdapter () {
      public void windowClosing (WindowEvent event) {
        stopListening ();
      }
    });

  }

  /**
   *  Remove this object as a feature and entry change listener.
   **/
  public void stopListening () {
    getEntry ().removeEntryChangeListener (this);
    getFeature ().removeFeatureChangeListener (this);
  }

  /**
   *  Implementation of the EntryChangeListener interface.  We listen to
   *  EntryChange events so we can notify the user if of this component if the
   *  feature gets deleted.
   **/
  public void entryChanged (EntryChangeEvent event) {
//    System.out.println ("RawFeatureEdit.entryChanged ()");
    switch (event.getType ()) {
    case EntryChangeEvent.FEATURE_DELETED:
      if (event.getFeature () == edit_feature) {
        stopListening ();
        file_viewer.dispose ();
      }
      break;
    default:
      // do nothing;
      break;
    }
  }

  /**
   *  Implementation of the TextListener interface.  When the text changes we
   *  update the Feature object that we are showing.
   **/
  public void textValueChanged (java.awt.event.TextEvent event) {
    if (current_text.equals (file_viewer.text_area.getText ())) {
      // the text hasn't really changed
      return;
    }
    
    current_text = file_viewer.text_area.getText ();

//    System.out.println ("RawFeatureEdit.textValueChanged ()");

    try {
      // ignore text change events while reading
      edit_feature.setFromString (current_text);
      error_text.setText ("");
    } catch (uk.ac.sanger.pathogens.embl.ReadFormatException e) {
      error_text.setText (e + " at line: " + e.getLineNumber ());
    } catch (java.io.IOException e) {
      error_text.setText (e.toString ());
    } catch (uk.ac.sanger.pathogens.embl.InvalidRelationException e) {
      error_text.setText (e.toString ());
    }
  }

  /**
   *  Implementation of the FeatureChangeListener interface.  We need to
   *  listen to feature change events from the Features in this object so that
   *  we can update the display.
   *  @param event The change event.
   **/
  public void featureChanged (FeatureChangeEvent event) {

//      System.out.println ("RawFeatureEdit: feature change event type: " +
//                          event.getType () + " source " +
//                          event.getSource ());

    if (event.getSource () == current_text) {
      // don't bother with events from us
      return;
    }
    
    // re-read the information from the feature
    readFeature ();
  }

  /**
   *  Read edit_feature into this RawFeatureEdit object.
   **/
  public void readFeature () {
//    System.out.println ("RawFeatureEdit.readFeature ()");
    try {
      file_viewer.readFile (edit_feature.toReader ());

      current_text = file_viewer.text_area.getText ();
    
    } catch (uk.ac.sanger.pathogens.embl.ReadFormatException e) {
      error_text.setText (e.getMessage () + " at line: " + e.getLineNumber ());
    } catch (java.io.IOException e) {
      error_text.setText (e.getMessage ());
    } finally {
    }
  }

  /**
   *  Return the Feature we are editing.
   **/
  public Feature getFeature () {
    return edit_feature;
  }
  
  /**
   *  Return the Entry that contains the Feature this object is displaying.
   **/
  private Entry getEntry () {
    return edit_entry;
  }

  /**
   *  The Feature this object is displaying.
   **/
  private Feature edit_feature;

  /**
   *  The Entry that contains the Feature this object is displaying.
   **/
  private Entry edit_entry;

  /**
   *  The FileViewer object that is displaying the feature.
   **/
  private FileViewer file_viewer;

  /**
   *  The text version of the Feature we are currently displaying.
   **/
  private String current_text = "";

  /**
   *  A Label for showing errors and messages.
   **/
  private Label error_text = new Label ("");
}
