/* TextRequester.java
 *
 * created: Mon Jan 11 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1998,1999,2000  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/TextRequester.java,v 1.7 2000/07/11 15:49:45 kmr Exp $
 **/

package diana.components;

import java.awt.*;
import java.awt.event.*;
import java.util.Vector;

/**
 *  A popup TextField with an OK and a Cancel button.
 *
 *  @author Kim Rutherford
 *  @version $Id: TextRequester.java,v 1.7 2000/07/11 15:49:45 kmr Exp $
 **/

public class TextRequester extends Frame {
  /**
   *  Create a new TextRequester component with the given prompt. Other
   *  components can listen for TextRequesterEvent object.
   *  @param prompt A message that is displayed in the component beside the
   *    TextArea that the user types into.  This String is also used as the
   *    Frame title.
   *  @param width The width of the TextField in the new requester.
   *  @param initial_text The initial text to put in the TextField.
   **/
  public TextRequester (final String prompt,
                        final int width,
                        final String initial_text) {
    super (prompt);

    add (new Label (prompt), "North");

    final Panel panel = new Panel ();

    panel.add (ok_button);
    ok_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent e) {
        performOK ();
      }
    });

    panel.add (cancel_button);
    cancel_button.addActionListener (new ActionListener () {
      public void actionPerformed (ActionEvent e) {
        performCancel ();
      }
    });

    add (panel, "South");

    text_field = new TextField (initial_text, width);

    text_field.addKeyListener (new KeyAdapter () {
      public void keyTyped(final KeyEvent e) {
//    System.out.println ("key:" + e.getKeyChar () + "  " + e.getKeyCode ());


//  disabled because this causes a lockup under jdk1.1
//          if (e.getKeyChar () == '\n') {
//            performOK ();
//          }

      }
    });
    
    add (text_field, "Center");

    pack ();

    final Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
    setLocation (new Point ((screen.width - getSize ().width) / 2,
                            (screen.height - getSize ().height) / 2));
  }

  /**
   *  Add the given object as a listen for TextRequester events from this
   *  TextRequester.
   **/
  public void addTextRequesterListener (final TextRequesterListener l) {
    listeners.addElement (l);
  }

  /**
   *  Return the text the is currently displayed in this requester.
   **/
  protected String getText () {
    return text_field.getText ();
  }

  /**
   *  Send a TextRequesterEvent of type OK to all the listeners.
   **/
  protected void performOK () {
    final TextRequesterEvent new_event =
      new TextRequesterEvent (this, getText (), TextRequesterEvent.OK);

    sendEvent (new_event);

    TextRequester.this.dispose ();
  }

  /**
   *  Send a TextRequesterEvent of type CANCEL to all the listeners.
   **/
  protected void performCancel () {
    final TextRequesterEvent new_event =
      new TextRequesterEvent (this, getText (), TextRequesterEvent.CANCEL);

    sendEvent (new_event);

    TextRequester.this.dispose ();
  }

  /**
   *  Send the given TextRequesterEvent to all the object that are listening
   *  for the event.
   **/
  private void sendEvent (final TextRequesterEvent event) {
    for (int i = 0 ; i < listeners.size () ; ++i) {
      final TextRequesterListener listener =
        ((TextRequesterListener) listeners.elementAt (i));

      listener.actionPerformed (event);
    }
  }

  private final Button ok_button = new Button ("OK");
  private final Button cancel_button = new Button ("Cancel");
  private TextField text_field = null;

  /**
   *  This contains the objects that are listening for TextRequester events
   *  from this TextRequester.
   **/
  private Vector listeners = new Vector ();
}
