/* FastVector.java
 *
 * created: Sun Feb 20 2000
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 2000  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/uk/ac/sanger/pathogens/FastVector.java,v 1.2 2000/03/20 10:29:02 kmr Exp $
 */

package uk.ac.sanger.pathogens;


import collections.Dynarray;
import collections.Comparator;
import java.util.Hashtable;

/**
 *  This class implements a Vector of Objects with a fast version of
 *  contains().
 *
 *  @author Kim Rutherford <kmr@sanger.ac.uk>
 *  @version $Id: FastVector.java,v 1.2 2000/03/20 10:29:02 kmr Exp $
 *
 **/

public class FastVector {
  /**
   *  Create a new vector of objects.
   **/
  public FastVector () {

  }

  /**
   *  Performs the same function as Vector.addElement ()
   */
  public void add (Object object) {
    if (contains (object)) {
      throw new Error ("internal error - object added a second time");
    }

    vector.insertLast (object);
    table.put (object, object);
  }

  /**
   *  Add an Object to the end of the Vector.
   **/
  public final void addElementAtEnd (Object object) {
    if (contains (object)) {
      throw new Error ("internal error - object added a second time");
    }

    vector.insertLast (object);
    table.put (object, object);
  }

  /**
   *  Performs the same function as Vector.elementAt ()
   */
  public Object elementAt (int index) {
    return (Object) vector.at (index);
  }

  /**
   *  Performs the same function as Vector.lastElement ()
   **/
  public Object lastElement () {
    return (Object) vector.last ();
  }

  /**
   *  Performs the same function as Vector.removeElement ()
   **/
  public boolean remove (Object object) {
    if (contains (object)) {
      vector.removeOneOf (object);
      if (table.remove (object) == null) {
        throw new Error ("internal error - remove could not find target");
      }
      return true;
    } else {
      return false;
    }
  }

  /**
   *  Return true if this object contains the given Object.
   **/
  public boolean contains (Object object) {
    return table.containsKey (object);
  }

  /**
   *  Performs the same function as Vector.removeElement ()
   **/
  public int indexOf (Object object) {
    if (!contains (object)) {
      return -1;
    } else {
      return vector.firstIndexOf (object);
    }
  }

  /**
   *  Performs the same function as Vector.size ()
   */
  public int size () {
    return vector.size ();
  }

  /**
   *  Performs the same function as Vector.removeAllElement ()
   **/
  public void removeAllElements () {
    vector.clear ();
    table.clear ();
  }

  /**
   *  Performs the same function as Vector.removeElementAt ()
   **/
  public void removeElementAt(int index) {
    final Object object = (Object) vector.at (index);
    vector.removeAt (index);
    table.remove (object);
  }

  /**
   *  Performs the same function as Vector.insertElementAt ()
   **/
  public final void insertElementAt (Object object, int index) {
    vector.insertAt (index, object);
    table.put (object, object);
  }

  /**
   *  Insert an Object after another.
   *  @param old_object The new_object will be inserted after this object
   *    or at the start if old_object isn't in the vector.
   *  @param new_object The new object to insert.
   **/
  public void insertElementAfter (Object old_object, Object new_object) {
    final int old_object_index = indexOf (old_object);

    if (old_object_index == -1) {
      insertElementAt (new_object, 0);
    } else {
      insertElementAt (new_object, old_object_index + 1);
    }
    table.put (new_object, new_object);
  }

  /**
   *  Replace the Object at the given index.  (Performs the same function as
   *  Vector.elementAt())
   **/
  public void setElementAt (final Object object, final int index) {
    removeElementAt (index);
    insertElementAt (object, index);
  }

  /**
   *  Create a new FastVector with the same contents as this one.  Note that
   *  this does only a shallow copy.
   **/
  public Object clone () {
    final FastVector return_vector = new FastVector ();

    for (int i = 0 ; i < vector.size () ; ++i) {
      return_vector.vector.insertLast (vector.at (i));
    }

    return_vector.table = (Hashtable) table.clone ();
    return return_vector;
  }

  /**
   *  Return a sorted copy of this vector.
   *  @param cmp The returned vector will be sorted with this Comparator.
   **/
  public FastVector sort (final Comparator cmp) {
    final FastVector return_vector = (FastVector) clone ();

    return_vector.vector.sort (cmp);

    return return_vector;
  }

  /**
   *  The initial size used for the Vector and the Hashtable.
   **/
  final static private int INITIAL_CAPACITY = 20;

  /**
   *  Storage for objects.
   **/
  private Dynarray vector = new Dynarray ();

  /**
   *  This is used for fast lookup of objects in contains ().
   **/
  private Hashtable table = new Hashtable (INITIAL_CAPACITY);
}
