/* FastaStreamSequence.java
 *
 * created: Mon Jun 14 1999
 *
 * This file is part of Artemis
 * 
 * Copyright (C) 1999  Genome Research Limited
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/uk/ac/sanger/pathogens/embl/FastaStreamSequence.java,v 1.3 2000/02/24 12:58:23 kmr Exp $
 */

package uk.ac.sanger.pathogens.embl;

import java.io.IOException;
import java.io.Writer;

/**
 *  This is a subclass of StreamSequence containing Fasta sequence.
 *
 *  @author Kim Rutherford
 *  @version $Id: FastaStreamSequence.java,v 1.3 2000/02/24 12:58:23 kmr Exp $
 **/

public class FastaStreamSequence extends RawStreamSequence {
  /**
   *  Create a new FastaStreamSequence object from a stream.  The next line to
   *  read from the stream should be the header line of the sequence.
   *  @param in_stream The stream to read from.  When the constructor returns
   *    the stream will at the next line after the sequence.
   **/
  public FastaStreamSequence (final LinePushBackReader in_stream)
      throws IOException {
    super (in_stream);
  }

  /**
   *  Make a new FastaStreamSequence containing the same sequence as the given
   *  Sequence.  The header will be ">"
   **/
  public FastaStreamSequence (final Sequence sequence) {
    super (sequence);
    setHeader (">");
  }

  /**
   *  Make a new FastaStreamSequence containing the same sequence as the given
   *  String.
   *  @param sequence_string The String containing the sequence for the new
   *    FastaStreamSequence.
   *  @param header The header to use for the new object.
   **/
  public FastaStreamSequence (final String sequence_string,
                              final String header) {
    super (sequence_string);
    setHeader (">" + header);
  }

  /**
   *  Make a new FastaStreamSequence containing the same sequence as the given
   *  String.  The header will be ">"
   *  @param sequence_string The String containing the sequence for the new
   *    FastaStreamSequence.
   **/
  public FastaStreamSequence (final String sequence_string) {
    this (sequence_string, ">");
  }

  /**
   *  Return a new StreamSequence object that is a copy of this one.
   **/
  public StreamSequence copy () {
    return new FastaStreamSequence (this);
  }

  /**
   *  Return the sequence type (EMBL_FORMAT for this class).
   **/
  public int getFormatType () {
    return StreamSequenceFactory.FASTA_FORMAT;
  }

  /**
   *  Read the header for this sequence (if any).
   **/
  protected void readHeader (final LinePushBackReader in_stream)
      throws IOException {
    final String seq_header_line = in_stream.readLine ();

    setHeader (seq_header_line);
  }

  // uses readSequence () from the super class
  

  /**
   *  Write this Sequence to the given stream.  The output will be in FASTA
   *  format.
   *  @param writer The stream to write to.
   **/
  public void writeToStream (final Writer writer)
      throws IOException {
    writer.write (getHeader () + "\n");

    super.writeToStream (writer);
  }
}
