/////////////////////////////////////////////////////////////////////////////
// File:            gnDNXSource.h
// Purpose:         Implements gnBaseSource for .DNX files
// Description:     
// Changes:        
// Version:         libGenome 0.1.0 
// Author:          Aaron Darling 
// Last Edited:     April 15, 2001, 10:34:50pm 
// Modified by:     
// Copyright:       (c) Aaron Darling 
// Licenses:        Proprietary 
/////////////////////////////////////////////////////////////////////////////
#ifndef _gnDNXSource_h_
#define _gnDNXSource_h_

#include "gn/gnDefs.h"


#include <string>
#include <vector>
#include <fstream>
#include <time.h>

#include "gn/gnSetup.h"
#include "gn/gnFileSource.h"
#include "gn/gnSourceSpec.h"
#include "gn/gnSequence.h"

/**
 * gnDNXSource implements the DNX file source.
 * This class reads and writes the Blattner Lab DNX file format.
 * DNX files may be written by calling:
 * gnDNXSource::Write( mySpec, "C:\myFile.dnx");
 */

class GNDLLEXPORT gnDNXSource : public gnFileSource
{
public:
	/**
	 * Empty Constructor, does nothing.
	 */
	gnDNXSource();	
	/**
	 * The copy constructor copies the given gnDNXSource.
	 * @param s A gnDNXSource to copy.
	 */
	gnDNXSource( const gnDNXSource& s );
	/**
	 * Destructor, frees memory.
	 */
	~gnDNXSource();

	gnDNXSource* Clone() const;

	uint32 GetContigListLength() const;
	boolean HasContig( const string& name ) const;
	uint32 GetContigID( const string& name ) const;
	string GetContigName( const uint32 i ) const;
	gnSeqI GetContigSeqLength( const uint32 i ) const;

	boolean SeqRead( const gnSeqI start, char* buf, uint32& bufLen, const uint32 contigI=ALL_CONTIGS ) ;

	/**
	 * Writes the given gnSequence to a DNX file named "filename".
	 * Actual sequence and annotation data will be written to GenBank
	 * files.  The DNX file will then reference any created GenBank files.
	 * @param sequence The gnSequence to write out.
	 * @param filename The name of the file to write.
	 * @return True if successful, false otherwise.
	 */
	static boolean Write(gnSequence& sequence, const string& filename);
	/**
	 * This method is deprecated.
	 * Writes the given gnBaseSource to a DNX file named "filename".
	 * Actual sequence and annotation data will be written to GenBank
	 * files.  The DNX file will then reference any created GenBank files.
	 * @param source The source to write out.
	 * @param filename The name of the file to write.
	 * @return True if successful, false otherwise.
	 */
	static boolean Write(gnBaseSource *source, const string& filename);
	/**
	 * Writes the given gnGenomeSpec to a DNX file named "filename".
	 * Actual sequence and annotation data will be written to GenBank
	 * files.  The DNX file will then reference any created GenBank files.
	 * @param spec The spec to write out.
	 * @param filename The name of the file to write.
	 * @return True if successful, false otherwise.
	 */
	static boolean Write(gnGenomeSpec *spec, const string& filename);

	gnGenomeSpec* GetSpec() const;
	gnFileContig* GetFileContig( const uint32 contigI ) const;
private:
	boolean ParseStream( istream& fin );
	static void ValidateName(string& name);

	gnGenomeSpec *m_DNXSpec;
};// class gnDNXSource

inline
gnDNXSource* gnDNXSource::Clone() const
{
	return new gnDNXSource( *this );
}

inline
uint32 gnDNXSource::GetContigListLength() const
{
	return m_DNXSpec->GetSpecListLength();
}
//read sequence data from the file
inline
boolean gnDNXSource::SeqRead( const gnSeqI start, char* buf, uint32& bufLen, const uint32 contigI ) 
{
	return m_DNXSpec->SeqRead(start, buf, bufLen, contigI);
}

inline
boolean gnDNXSource::Write(gnSequence& sequence, const string& filename){
	return Write(sequence.GetSpec(), filename);
}
inline
boolean gnDNXSource::Write(gnBaseSource *source, const string& filename){
	return Write(source->GetSpec(), filename);
}
inline
gnGenomeSpec *gnDNXSource::GetSpec() const{
	return m_DNXSpec->Clone();
}

#endif
	// _gnDNXSource_h_

