//----------------------------------------------------------------------------
//
// Module:      JDialogDocumentHandler.java      
//
// Description: Handles JDialog elements
//
// FESI Copyright (c) Jean-Marc Lugrin, 2000
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//----------------------------------------------------------------------------

package FesiForm;

import org.xml.sax.*;
import FESI.jslib.*;

import javax.swing.*;
import java.awt.*;
import java.util.Vector;


public class JDialogDocumentHandler extends DefaultDocumentHandler 
                                   implements DocumentHandler {

    DocumentHandler previousDocumentHandler;
    String script = null;
    JDialog dialog = null;
    String containerName;
    
    public JDialogDocumentHandler(String cn,
                                    DocumentHandler pdh, 
                                    GuiContext guiContext, 
                                    boolean trace) {
        super(guiContext, trace);
        this.containerName = cn;
        this.previousDocumentHandler = pdh;
        // Validate assumption
        if (!cn.equals("JDialog")) throw new IllegalArgumentException("** Unexpected container name for JDialog: " + cn);
    }
  
      
    // Receive notification of the beginning of an element.
    public void startElement(String name, AttributeList atts) throws SAXException {
        if (trace) System.out.println("JDialog START_ELEMENT: <" + name + ">, " + 
                                         atts.getLength() + " attribute(s)");
        if (name.equals("JDialog")) {
            String id = getAttribute("id", atts);
            if (id==null) {
                throw new SAXException("Missing ID in JDialog");
            } else {
                String ownerId = getAttribute("ownerId", atts);
                Frame owner=null;
                if (ownerId != null) {
                    Object oi = guiContext.getById(ownerId);
                    if (oi == null) {
                        throw new SAXException("ownerId " + ownerId + " is unknown");
                    }
                    if (! (oi instanceof Frame)) {
                        throw new SAXException("Id " + ownerId + " does not refer to a Frame");
                    }
                    owner = (Frame) oi;
                }
                if (owner==null) {
                    dialog = new JDialog();
                } else {
                    dialog = new JDialog(owner);
                }
                guiContext.putId(id, dialog);
                processComponentAttributes(dialog, atts);
                processRESIZABLE(dialog, atts);
                processTITLE(dialog, atts);
                processMODAL(dialog, atts);
                processLAYOUT(dialog.getContentPane(), atts);
                processDEFAULTCLOSEOPERATION(dialog, atts);
                script = saveScript(dialog, atts);
                DocumentHandler dh = 
                    new ContentDocumentHandler("JDialog", this, dialog, dialog.getContentPane(), guiContext, trace, -1);
                guiContext.parser.setDocumentHandler(dh);
                dh.startElement(name, atts);
            }
        } else {
            throw new SAXException("Unexpected ELEMENT in dialog ignored: " + name);
        }
    }

    // Receive notification of the end of an element. 
    public void endElement(String name) throws SAXException {
        if (trace) System.out.println("JDialog END_ELEMENT: </" + name + ">");
        if (name.equals("JDialog")) {  
            processScript(dialog, script);
            previousDocumentHandler.endElement(name);
        } else {
            throw new SAXException("Unexpected ELEMENT end in dialog ignored: " + name);
        }
    }
    
     protected String processTITLE(JDialog cp, AttributeList atts) {
         String title = getAttribute("title", atts);
         if (title != null) {
             if (trace) System.out.println("  TITLE: " + title);
             cp.setTitle(title); 
         }
         return title;
     }


     protected int processDEFAULTCLOSEOPERATION(JDialog cp, AttributeList atts)  throws SAXException {
         String op = getAttribute("defaultCloseOperation", atts);
         int opCode = -1;
         if (op != null) {
             if (trace) System.out.println("  DEFAULTCLOSEOPERATION: " + op);
             opCode = operationByName(op); 
             cp.setDefaultCloseOperation(opCode);
         }
         return opCode;
     }
     
     protected boolean processMODAL(JDialog jd, AttributeList atts) throws SAXException {
         boolean modal = false;
         String tf = getAttribute("modal", atts);
         if (tf != null) {
             tf = tf.toLowerCase();
             if (trace) System.out.println("  MODAL: " + tf);
             if (tf.equals("true")) {
                 modal = true;
             } else if (!tf.equals("false")) {
                 throw new SAXException("Bad true/false value for modal: " + tf);
             }
             jd.setModal(modal);
         }
         return modal;
     }
       
     int operationByName(String opName) throws SAXException {
         String cbn = opName.toLowerCase();
         if (cbn.equals("nothing")) {
             return JDialog.DO_NOTHING_ON_CLOSE;
         } else if (cbn.equals("hide")) { 
             return JDialog.HIDE_ON_CLOSE;
         } else if (cbn.equals("dispose")) {
             return JDialog.DISPOSE_ON_CLOSE;
         }
         throw new SAXException("Unrecognized on close operation: " + opName);
     }     

     protected boolean processRESIZABLE(JDialog jb, AttributeList atts) throws SAXException {
         boolean tf = false;
         String tfString = getAttribute("resizable", atts);
         if (tfString != null) {
             tfString = tfString.toLowerCase();
             if (trace) System.out.println("  RESIZABLE: " + tfString);
             if (tfString.equals("true")) {
                 tf = true;
             } else if (!tfString.equals("false")) {
                 throw new SAXException("Bad true/false value for resizeable: " + tf);
             }
             jb.setResizable(tf);
         }
         return tf;
     }
}