--          This file is part of SmartEiffel The GNU Eiffel Compiler.
--       Copyright (C) 1994-2002 LORIA - INRIA - U.H.P. Nancy 1 - FRANCE
--          Dominique COLNET and Suzanne COLLIN - SmartEiffel@loria.fr
--                       http://SmartEiffel.loria.fr
-- SmartEiffel is  free  software;  you can  redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by the Free
-- Software  Foundation;  either  version  2, or (at your option)  any  later
-- version. SmartEiffel is distributed in the hope that it will be useful,but
-- WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or  FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License
-- for  more  details.  You  should  have  received a copy of the GNU General
-- Public  License  along  with  SmartEiffel;  see the file COPYING.  If not,
-- write to the  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
-- Boston, MA 02111-1307, USA.
--
deferred class INSTRUCTION
   --
   -- For all differents kinds of Eiffel instruction.
   --

inherit GLOBALS

feature

   start_position: POSITION is
         -- Of the first character of the instruction.
      deferred
      ensure
         not Result.is_unknown
      end

   pretty_print is
      require
         pretty_printer.indent_level >= 3
      deferred
      ensure
         pretty_printer.indent_level = old pretty_printer.indent_level
      end

   use_current: BOOLEAN is
         -- Does instruction use Current ?
      require
         smart_eiffel.is_ready
      deferred
      end

   stupid_switch(run_time_set: RUN_TIME_SET): BOOLEAN is
      require
         smart_eiffel.is_ready
         ace.boost
         run_time_set.count > 1
      deferred
      end

   to_runnable(ct: E_TYPE): INSTRUCTION is
         -- Gives a checked instruction runnable in `ct'.
      require
         ct.run_type = ct
         ct.run_class /= Void
      deferred
      ensure
         nb_errors = 0 implies Result /= Void
      end

   end_mark_comment: BOOLEAN is
         -- True for instructions with a possible end mark comment
         -- like instruction "loop" "debug" or "check" for example.
      deferred
      end

   afd_check is
         -- After Falling Down Check.
      deferred
      end

   safety_check is
      require
	 for_boost_mode_only_or_asked_for: ace.boost or else ace.safety_check
         smart_eiffel.status.in_range(3,6)
      deferred
      end

   collect_c_tmp is
         -- Traverse the instruction to collect extra mandatory C tmp
	 -- variables (for user expanded) just before `compile_to_c'.
      require
         smart_eiffel.is_ready
      deferred
      end

   compile_to_c is
      require
         smart_eiffel.is_ready
         cpp.on_c
      deferred
      ensure
         cpp.on_c
      end

   compile_to_jvm is
      require
         smart_eiffel.is_ready
      deferred
      end

   is_pre_computable: BOOLEAN is
         -- Assume the current instruction is inside a once function.
         -- Result is true when the instruction can be precomputed.
      require
         smart_eiffel.is_ready
      deferred
      end

feature {COMPOUND,INSTRUCTION_WITH_COMMENT}

   verify_scoop(allowed: FORMAL_ARG_LIST) is
      deferred
      end

feature {EIFFEL_PARSER}

   frozen add_comment(c: COMMENT): INSTRUCTION is
         -- Attach `c' to the instruction.
      require
         eiffel_parser.is_running
	 c.count > 0
      do
	 if end_mark_comment then
            if c.count = 1 then
               Result := Current
            else
               !INSTRUCTION_WITH_COMMENT!Result.make(Current,c)
            end
         else
            !INSTRUCTION_WITH_COMMENT!Result.make(Current,c)
         end
      end

feature {NONE}

   pretty_print_assignment(rhs: EXPRESSION; op: STRING; lhs: EXPRESSION) is
      local
         semi_colon_flag: BOOLEAN
      do
         rhs.pretty_print
         pretty_printer.put_character(' ')
         pretty_printer.put_string(op)
         pretty_printer.put_character(' ')
         semi_colon_flag := pretty_printer.semi_colon_flag
         pretty_printer.level_incr
         pretty_printer.set_semi_colon_flag(false)
         lhs.pretty_print
         pretty_printer.set_semi_colon_flag(semi_colon_flag)
         if semi_colon_flag then
            pretty_printer.put_character(';')
         end
         pretty_printer.level_decr
      end

end -- INSTRUCTION

