/*
**  CXSC is a C++ library for eXtended Scientific Computing
**
**  Copyright (C) 1990-2000 Institut fuer Angewandte Mathematik,
**                          Universitaet Karlsruhe, Germany
**            (C) 2000-2001 Wiss. Rechnen/Softwaretechnologie
**                          Universitaet Wuppertal, Germany   
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Library General Public
**  License as published by the Free Software Foundation; either
**  version 2 of the License, or (at your option) any later version.
**
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**  Library General Public License for more details.
**
**  You should have received a copy of the GNU Library General Public
**  License along with this library; if not, write to the Free
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* CVS $Id: intmatri.hpp,v 1.4 2001/08/09 12:52:39 cxsc Exp $ */

#ifndef _CXSC_INTMATRIX_HPP_INCLUDED
#define _CXSC_INTMATRIX_HPP_INCLUDED

#include "xscclass.hpp"

#include "dot.hpp"
#include "intvecto.hpp"
#include "except.hpp"
#include "matrix.hpp"

namespace cxsc {

class intmatrix;
class intmatrix_slice;

class intmatrix_subv
{
	friend class intvector;
	friend class intmatrix;
	friend class intmatrix_slice;
	private:
	int *dat;
	int lb,ub;
	int size,start,offset; // start=first element index 0..n-1
	
	public:
	friend INLINE intmatrix_subv Row(intmatrix &m,const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
	friend INLINE intmatrix_subv Col(intmatrix &m,const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
//#if(IndCheck)
#ifdef _CXSC_FRIEND_TPL
	//----------------- Templates ---------------------------------------
template <class MV1,class MV2> friend  MV1 &_mvmvassign(MV1 &v,const MV2 &rv)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<MV1>);
#else
	throw();
#endif
template <class MV,class S> friend  MV &_mvsassign(MV &v,const  S &r) throw();
template <class MV,class V> friend  MV &_mvvassign(MV &v,const V &rv)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<MV>);
#else
	throw();
#endif
template <class V,class MV2,class S> friend  V &_vmvassign(V &v,const MV2 &rv) throw();
template <class MV,class V> friend  V _mvabs(const MV &mv) throw();
template <class DP,class V,class SV> friend 	 void _vmvaccu(DP &dp, const V & rv1, const SV &rv2)
#if(IndCheck)
		throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
template <class DP,class MV1,class MV2> friend 	 void _mvmvaccu(DP &dp, const MV1 & rv1, const MV2 &rv2)
#if(IndCheck)
		throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
 template <class MV,class S,class E> friend 	 E _mvsmult(const MV &rv, const S &s) throw();
 template <class MV1,class MV2,class E> friend 	 E _mvmvplus(const MV1 &rv1, const MV2 &rv2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MV1,class MV2,class E> friend 	 E _mvmvminus(const MV1 &rv1, const MV2 &rv2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MV,class V,class E> friend 	 E _mvvplus(const MV &rv1, const V &rv2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MV,class V,class E> friend 	 E _mvvminus(const MV &rv1, const V &rv2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class V,class MV,class E> friend 	 E _vmvminus(const V &rv1, const MV &rv2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MV,class S,class E> friend 	 E _mvsdiv(const MV &rv, const S &s) throw();
template <class MV,class S> friend  MV &_mvsmultassign(MV &v,const S &r) throw();
template <class MV, class S> friend  MV &_mvsplusassign(MV &v,const S &r) throw();
template <class MV,class S> friend  MV &_mvsminusassign(MV &v,const S &r) throw();
template <class MV,class S> friend  MV &_mvsdivassign(MV &v,const S &r) throw();
template <class MV,class V> friend  MV &_mvvplusassign(MV &v,const V &rv)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<MV>);
#else
	throw();
#endif
template <class V,class MV> friend  V &_vmvplusassign(V &rv,const MV &v)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<V>);
#else
	throw();
#endif
template <class MV,class V> friend  MV &_mvvminusassign(MV &v,const V &rv)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<MV>);
#else
	throw();
#endif
template <class V,class MV> friend  V &_vmvminusassign(V &rv,const MV &v)
#if(IndCheck)
	throw(ERROR__OP_WITH_WRONG_DIM<V>);
#else
	throw();
#endif

#endif
	
	//----------------- Konstruktoren ----------------------------------

	explicit INLINE intmatrix_subv (int *d, const int &l, const int &u, const int &s, const int &st, const int &o) throw():dat(d),lb(l),ub(u),size(s),start(st),offset(o) { }
	protected:
	INLINE intmatrix_subv(const intmatrix_subv &v) throw():dat(v.dat),lb(v.lb),ub(v.ub),size(v.size),start(v.start),offset(v.offset) { }
	public:

	//---------------------- Standardfunktionen ------------------------

	friend INLINE intvector::intvector(const intmatrix_subv &) throw();
	INLINE intmatrix_subv &operator =(const intmatrix_subv &rv) throw();
	INLINE intmatrix_subv &operator =(const int &r) throw();
	INLINE intmatrix_subv &operator =(const intmatrix &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_TYPE_CAST_OF_THICK_OBJ);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator =(const intmatrix_slice &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_TYPE_CAST_OF_THICK_OBJ);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator =(const intvector &v)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator =(const intvector_slice &v)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	friend INLINE int Lb(const intmatrix_subv &rv) throw() { return rv.lb; }
	friend INLINE int Ub(const intmatrix_subv &rv) throw() { return rv.ub; }
	INLINE int &operator [](const int &i) const
#if(IndCheck)
	throw(ERROR_INTVECTOR_ELEMENT_NOT_IN_VEC);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator ()() throw() { return *this; }
	INLINE intmatrix_subv operator ()(const int &i)
#if(IndCheck)
	throw(ERROR_INTVECTOR_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	INLINE intmatrix_subv operator ()(const int &i1,const int &i2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	
	INLINE intmatrix_subv &operator *=(const int &c) throw();
	INLINE intmatrix_subv &operator +=(const int &c) throw();
	INLINE intmatrix_subv &operator -=(const int &c) throw();
	INLINE intmatrix_subv &operator /=(const int &c) throw();
	INLINE intmatrix_subv &operator -=(const intvector &rv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator -=(const intvector_slice &rv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator +=(const intvector &rv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_subv &operator +=(const intvector_slice &rv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
//#else
//#endif	

};


//----------------------- Matrix -----------------------------------------------

class intmatrix_slice;

class intmatrix
{
	friend class intmatrix_slice;
	friend class intmatrix_subv;
	private:
	int *dat;
	int lb1,ub1,lb2,ub2,xsize,ysize;

	public:
//#if(IndCheck)
#ifdef _CXSC_FRIEND_TPL
	//----------------- Templates ---------------------------------------
template <class S,class M> friend  void _smconstr(S &s,const M &m)
#if(IndCheck)
	throw(ERROR__TYPE_CAST_OF_THICK_OBJ<M>,ERROR__USE_OF_UNINITIALIZED_OBJ<M>);
#else
	throw();
#endif
template <class V,class M,class S> friend  void _vmconstr(V &v,const M &m)
#if(IndCheck)
	throw(ERROR__TYPE_CAST_OF_THICK_OBJ<M>);
#else
	throw();
#endif
 template <class M1,class M2,class S> friend 	 M1 &_mmassign(M1 &m1,const M2 &m,S ms) throw();
 template <class M,class MS2,class S> friend 	 M &_mmsassign(M &m,const MS2 &ms) throw();
 template <class MS,class M> friend 	 MS &_msmassign(MS &ms,const M &m)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
 template <class M,class S> friend 	 M &_msassign(M &m,const S &r) throw();
template <class V,class M,class S> friend  V &_vmassign(V &v,const M &m)
#if(IndCheck)
	throw(ERROR__TYPE_CAST_OF_THICK_OBJ<M>);
#else
	throw();
#endif
template <class M,class V,class S> friend  M &_mvassign(M &m,const V &v) throw();
 template <class M> friend 	 int _mlb(const M &m, const int &i)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<M>);
#else
	throw();
#endif
 template <class M> friend 	 int _mub(const M &m, const int &i)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<M>);
#else
	throw();
#endif
 template <class M> friend 	 M &_msetlb(M &m, const int &i,const int &j)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<M>);
#else
	throw();
#endif
 template <class M> friend 	 M &_msetub(M &m, const int &i,const int &j)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<M>);
#else
	throw();
#endif
 template <class M> friend 	 void _mresize(M &A) throw();
 template <class M,class S> friend 	 void _mresize(M &A,const int &m, const int &n)
#if(IndCheck)
		throw(ERROR__WRONG_BOUNDARIES<M>);
#else
	throw();
#endif
 template <class M,class S> friend 	 void _mresize(M &A,const int &m1, const int &m2,const int &n1,const int &n2)
#if(IndCheck)
		throw(ERROR__WRONG_BOUNDARIES<M>);
#else
	throw();
#endif
 template <class M,class E> friend 	 E _mabs(const M &m) throw();
 template <class MS,class E> friend 	 E _msabs(const MS &ms) throw();
	//------- matrix-matrix --------------
 template <class M1,class M2,class E> friend 	 E _mmplus(const M1 &m1,const M2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M1>);
#else
	throw();
#endif
 template <class M,class MS,class E> friend 	 E _mmsplus(const M &m,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS1,class MS2,class E> friend 	 E _msmsplus(const MS1 &m1,const MS2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class M> friend 	 M _mminus(const M &m) throw();
 template <class MS,class E> friend 	 E _msminus(const MS &ms) throw();
 template <class M1,class M2,class E> friend 	 E _mmminus(const M1 &m1,const M2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M1>);
#else
	throw();
#endif
 template <class M1,class M2> friend 	 M1 &_mmplusassign(M1 &m1,const M2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M1>);
#else
	throw();
#endif
 template <class M,class MS> friend 	 M &_mmsplusassign(M &m1,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS,class M> friend 	 MS &_msmplusassign(MS &ms,const M &m1)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
 template <class M,class MS,class E> friend 	 E _mmsminus(const M &m,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MS,class M,class E> friend 	 E _msmminus(const MS &ms,const M &m)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MS1,class MS2,class E> friend 	 E _msmsminus(const MS1 &ms1,const MS2 &ms2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class M1,class M2> friend 	 M1 &_mmminusassign(M1 &m1,const M2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M1>);
#else
	throw();
#endif
 template <class M,class MS> friend 	 M &_mmsminusassign(M &m1,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS,class M> friend 	 MS &_msmminusassign(MS &ms,const M &m1)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
	//-------- matrix-scalar ---------------------
 template <class S,class M,class E> friend 	 E _smmult(const S &c, const M &m) throw();
 template <class M,class S> friend 	 M &_msmultassign(M &m,const S &c) throw();
 template <class S,class MS,class E> friend 	 E _smsmult(const S &c, const MS &ms) throw();
 template <class M,class S,class E> friend 	 E _msdiv(const M &m,const S &c) throw();
 template <class M,class S> friend 	 M &_msdivassign(M &m,const S &c) throw();
 template <class MS,class S,class E> friend 	 E _mssdiv(const MS &ms, const S &c) throw();
	//--------- matrix-vector --------------------

 template <class M> friend 	 void *_mvoid(const M &m) throw();
 template <class M> friend 	 bool _mnot(const M &m) throw();
 template <class MS> friend 	 void *_msvoid(const MS &ms) throw();
 template <class MS> friend 	 bool _msnot(const MS &ms) throw();
 template <class M1,class M2> friend 	 bool _mmeq(const M1 &m1,const M2 &m2) throw();
 template <class M1,class M2> friend 	 bool _mmneq(const M1 &m1,const M2 &m2) throw();
 template <class M1,class M2> friend 	 bool _mmless(const M1 &m1,const M2 &m2) throw();
 template <class M1,class M2> friend 	 bool _mmleq(const M1 &m1,const M2 &m2) throw();
 template <class M,class MS> friend 	 bool _mmseq(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsneq(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsless(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsleq(const M &m1,const MS &ms) throw();
 template <class MS,class M> friend 	 bool _msmless(const MS &ms,const M &m1) throw();
 template <class MS,class M> friend 	 bool _msmleq(const MS &ms,const M &m1) throw();
 template <class M> friend 	std::ostream &_mout(std::ostream &s,const M &r) throw();
 template <class M> friend 	std::istream &_min(std::istream &s,M &r) throw();

#endif

	//--------------------------  Konstruktoren ----------------------------

	INLINE intmatrix(const intmatrix &rm) throw();
	INLINE intmatrix(const intmatrix_slice &rm) throw();
	INLINE intmatrix() throw();
	explicit INLINE intmatrix(const int &m, const int &n)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_BOUNDARIES);
#else
	throw();
#endif
	explicit INLINE intmatrix(const int &m1, const int &n1, const int &m2, const int &n2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_BOUNDARIES);
#else
	throw();
#endif
	explicit INLINE intmatrix(const intvector &v) throw();
	explicit INLINE intmatrix(const intvector_slice &v) throw();
	explicit INLINE intmatrix(const int &r) throw();
	INLINE intmatrix &operator =(const int &r) throw();
	INLINE intmatrix &operator =(const intmatrix &m) throw();
	INLINE intmatrix &operator =(const intmatrix_slice &ms) throw();
	INLINE intmatrix &operator =(const intvector &v) throw();
	INLINE intmatrix &operator =(const intvector_slice &v) throw();

	//--------------------------- Destruktoren -----------------------------

	INLINE ~intmatrix() throw() { delete [] dat; }

	//------------------------- Standardfunktionen -------------------------

	friend INLINE intvector::intvector(const intmatrix &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_TYPE_CAST_OF_THICK_OBJ);
#else
	throw();
#endif
	INLINE intmatrix_subv operator [](const int &i) const
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
	INLINE intmatrix_subv operator [](const cxscmatrix_column &i) const
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
	INLINE intmatrix &operator ()() throw() { return *this; }
	INLINE intmatrix_slice operator ()(const int &m, const int &n)
#if(IndCheck)
	throw(ERROR_INTMATRIX_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	INLINE intmatrix_slice operator ()(const int &m1, const int &m2, const int &n1, const int &n2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	INLINE operator void*() throw();
//#else
//#endif
};

	
class intmatrix_slice
{
	friend class intmatrix;
	private:
	int *dat;
	int offset1,offset2,mxsize,mysize;
	int start1,end1,start2,end2,sxsize,sysize;     // slice size

	public:
//#if(IndCheck)
#ifdef _CXSC_FRIEND_TPL
	//----------------- Templates ---------------------------------------
template <class V,class MS,class S> friend  void _vmsconstr(V &v,const MS &m)
#if(IndCheck)
	throw(ERROR__TYPE_CAST_OF_THICK_OBJ<MS>);
#else
	throw();
#endif
 template <class MS,class M> friend 	 MS &_msmassign(MS &ms,const M &m)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
 template <class MS1,class MS2> friend 	 MS1 &_msmsassign(MS1 &ms1,const MS2 &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS1>);
#else
	throw();
#endif
 template <class M,class MS2,class S> friend 	 M &_mmsassign(M &m,const MS2 &ms) throw();
 template <class MS,class S> friend 	 MS &_mssassign(MS &ms,const S &r) throw();
 template <class MS> friend 	 int _mslb(const MS &ms, const int &i)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<MS>);
#else
	throw();
#endif
 template <class MS> friend 	 int _msub(const MS &ms, const int &i)
#if(IndCheck)
	throw(ERROR__WRONG_ROW_OR_COL<MS>);
#else
	throw();
#endif
 template <class MS,class E> friend 	 E _msabs(const MS &ms) throw();
	//------------ matrix-matrix --------------------
 template <class MS,class E> friend 	 E _msminus(const MS &ms) throw();
 template <class M,class MS,class E> friend 	 E _mmsplus(const M &m,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS1,class MS2,class E> friend 	 E _msmsplus(const MS1 &m1,const MS2 &m2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class M,class MS> friend 	 M &_mmsplusassign(M &m1,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS,class M> friend 	 MS &_msmplusassign(MS &ms,const M &m1)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
 template <class MS1,class MS2> friend 	 MS1 &_msmsplusassign(MS1 &ms1,const MS2 &ms2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS1>);
#else
	throw();
#endif
 template <class M,class MS,class E> friend 	 E _mmsminus(const M &m,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MS,class M,class E> friend 	 E _msmminus(const MS &ms,const M &m)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class MS1,class MS2,class E> friend 	 E _msmsminus(const MS1 &ms1,const MS2 &ms2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<E>);
#else
	throw();
#endif
 template <class M,class MS> friend 	 M &_mmsminusassign(M &m1,const MS &ms)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<M>);
#else
	throw();
#endif
 template <class MS,class M> friend 	 MS &_msmminusassign(MS &ms,const M &m1)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS>);
#else
	throw();
#endif
 template <class MS1,class MS2> friend 	 MS1 &_msmsminusassign(MS1 &ms1,const MS2 &ms2)
#if(IndCheck)
		throw(ERROR__OP_WITH_WRONG_DIM<MS1>);
#else
	throw();
#endif
	//--------- matrix-scalar -----------------
 template <class S,class MS,class E> friend 	 E _smsmult(const S &c, const MS &ms) throw();
 template <class MS,class S> friend 	 MS &_mssmultassign(MS &ms,const S &c) throw();
 template <class MS,class S,class E> friend 	 E _mssdiv(const MS &ms, const S &c) throw();
 template <class MS,class S> friend 	 MS &_mssdivassign(MS &ms,const S &c) throw();

 template <class MS> friend 	 void *_msvoid(const MS &ms) throw();
 template <class MS> friend 	 bool _msnot(const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmseq(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsneq(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsless(const M &m1,const MS &ms) throw();
 template <class M,class MS> friend 	 bool _mmsleq(const M &m1,const MS &ms) throw();
 template <class MS,class M> friend 	 bool _msmless(const MS &ms,const M &m1) throw();
 template <class MS,class M> friend 	 bool _msmleq(const MS &ms,const M &m1) throw();
 template <class MS1,class MS2> friend 	 bool _msmseq(const MS1 &ms1,const MS2 &ms2) throw();
 template <class MS1,class MS2> friend 	 bool _msmsneq(const MS1 &ms1,const MS2 &ms2) throw();
 template <class MS1,class MS2> friend 	 bool _msmsless(const MS1 &ms1,const MS2 &ms2) throw();
 template <class MS1,class MS2> friend 	 bool _msmsleq(const MS1 &ms1,const MS2 &ms2) throw();
 template <class MS> friend 	std::ostream &_msout(std::ostream &s,const MS &r) throw();
 template <class MS> friend 	std::istream &_msin(std::istream &s,MS &r) throw();
	
#endif


	//--------------- Konstruktoren ----------------------------------------

	explicit INLINE intmatrix_slice(intmatrix &a,const int &l1,const int &u1,const int &l2, const int &u2) throw():dat(a.dat),offset1(l1-a.lb1),offset2(l2-a.lb2),mxsize(a.xsize),mysize(a.ysize),start1(l1),end1(u1),start2(l2),end2(u2),sxsize(u2-l2+1),sysize(u1-l1+1) { }
	explicit INLINE intmatrix_slice(intmatrix_slice &a,const int &l1,const int &u1,const int &l2, const int &u2) throw():dat(a.dat),offset1(a.offset1+l1-a.start1),offset2(a.offset2+l2-a.start2),mxsize(a.mxsize),mysize(a.mysize),start1(l1),end1(u1),start2(l2),end2(u2),sxsize(u2-l2+1),sysize(u1-l1+1) { }
	protected: 
	INLINE intmatrix_slice(const intmatrix_slice &ms) throw():dat(ms.dat),offset1(ms.offset1),offset2(ms.offset2),mxsize(ms.mxsize),mysize(ms.mysize),start1(ms.start1),end1(ms.end1),start2(ms.start2),end2(ms.end2),sxsize(ms.sxsize),sysize(ms.sysize) { }
	public:

	//---------------- Standardfunktionen -----------------------------------

	friend  intvector::intvector(const intmatrix_slice &sl)
#if(IndCheck)
	throw(ERROR_INTMATRIX_TYPE_CAST_OF_THICK_OBJ);
#else
	throw();
#endif
	friend INLINE intmatrix::intmatrix(const intmatrix_slice &) throw();
	INLINE intmatrix_slice &operator =(const intmatrix &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &operator =(const intmatrix_slice &ms)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &operator =(const int &r) throw();
	INLINE intmatrix_slice &operator =(const intvector &v)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &operator =(const intvector_slice &v)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &operator =(const intmatrix_subv &v)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_subv operator [](const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
	INLINE intmatrix_subv operator [](const cxscmatrix_column &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_ROW_OR_COL_NOT_IN_MAT);
#else
	throw();
#endif
	INLINE intmatrix_slice &operator ()() throw() { return *this; }
	INLINE intmatrix_slice operator ()(const int &m, const int &n)
#if(IndCheck)
	throw(ERROR_INTMATRIX_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	INLINE intmatrix_slice operator ()(const int &m1, const int &m2, const int &n1, const int &n2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_SUB_ARRAY_TOO_BIG);
#else
	throw();
#endif
	INLINE intmatrix_slice &intmatrix_slice::operator +=(const intmatrix &m1)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &intmatrix_slice::operator +=(const intmatrix_slice &ms2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &intmatrix_slice::operator -=(const intmatrix &m1)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &intmatrix_slice::operator -=(const intmatrix_slice &ms2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix_slice &intmatrix_slice::operator *=(const int &c) throw();
	INLINE intmatrix_slice &intmatrix_slice::operator /=(const int &c) throw();
	INLINE intmatrix_slice::operator void*() throw();
//#else
//#endif
};

//================================================================
//====================== Subvector Functions =====================

//=======================Vector / Scalar =========================

	INLINE intvector operator /(const intmatrix_subv &rv, const int &s) throw();
	INLINE intvector operator *(const intmatrix_subv &rv, const int &s) throw();
	INLINE intvector operator *(const int &s, const intmatrix_subv &rv) throw();
	INLINE intvector abs(const intmatrix_subv &mv) throw();

//======================== Vector / Vector ========================

	INLINE void accumulate(dotprecision &dp, const intmatrix_subv & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE void accumulate(dotprecision &dp, const intvector & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE void accumulate(dotprecision &dp, const intmatrix_subv & rv1, const intvector &rv2)
#if(IndCheck)
	throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE void accumulate(dotprecision &dp,const intvector_slice &sl,const intmatrix_subv &sv)
#if(IndCheck)
	throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE void accumulate(dotprecision &dp,const intmatrix_subv &mv,const intvector_slice &vs)
#if(IndCheck)
	throw(OP_WITH_WRONG_DIM);
#else
	throw();
#endif

	INLINE intvector operator +(const intmatrix_subv & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator +(const intmatrix_subv &rv1,const intvector &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator +(const intvector & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator +(const intvector_slice &sl,const intmatrix_subv &mv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator +(const intmatrix_subv &mv,const intvector_slice &sl)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	
	INLINE intvector operator -(const intmatrix_subv & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator -(const intvector & rv1, const intmatrix_subv &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator -(const intmatrix_subv &rv1,const intvector &rv2)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator -(const intvector_slice &sl,const intmatrix_subv &mv)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intvector operator -(const intmatrix_subv &mv,const intvector_slice &sl)
#if(IndCheck)
	throw(ERROR_INTVECTOR_OP_WITH_WRONG_DIM);
#else
	throw();
#endif

//====================================================================
//===================== Matrix Functions =============================

	INLINE intmatrix _intmatrix(const intmatrix &rm) throw();
	INLINE intmatrix _intmatrix(const intvector &v) throw();
	INLINE intmatrix _intmatrix(const intvector_slice &v) throw();
	INLINE intmatrix _intmatrix(const int &r) throw();

	INLINE int Lb(const intmatrix &rm, const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE int Ub(const intmatrix &rm, const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE int Lb(const intmatrix_slice &rm, const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE int Ub(const intmatrix_slice &rm, const int &i)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE intmatrix &SetLb(intmatrix &m, const int &i,const int &j)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE intmatrix &SetUb(intmatrix &m, const int &i,const int &j)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_ROW_OR_COL);
#else
	throw();
#endif
	INLINE void Resize(intmatrix &A) throw();
	INLINE void Resize(intmatrix &A,const int &m, const int &n)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_BOUNDARIES);
#else
	throw();
#endif
	INLINE void Resize(intmatrix &A,const int &m1, const int &m2,const int &n1,const int &n2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_WRONG_BOUNDARIES);
#else
	throw();
#endif

	INLINE intmatrix abs(const intmatrix &m) throw();
	INLINE intmatrix abs(const intmatrix_slice &ms) throw();

//===================== Matrix / Scalar ===============================

	INLINE intmatrix operator *(const int &c, const intmatrix &m) throw();
	INLINE intmatrix operator *(const int &c, const intmatrix_slice &ms) throw();
	INLINE intmatrix operator *(const intmatrix &m,const int &c) throw();
	INLINE intmatrix operator *(const intmatrix_slice &ms,const int &c) throw();
	INLINE intmatrix &operator *=(intmatrix &m,const int &c) throw();
	INLINE intmatrix operator /(const intmatrix &m,const int &c) throw();
	INLINE intmatrix operator /(const intmatrix_slice &ms, const int &c) throw();
	INLINE intmatrix &operator /=(intmatrix &m,const int &c) throw();


//================ Matrix / Matrix ============================

	INLINE const intmatrix &operator +(const intmatrix &m1) throw();
	INLINE intmatrix operator +(const intmatrix_slice &ms) throw();
	INLINE intmatrix operator +(const intmatrix &m1,const intmatrix &m2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator +(const intmatrix &m,const intmatrix_slice &ms)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator +(const intmatrix_slice &ms,const intmatrix &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator +(const intmatrix_slice &m1,const intmatrix_slice &m2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix &operator +=(intmatrix &m1,const intmatrix &m2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix &operator +=(intmatrix &m1,const intmatrix_slice &ms)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	
	INLINE intmatrix operator -(const intmatrix &m) throw();
	INLINE intmatrix operator -(const intmatrix_slice &ms) throw();
	INLINE intmatrix operator -(const intmatrix &m1,const intmatrix &m2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator -(const intmatrix &m,const intmatrix_slice &ms)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator -(const intmatrix_slice &ms,const intmatrix &m)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix operator -(const intmatrix_slice &ms1,const intmatrix_slice &ms2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix &operator -=(intmatrix &m1,const intmatrix &m2)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	INLINE intmatrix &operator -=(intmatrix &m1,const intmatrix_slice &ms)
#if(IndCheck)
	throw(ERROR_INTMATRIX_OP_WITH_WRONG_DIM);
#else
	throw();
#endif
	

//============== Compare Operator ==========================

//-------------- Matrix - Matrix   -------------------------

	INLINE bool operator ==(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator !=(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator <(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator <=(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator >(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator >=(const intmatrix &m1,const intmatrix &m2) throw();
	INLINE bool operator ==(const intmatrix &m1,const intmatrix_slice &ms) throw();
	INLINE bool operator !=(const intmatrix &m1,const intmatrix_slice &ms) throw();
	INLINE bool operator <(const intmatrix &m1,const intmatrix_slice &ms) throw();
	INLINE bool operator <=(const intmatrix &m1,const intmatrix_slice &ms) throw();
	INLINE bool operator >(const intmatrix &m1,const intmatrix_slice &ms) throw();
	INLINE bool operator >=(const intmatrix &m1,const intmatrix_slice &ms) throw();

//---------------- Matrix - Matrix_slice ----------------------

	INLINE bool operator ==(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();
	INLINE bool operator !=(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();
	INLINE bool operator <(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();
	INLINE bool operator <=(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();
	INLINE bool operator >(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();
	INLINE bool operator >=(const intmatrix_slice &m1,const intmatrix_slice &m2) throw();

//=================== Not Operator =============================

	INLINE bool operator !(const intmatrix &ms) throw();
	INLINE bool operator !(const intmatrix_slice &ms) throw();

//======================== Input / Output ========================

	INLINE std::ostream &operator <<(std::ostream &s,const intmatrix &r) throw();
	INLINE std::ostream &operator <<(std::ostream &s,const intmatrix_slice &r) throw();
	INLINE std::istream &operator >>(std::istream &s,intmatrix &r) throw();
	INLINE std::istream &operator >>(std::istream &s,intmatrix_slice &r) throw();


} // namespace cxsc 

#ifdef _CXSC_INCL_INL
# include "matrix.inl"
# include "intmatri.inl"
#endif

#endif

