/*
    GSK - a library to write servers
    Copyright (C) 1999-2000 Dave Benson

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

    Contact:
        daveb@ffem.org <Dave Benson>
*/


/* GskSocketAddress: A socket address (a port on a host, in TCP)  */

#ifndef __GSK_SOCKET_ADDRESS_H_
#define __GSK_SOCKET_ADDRESS_H_

#include <glib.h>

/* --- address families --- */
typedef enum
{
  GSK_SOCKET_ADDRESS_IPv4 = 100,
  GSK_SOCKET_ADDRESS_IPv6,
  GSK_SOCKET_ADDRESS_ETHERNET,
  GSK_SOCKET_ADDRESS_APPLETALK,
  GSK_SOCKET_ADDRESS_PACKET
} GskSocketAddressFamily;

/* --- structures & typedefs --- */
typedef struct _GskSocketAddressIPv4 GskSocketAddressIPv4;
typedef struct _GskSocketAddressEthernet GskSocketAddressEthernet;
typedef struct _GskSocketAddressAppletalk GskSocketAddressAppletalk;
typedef struct _GskSocketAddressPacket GskSocketAddressPacket;
typedef union _GskSocketAddress GskSocketAddress;

struct _GskSocketAddressIPv4
{
  GskSocketAddressFamily    address_family; /* == GSK_SOCKET_ADDRESS_IPv4 */
  guint8                    ip_address[4];
  guint16                   ip_port;
};

struct _GskSocketAddressEthernet
{
  GskSocketAddressFamily    address_family; /* == GSK_SOCKET_ADDRESS_ETHERNET */
  guint8                    mac_address[6];
};

struct _GskSocketAddressAppletalk
{
  GskSocketAddressFamily    address_family; /* == GSK_SOCKET_ADDRESS_APPLETALK*/
  guint16		    net;
  guint8                    node;
  guint8                    port;
};

union _GskSocketAddress
{
  int	                    address_family;
  GskSocketAddressIPv4      ipv4;
  GskSocketAddressEthernet  ethernet;
  gulong		    __padding[8];
};

/* --- prototypes --- */
char        *gsk_socket_address_to_string     (const GskSocketAddress *);
gboolean     gsk_socket_address_lookup        (GskSocketAddress *to_fill,
                                               const char       *spec,
				               int               default_port);
gboolean     gsk_socket_address_raw_lookup    (GskSocketAddress *to_fill,
                                               const char       *host,
				               int               port);
   

/* Two unfreeable addresses -- your local address and an address
 * which accepts connections from anywhere. */
GskSocketAddress* gsk_socket_address_local();
GskSocketAddress* gsk_socket_address_any();

/* convert a socketaddress from and to the native representation */
gboolean     gsk_socket_address_to_native     (const GskSocketAddress *,
                                               void             *sockaddr_ptr,
					       int              *sa_len_inout);
gboolean     gsk_socket_address_from_native   (GskSocketAddress *address,
                                               const void       *sockaddr_ptr);

void         gsk_socket_address_from_ethernet (GskSocketAddress *address,
					       char             *mac_addr);

/* --- querying the address --- */
gboolean     gsk_socket_address_is_multicast  (GskSocketAddress *address);

/* --- hash-table functions --- */
guint        gsk_socket_address_hash          (gconstpointer     address);
gint         gsk_socket_address_equal         (gconstpointer     address_a,
					       gconstpointer     address_b);

#endif
