/*
    GSK - a library to write servers
    Copyright (C) 1999-2000 Dave Benson

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

    Contact:
        daveb@ffem.org <Dave Benson>
*/


#ifndef __GSK_URL_H_
#define __GSK_URL_H_

typedef struct _GskUrl GskUrl;
typedef struct _GskUrlClass GskUrlClass;

#include "../gskbasic.h"
#include "../gskgtk.h"

/* Characters having specific meaning within a url; 
 * these should be escaped to be passed on to the underlying scheme.
 * From RFC 2396, section 2.2.
 */
#define GSK_URL_RESERVED_CHARSET			\
	";/?:@&=+$,"

/* Characters which do not normally need escaping within a URL.
 * From RFC 2396, section 2.3.
 */
#define GSK_URL_UNRESERVED_CHARSET			\
	"abcdefghijklmnopqrstuvwxyz"			\
	"ABCDEFGHIJKLMNOPQRSTUVWXYZ"			\
	"0123456789"					\
	"-_.!~*'()"

typedef enum 
{
  GSK_URL_SCHEME_FILE,
  GSK_URL_SCHEME_HTTP,
  GSK_URL_SCHEME_HTTPS,
  GSK_URL_SCHEME_FTP,
  GSK_URL_SCHEME_OTHER = 0x10000,
} GskUrlScheme;

struct _GskUrlClass
{
  GtkObjectClass                parent_class;
};

struct _GskUrl
{
  GtkObject                     parent_instance;

  GskUrlScheme			scheme;
  char                         *scheme_name;

  /* The place where the resource can be obtained.
   * (The RFC refers to this as an `authority', which is more general, sect 3.2)
   */
  char                         *host;

  /* XXX: refer to RFC again for if this is general... */
  char                         *password;

  /* For server-based urls, the port where the resource may be obtained,
   * or 0, indicating that none was specified. (Section 3.2.2)
   */
  int				port;

  /* The username or NULL if none was given. */
  char			       *user_name;

  /* The scheme-specific data (we separate the section and query pieces though)
   * RFC 2396, Section 3.3.
   */
  char                         *path;

  /* The query portion of the URL. */
  char                         *query;

  /* The portion after a `#' character. (Section 4.1) */
  char			       *fragment;
};

GtkType gsk_url_get_type();
#define GSK_TYPE_URL			(gsk_url_get_type ())
#define GSK_URL(obj)              (GTK_CHECK_CAST ((obj), GSK_TYPE_URL, GskUrl))
#define GSK_URL_CLASS(klass)      (GTK_CHECK_CLASS_CAST ((klass), GSK_TYPE_URL, GskUrlClass))
#define GSK_URL_GET_CLASS(obj)    (GSK_URL_CLASS(GTK_OBJECT(obj)->klass))
#define GSK_IS_URL(obj)           (GTK_CHECK_TYPE ((obj), GSK_TYPE_URL))
#define GSK_IS_URL_CLASS(klass)   (GTK_CHECK_CLASS_TYPE ((klass), GSK_TYPE_URL))

GskUrl         *gsk_url_new                 (const char      *spec);
GskUrl         *gsk_url_new_in_context      (const char      *spec,
                                             GskUrlScheme     context);
GskUrl         *gsk_url_new_from_parts      (GskUrlScheme     scheme,
					     const char      *host,
					     int              port,
					     const char      *user_name,
					     const char      *password,
					     const char      *path,
					     const char      *query,
					     const char      *fragment);
              
char           *gsk_url_get_relative_path   (GskUrl          *url);

#if 0
/* Allocating new url schemes. */
/* XXX: this would be helpful. */
GskUrlScheme   *gsk_url_scheme_get_unique   (const char      *url_scheme,
					     GskUrlParser     parse_func,
					     GskUrlPrinter    print_func);
#endif

/* Url-encoding helper functions. */
char           *gsk_url_decode              (const char      *encoded);
char           *gsk_url_encode              (const char      *decoded);

#ifndef GSK_DISABLE_COMPAT
#define gsk_url_ref(url)   gtk_object_ref(GTK_OBJECT(url))
#define gsk_url_unref(url) gtk_object_unref(GTK_OBJECT(url))
#endif

#endif
