/*************************************************
* BigInt Header File                             *
* (C) 1999-2002 The Botan Project                *
*************************************************/

#ifndef BOTAN_BIGINT_H__
#define BOTAN_BIGINT_H__

#include <botan/base.h>
#include <botan/mp_types.h>

namespace Botan {

enum Encoding { Octal = 8, Decimal = 10, Hexadecimal = 16, Binary = 256 };
enum Sign { Negative = 0, Positive = 1 };
enum NumberType { Random, Prime, SafePrime };

/*************************************************
* BigInt                                         *
*************************************************/
class BigInt
   {
   public:
      struct DivideByZero : public Exception
         { DivideByZero() : Exception("A BigInt tried to divide by zero") {} };

      BigInt& operator+=(const BigInt&);
      BigInt& operator-=(const BigInt&);

      BigInt& operator*=(const BigInt&);
      BigInt& operator/=(const BigInt&);
      BigInt& operator%=(const BigInt&);
      word    operator%=(word);
      BigInt& operator<<=(u32bit);
      BigInt& operator>>=(u32bit);

      BigInt& operator++();
      BigInt& operator--();
      BigInt  operator++(int) { BigInt tmp = *this; ++*this; return tmp; }
      BigInt  operator--(int) { BigInt tmp = *this; --*this; return tmp; }

      BigInt operator-() const;
      bool operator !() const { return (!is_nonzero()); }

      void add(word);
      void sub(word);

      s32bit cmp(const BigInt&, bool = true) const;
      bool is_even() const { return (get_bit(0) == 0); }
      bool is_odd()  const { return (get_bit(0) == 1); }
      bool is_nonzero() const { return (!is_zero()); }
      bool is_zero() const;

      void set_bit(u32bit);
      void clear_bit(u32bit);
      bool get_bit(u32bit) const;
      byte byte_at(u32bit) const;
      word word_at(u32bit) const;

      bool is_negative() const { return (sign() == Negative); }
      bool is_positive() const { return (sign() == Positive); }
      Sign sign() const { return (signedness); }
      Sign reverse_sign() const;
      void flip_sign();
      void set_sign(Sign);
      BigInt abs() const;

      u32bit size() const { return reg.size(); }
      u32bit sig_words() const;
      u32bit bytes() const;
      u32bit bits() const;

      const word* data() const { return reg.ptr(); }
      SecureVector<word>& get_reg() { return reg; }
      void grow_reg(u32bit n) { grow_to(n + size()); }
      void shrink() { reg.resize(sig_words()); }

      word& operator[](u32bit index) { return reg[index]; }
      word operator[](u32bit index) const { return reg[index]; }

      void randomize(u32bit = 0);

      SecureVector<byte> binary_encode() const;
      void binary_encode(byte[]) const;
      void binary_decode(const SecureVector<byte>&);
      void binary_decode(const byte[], u32bit);

      static const BigInt& zero();
      static const BigInt& one();
      static const BigInt& ten();

      void swap(BigInt&);

      friend void modifying_divide(BigInt&, BigInt&, BigInt&);

      BigInt(u64bit = 0);
      BigInt(const BigInt&);
      BigInt(const std::string&);
      BigInt(NumberType, u32bit);
      BigInt(const byte[], u32bit, Encoding = Binary);

      BigInt(const SecureVector<byte>&, Encoding = Binary);
      BigInt(const word*, u32bit, Sign);
      BigInt(Sign, u32bit);
      ~BigInt();
   private:
      void grow_to(u32bit n) const { reg.grow_to(n); }
      Sign signedness;
      SecureVector<word> reg;
   };

/*************************************************
* Arithmetic Operations                          *
*************************************************/
BigInt operator+(const BigInt&, const BigInt&);
BigInt operator-(const BigInt&, const BigInt&);
BigInt operator*(const BigInt&, const BigInt&);
BigInt operator/(const BigInt&, const BigInt&);
BigInt operator%(const BigInt&, const BigInt&);
word   operator%(const BigInt&, word);
BigInt operator<<(const BigInt&, u32bit);
BigInt operator>>(const BigInt&, u32bit);

/*************************************************
* Comparison Functions                           *
*************************************************/
inline bool operator==(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) == 0); }
inline bool operator!=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) != 0); }
inline bool operator<=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) <= 0); }
inline bool operator>=(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) >= 0); }
inline bool operator<(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) < 0); }
inline bool operator>(const BigInt& a, const BigInt& b)
   { return (a.cmp(b) > 0); }

/*************************************************
* Miscellaneous Functions                        *
*************************************************/
u32bit encoded_size(const BigInt&, Encoding = Binary);
SecureVector<byte> encode(const BigInt&, Encoding = Binary);
void encode(byte[], const BigInt&, Encoding = Binary);
BigInt decode(const byte[], u32bit, Encoding = Binary);
BigInt decode(const SecureVector<byte>&, Encoding = Binary);

std::ostream& operator<<(std::ostream&, const BigInt&);
std::istream& operator>>(std::istream&, BigInt&);

}

namespace std {

inline void swap(Botan::BigInt& a, Botan::BigInt& b) { a.swap(b); }

}

#endif
