/*************************************************
* Exceptions Header File                         *
* (C) 1999-2002 The Botan Project                *
*************************************************/

#ifndef BOTAN_EXCEPTION_H__
#define BOTAN_EXCEPTION_H__

#include <botan/config.h>
#include <exception>
#include <string>

namespace Botan {

/*************************************************
* Exception Base Class                           *
*************************************************/
class Exception : public std::exception
   {
   public:
      const char* what() const throw() { return msg.c_str(); }
      Exception(const std::string& m = "Unknown Error") { set_msg(m); }
      virtual ~Exception() throw() {}
   protected:
      void set_msg(const std::string& m) { msg = "Botan: " + m; }
   private:
      std::string msg;
   };

/*************************************************
* Invalid_Argument Exception                     *
*************************************************/
struct Invalid_Argument : public Exception
   {
   Invalid_Argument(const std::string& str = "") : Exception(str) {}
   };

/*************************************************
* Invalid_Key_Length Exception                   *
*************************************************/
struct Invalid_Key_Length : public Invalid_Argument
   {
   Invalid_Key_Length(const std::string&, u32bit);
   };

/*************************************************
* Invalid_Block_Size Exception                   *
*************************************************/
struct Invalid_Block_Size : public Invalid_Argument
   {
   Invalid_Block_Size(const std::string&, const std::string&);
   };

/*************************************************
* Invalid_IV_Length Exception                    *
*************************************************/
struct Invalid_IV_Length : public Invalid_Argument
   {
   Invalid_IV_Length(const std::string&, u32bit);
   };

/*************************************************
* Invalid_Message_Number Exception               *
*************************************************/
struct Invalid_Message_Number : public Invalid_Argument
   {
   Invalid_Message_Number(const std::string&, u32bit);
   };

/*************************************************
* Algorithm_Not_Found Exception                  *
*************************************************/
struct Algorithm_Not_Found : public Exception
   {
   Algorithm_Not_Found(const std::string&);
   };

/*************************************************
* Input_Too_Large Exception                      *
*************************************************/
struct Input_Too_Large : public Exception
   {
   Input_Too_Large(const std::string&, u32bit, u32bit);
   };

/*************************************************
* Format_Error Exception                         *
*************************************************/
struct Format_Error : public Exception
   {
   Format_Error(const std::string& str = "") : Exception(str) {}
   };

/*************************************************
* Invalid_Algorithm_Name Exception               *
*************************************************/
struct Invalid_Algorithm_Name : public Format_Error
   {
   Invalid_Algorithm_Name(const std::string&);
   };

/*************************************************
* Decoding_Error Exception                       *
*************************************************/
struct Decoding_Error : public Format_Error
   {
   Decoding_Error(const std::string& name)
      : Format_Error("A decoding error occcured in " + name) {}
   };

/*************************************************
* Stream_IO_Error Exception                      *
*************************************************/
struct Stream_IO_Error : public Exception
   {
   Stream_IO_Error(const std::string& error)
      : Exception("I/O error: " + error) {}
   };

/*************************************************
* Internal_Error Exception                       *
*************************************************/
struct Internal_Error : public Exception
   {
   Internal_Error(const std::string& error) :
      Exception("Internal error: " + error) {}
   };

}

#endif
