/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_address module

  Functions herein allow address creation and conversion.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_address

  Les fonctions presentes dans ce module permettent d'assurer la
  creation et la conversion d'adresses.
*/

/*-------------------------------------------------------------*/
/*EN                       ADDRESS TYPES                         */
/*FR                 REPRESENTATION D'ADRESSES                   */
/*-------------------------------------------------------------*/
/*en to store a computer name as a string like "machine.entr.fr" */
/*fr type permettant de representer un nom de machine sous la forme 
   d'une chaine de caracteres comme "machine.entr.fr" */
#define LCRZO_HS_MAXBYTES 240
typedef char lcrzo_hs[LCRZO_HS_MAXBYTES+1]; /* +1 for/pour '\0'*/
typedef const lcrzo_hs lcrzo_consths;

/*en to store an IP address as a string like "193.52.95.43" */
/*fr type permettant de representer une adresse ip sous la forme d'une 
   chaine de caracteres comme "193.52.95.43" */
#define LCRZO_IPS_MAXBYTES 15
typedef char lcrzo_ips[LCRZO_IPS_MAXBYTES+1]; /*+1 for/pour '\0'*/
typedef const lcrzo_ips lcrzo_constips;

/*en to store an IP address as an array like [ii][ii][ii][ii] */
/*fr type permettant de representer une adresse ip sous forme de 
   tableau de 4 entiers comme [ii][ii][ii][ii] */
#define LCRZO_IPA_MAXBYTES 4
typedef lcrzo_uint8 lcrzo_ipa[LCRZO_IPA_MAXBYTES];
typedef const lcrzo_ipa lcrzo_constipa;

/*en to store an IP address as a 32 bits integer like 0x3456329D 
  (it is in host byte order (and not in network byte order)) */
/*fr type permettant de representer une adresse ip sous la forme d'un 
   entier de 32 bits comme 0x3456329D (l'ordre des octets est host
   et non pas network) */
typedef lcrzo_uint32 lcrzo_ipl;

/*en to store an Ethernet address as a string like "8:0:20:74:dc:2e" */
/*fr type permettant de representer une adresse ethernet sous la forme
   d'une chaine de caracteres comme "8:0:20:74:dc:2e" */
#define LCRZO_ETHS_MAXBYTES 17
typedef char lcrzo_eths[LCRZO_ETHS_MAXBYTES+1]; /*+1for/pour'\0'*/
typedef const lcrzo_eths lcrzo_consteths;

/*en to store an Ethernet address as an array like
   [ii][ii][ii][ii][ii][ii] */
/*fr type permettant de representer une adresse ethernet sous forme de
   tableau de 6 entiers comme [ii][ii][ii][ii][ii][ii] */
#define LCRZO_ETHA_MAXBYTES 6
typedef lcrzo_uint8 lcrzo_etha[LCRZO_ETHA_MAXBYTES];
typedef const lcrzo_etha lcrzo_constetha;


/*-------------------------------------------------------------*/
/*en this define permits to verify that there is no buffer overflow
   before setting a lcrzo_hs */
/*fr ce define permet de verifier qu'il n'y ait pas
   de buffer overflow dans une chaine lcrzo_hs*/
#define lcrzo_hs_verifbof(hs) if(strlen(hs)>LCRZO_HS_MAXBYTES)return(LCRZO_ERR_PAOVERFLOW)

/*en Name : lcrzo_hs_init
   Description :
     Initialize an lcrzo_hs address
   Input parameter(s) :
     inhs : string containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     ouths : lcrzo_hs set with inhs
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in ouths
*/
/*fr Nom : lcrzo_hs_init
   Description :
     Initialise une adresse sous la forme lcrzo_hs
   Parametre(s) d'entree :
     inhs : chaine contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ouths : lcrzo_hs affecte par inhs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans ouths 
*/
int lcrzo_hs_init(lcrzo_conststring instring,
		  lcrzo_hs ouths);

/*en Name : lcrzo_hs_init_hs
   Description :
     Initialize an lcrzo_hs address
   Input parameter(s) :
     inhs : lcrzo_hs containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     ouths : lcrzo_hs set with inhs
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in ouths
*/
/*fr Nom : lcrzo_hs_init_hs
   Description :
     Initialise une adresse sous la forme lcrzo_hs
   Parametre(s) d'entree :
     inhs : lcrzo_hs contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ouths : lcrzo_hs affecte par inhs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans ouths 
*/
int lcrzo_hs_init_hs(lcrzo_consths inhs,
		     lcrzo_hs ouths);

/*en Name : lcrzo_hs_initdefault
   Description :
     Initialize a default lcrzo_hs address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with "" (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hs_initdefault
   Description :
     Initialise une adresse lcrzo_hs avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte par "" (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hs_initdefault(lcrzo_hs hs);

/*en Name : lcrzo_hs_init_ips
   Description :
     Convert a lcrzo_ips to a lcrzo_hs
   Input parameter(s) :
     ips : lcrzo_ips to convert
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with ips value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_hs_init_ips
   Description :
     Convertit un lcrzo_ips en lcrzo_hs
   Parametre(s) d'entree :
     ips : lcrzo_ips a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte avec la valeur de ips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_hs_init_ips(lcrzo_constips ips, lcrzo_hs hs);

/*en Name : lcrzo_hs_init_ipa
   Description :
     Convert a lcrzo_ipa to a lcrzo_hs
   Input parameter(s) :
     ipa : lcrzo_ipa to convert
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with ipa value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_hs_init_ipa
   Description :
     Convertit un lcrzo_ipa en lcrzo_hs
   Parametre(s) d'entree :
     ipa : lcrzo_ipa a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte avec la valeur de ipa 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_hs_init_ipa(lcrzo_constipa ipa, lcrzo_hs hs);

/*en Name : lcrzo_hs_init_ipl
   Description :
     Convert a lcrzo_ipl to a lcrzo_hs
   Input parameter(s) :
     ipl : lcrzo_ipl to convert
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with ipl value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_hs_init_ipl
   Description :
     Convertit un lcrzo_ipl en lcrzo_hs
   Parametre(s) d'entree :
     ipl : lcrzo_ipl a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte avec la valeur de ipl 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_hs_init_ipl(lcrzo_ipl ipl, lcrzo_hs hs);

/*en Name : lcrzo_hs_init_eths
   Description :
     Convert a lcrzo_eths to a lcrzo_hs
   Input parameter(s) :
     eths : lcrzo_eths to convert
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with eths value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_hs_init_eths
   Description :
     Convertit un lcrzo_eths en lcrzo_hs
   Parametre(s) d'entree :
     eths : lcrzo_eths a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte avec la valeur de eths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_hs_init_eths(lcrzo_consteths eths, lcrzo_hs hs);

/*en Name : lcrzo_hs_init_etha
   Description :
     Convert a lcrzo_etha to a lcrzo_hs
   Input parameter(s) :
     etha : lcrzo_etha to convert
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with etha value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_hs_init_etha
   Description :
     Convertit un lcrzo_etha en lcrzo_hs
   Parametre(s) d'entree :
     etha : lcrzo_etha a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte avec la valeur de etha 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_hs_init_etha(lcrzo_constetha etha, lcrzo_hs hs);

/*en Name : lcrzo_hs_equal
   Description :
     Verify if two lcrzo_hs are equal
   Input parameter(s) :
     hs1 : lcrzo_hs to compare with hs2
     hs2 : lcrzo_hs to compare with hs1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if hs1 and hs2 are different
     1 : if hs1 and hs2 are equal
*/
/*fr Nom : lcrzo_hs_equal
   Description :
     Verifie l'egalite de deux lcrzo_hs
   Parametre(s) d'entree :
     hs1 : lcrzo_hs a comparer avec hs2
     hs2 : lcrzo_hs a comparer avec hs1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si hs1 et hs2 sont differents
     1 : si hs1 et hs2 sont egaux
*/
lcrzo_bool lcrzo_hs_equal(lcrzo_consths hs1, lcrzo_consths hs2); 

/*en Name : lcrzo_hs_cmp
   Description :
     Compare two lcrzo_hs
   Input parameter(s) :
     hs1 : lcrzo_hs to compare with hs2
     hs2 : lcrzo_hs to compare with hs1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : hs1<hs2
      0 : if hs1 and hs2 are equal
     +1 : hs1>hs2
*/
/*fr Nom : lcrzo_hs_cmp
   Description :
     Compare deux lcrzo_hs
   Parametre(s) d'entree :
     hs1 : lcrzo_hs a comparer avec hs2
     hs2 : lcrzo_hs a comparer avec hs1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : hs1<hs2
      0 : si hs1 et hs2 sont egaux
     +1 : hs1>hs2
*/
lcrzo_int8 lcrzo_hs_cmp(lcrzo_consths hs1, lcrzo_consths hs2); 

/*en Name : lcrzo_hs_init_stdin
   Description :
     Initialize an lcrzo_hs with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulths : default lcrzo_hs to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     hs : lcrzo_hs set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hs_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulths : lcrzo_hs par defaut a utiliser si l'utilisateur
                   n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hs : lcrzo_hs affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hs_init_stdin(lcrzo_conststring message,
			lcrzo_consths defaulths,
			lcrzo_hs hs);

/*-------------------------------------------------------------*/
/*en this define permits to verify that there is no buffer overflow
   before setting a lcrzo_ips */
/*fr ce define permet de verifier qu'il n'y ait pas
   de buffer overflow dans une chaine lcrzo_ips*/
#define lcrzo_ips_verifbof(ips) if(strlen(ips)>LCRZO_IPS_MAXBYTES)return(LCRZO_ERR_PAOVERFLOW)

/*en Name : lcrzo_ips_init
   Description :
     Initialize an lcrzo_ips address
   Input parameter(s) :
     inips : string containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     outips : lcrzo_ips set with inips
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in outips
*/
/*fr Nom : lcrzo_ips_init
   Description :
     Initialise une adresse sous la forme lcrzo_ips
   Parametre(s) d'entree :
     inips : chaine contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outips : lcrzo_ips affecte par inips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans outips 
*/
int lcrzo_ips_init(lcrzo_conststring inips,
		   lcrzo_ips outips);

/*en Name : lcrzo_ips_init_ips
   Description :
     Initialize an lcrzo_ips address
   Input parameter(s) :
     inips : lcrzo_ips containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     outips : lcrzo_ips set with inips
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in outips
*/
/*fr Nom : lcrzo_ips_init_ips
   Description :
     Initialise une adresse sous la forme lcrzo_ips
   Parametre(s) d'entree :
     inips : lcrzo_ips contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outips : lcrzo_ips affecte par inips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans outips 
*/
int lcrzo_ips_init_ips(lcrzo_constips inips,
		       lcrzo_ips outips);

/*en Name : lcrzo_ips_initdefault
   Description :
     Initialize a default lcrzo_ips address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with "0.0.0.0" (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ips_initdefault
   Description :
     Initialise une adresse lcrzo_ips avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte par "0.0.0.0" (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ips_initdefault(lcrzo_ips ips);

/*en Name : lcrzo_ips_init_hs
   Description :
     Convert a lcrzo_hs to a lcrzo_ips
   Input parameter(s) :
     hs : lcrzo_hs to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with hs value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ips_init_hs
   Description :
     Convertit un lcrzo_hs en lcrzo_ips
   Parametre(s) d'entree :
     hs : lcrzo_hs a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte avec la valeur de hs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ips_init_hs(lcrzo_consths hs, lcrzo_ips ips);

/*en Name : lcrzo_ips_init_ipa
   Description :
     Convert a lcrzo_ipa to a lcrzo_ips
   Input parameter(s) :
     ipa : lcrzo_ipa to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with ipa value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ips_init_ipa
   Description :
     Convertit un lcrzo_ipa en lcrzo_ips
   Parametre(s) d'entree :
     ipa : lcrzo_ipa a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte avec la valeur de ipa 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ips_init_ipa(lcrzo_constipa ipa, lcrzo_ips ips);

/*en Name : lcrzo_ips_init_ipl
   Description :
     Convert a lcrzo_ipl to a lcrzo_ips
   Input parameter(s) :
     ipl : lcrzo_ipl to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with ipl value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ips_init_ipl
   Description :
     Convertit un lcrzo_ipl en lcrzo_ips
   Parametre(s) d'entree :
     ipl : lcrzo_ipl a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte avec la valeur de ipl 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ips_init_ipl(lcrzo_ipl ipl, lcrzo_ips ips); 

/*en Name : lcrzo_ips_init_eths
   Description :
     Convert a lcrzo_eths to a lcrzo_ips
   Input parameter(s) :
     eths : lcrzo_eths to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with eths value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ips_init_eths
   Description :
     Convertit un lcrzo_eths en lcrzo_ips
   Parametre(s) d'entree :
     eths : lcrzo_eths a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte avec la valeur de eths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ips_init_eths(lcrzo_consteths eths, lcrzo_ips ips);

/*en Name : lcrzo_ips_init_etha
   Description :
     Convert a lcrzo_etha to a lcrzo_ips
   Input parameter(s) :
     etha : lcrzo_etha to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with etha value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ips_init_etha
   Description :
     Convertit un lcrzo_etha en lcrzo_ips
   Parametre(s) d'entree :
     etha : lcrzo_etha a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte avec la valeur de etha 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ips_init_etha(lcrzo_constetha etha, lcrzo_ips ips);

/*en Name : lcrzo_ips_equal
   Description :
     Verify if two lcrzo_ips are equal
   Input parameter(s) :
     ips1 : lcrzo_ips to compare with ips2
     ips2 : lcrzo_ips to compare with ips1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if ips1 and ips2 are different
     1 : if ips1 and ips2 are equal
*/
/*fr Nom : lcrzo_ips_equal
   Description :
     Verifie l'egalite de deux lcrzo_ips
   Parametre(s) d'entree :
     ips1 : lcrzo_ips a comparer avec ips2
     ips2 : lcrzo_ips a comparer avec ips1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si ips1 et ips2 sont differents
     1 : si ips1 et ips2 sont egaux
*/
lcrzo_bool lcrzo_ips_equal(lcrzo_constips ips1, lcrzo_constips ips2);

/*en Name : lcrzo_ips_cmp
   Description :
     Compare two lcrzo_ips
   Input parameter(s) :
     ips1 : lcrzo_ips to compare with ips2
     ips2 : lcrzo_ips to compare with ips1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : ips1<ips2
      0 : if ips1 and ips2 are equal
     +1 : ips1>ips2
*/
/*fr Nom : lcrzo_ips_cmp
   Description :
     Compare deux lcrzo_ips
   Parametre(s) d'entree :
     ips1 : lcrzo_ips a comparer avec ips2
     ips2 : lcrzo_ips a comparer avec ips1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : ips1<ips2
      0 : si ips1 et ips2 sont egaux
     +1 : ips1>ips2
*/
lcrzo_int8 lcrzo_ips_cmp(lcrzo_constips ips1, lcrzo_constips ips2); 

/*en Name : lcrzo_ips_init_stdin
   Description :
     Initialize an lcrzo_ips with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulths : default lcrzo_hs to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     ips : lcrzo_ips set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ips_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulths : lcrzo_hs par defaut a utiliser si l'utilisateur
                 n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ips : lcrzo_ips affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ips_init_stdin(lcrzo_conststring message,
			 lcrzo_consths defaulths, 
			 lcrzo_ips ips);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipa_init
   Description :
     Initialize an lcrzo_ipa address
   Input parameter(s) :
     a,b,c,d : bytes of the IP address
   Input/output parameter(s) :
   Output parameter(s) :
     outipa : lcrzo_ipa set with a.b.c.d
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_init
   Description :
     Initialise une adresse sous la forme lcrzo_ipa
   Parametre(s) d'entree :
     a,b,c,d : octets de l'adresse IP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outipa : lcrzo_ipa affecte par a.b.c.d
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_init(lcrzo_uint8 a,
		   lcrzo_uint8 b,
		   lcrzo_uint8 c,
		   lcrzo_uint8 d, 
		   lcrzo_ipa outipa);

/*en Name : lcrzo_ipa_init_ipa
   Description :
     Initialize an lcrzo_ipa address
   Input parameter(s) :
     inipa : input lcrzo_ipa
   Input/output parameter(s) :
   Output parameter(s) :
     outipa : output lcrzo_ipa
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_init_ipa
   Description :
     Initialise une adresse sous la forme lcrzo_ipa
   Parametre(s) d'entree :
     inipa : lcr_ipa d'entree
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outipa : lcrzo_ipa affecte par inipa
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_init_ipa(lcrzo_constipa inipa, 
		       lcrzo_ipa outipa);

/*en Name : lcrzo_ipa_initdefault
   Description :
     Initialize a default lcrzo_ipa address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with [0][0][0][0] (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_initdefault
   Description :
     Initialise une adresse lcrzo_ipa avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outipa : lcrzo_ipa affecte par [0][0][0][0] (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_initdefault(lcrzo_ipa ipa);

/*en Name : lcrzo_ipa_init_hs
   Description :
     Convert a lcrzo_hs to a lcrzo_ipa
   Input parameter(s) :
     hs : lcrzo_hs to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with hs value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipa_init_hs
   Description :
     Convertit un lcrzo_hs en lcrzo_ipa
   Parametre(s) d'entree :
     hs : lcrzo_hs a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte avec la valeur de hs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipa_init_hs(lcrzo_consths hs, lcrzo_ipa ipa);

/*en Name : lcrzo_ipa_init_ips
   Description :
     Convert a lcrzo_ips to a lcrzo_ipa
   Input parameter(s) :
     ips : lcrzo_ips to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with ips value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipa_init_ips
   Description :
     Convertit un lcrzo_ips en lcrzo_ipa
   Parametre(s) d'entree :
     ips : lcrzo_ips a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte avec la valeur de ips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipa_init_ips(lcrzo_constips ips, lcrzo_ipa ipa);

/*en Name : lcrzo_ipa_init_ipl
   Description :
     Convert a lcrzo_ipl to a lcrzo_ipa
   Input parameter(s) :
     ipl : lcrzo_ipl to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with ipl value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipa_init_ipl
   Description :
     Convertit un lcrzo_ipl en lcrzo_ipa
   Parametre(s) d'entree :
     ipl : lcrzo_ipl a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte avec la valeur de ipl 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipa_init_ipl(lcrzo_ipl ipl, lcrzo_ipa ipa);

/*en Name : lcrzo_ipa_init_eths
   Description :
     Convert a lcrzo_eths to a lcrzo_ipa
   Input parameter(s) :
     eths : lcrzo_eths to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with eths value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipa_init_eths
   Description :
     Convertit un lcrzo_eths en lcrzo_ipa
   Parametre(s) d'entree :
     eths : lcrzo_eths a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte avec la valeur de eths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipa_init_eths(lcrzo_consteths eths, lcrzo_ipa ipa);

/*en Name : lcrzo_ipa_init_etha
   Description :
     Convert a lcrzo_etha to a lcrzo_ipa
   Input parameter(s) :
     etha : lcrzo_etha to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with etha value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipa_init_etha
   Description :
     Convertit un lcrzo_etha en lcrzo_ipa
   Parametre(s) d'entree :
     etha : lcrzo_etha a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte avec la valeur de etha 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipa_init_etha(lcrzo_constetha etha, lcrzo_ipa ipa); 

/*en Name : lcrzo_ipa_equal
   Description :
     Verify if two lcrzo_ipa are equal
   Input parameter(s) :
     ipa1 : lcrzo_ipa to compare with ipa2
     ipa2 : lcrzo_ipa to compare with ipa1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if ipa1 and ipa2 are different
     1 : if ipa1 and ipa2 are equal
*/
/*fr Nom : lcrzo_ipa_equal
   Description :
     Verifie l'egalite de deux lcrzo_ipa
   Parametre(s) d'entree :
     ipa1 : lcrzo_ipa a comparer avec ipa2
     ipa2 : lcrzo_ipa a comparer avec ipa1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si ipa1 et ipa2 sont differents
     1 : si ipa1 et ipa2 sont egaux
*/
lcrzo_bool lcrzo_ipa_equal(lcrzo_constipa ipa1, lcrzo_constipa ipa2); 

/*en Name : lcrzo_ipa_cmp
   Description :
     Compare two lcrzo_ipa
   Input parameter(s) :
     ipa1 : lcrzo_ipa to compare with ipa2
     ipa2 : lcrzo_ipa to compare with ipa1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : ipa1<ipa2
      0 : if ipa1 and ipa2 are equal
     +1 : ipa1>ipa2
*/
/*fr Nom : lcrzo_ipa_cmp
   Description :
     Compare deux lcrzo_ipa
   Parametre(s) d'entree :
     ipa1 : lcrzo_ipa a comparer avec ipa2
     ipa2 : lcrzo_ipa a comparer avec ipa1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : ipa1<ipa2
      0 : si ipa1 et ipa2 sont egaux
     +1 : ipa1>ipa2
*/
lcrzo_int8 lcrzo_ipa_cmp(lcrzo_constipa ipa1, lcrzo_constipa ipa2); 

/*en Name : lcrzo_ipa_init_stdin
   Description :
     Initialize an lcrzo_ipa with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulths : default lcrzo_hs to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     ipa : lcrzo_ipa set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipa_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulths : lcrzo_hs par defaut a utiliser si l'utilisateur
                 n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ipa : lcrzo_ipa affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipa_init_stdin(lcrzo_conststring message,
		    lcrzo_consths defaulths, 
		    lcrzo_ipa ipa);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_ipl_init
   Description :
     Initialize an lcrzo_ipl address
   Input parameter(s) :
     a,b,c,d : bytes of the IP address
   Input/output parameter(s) :
   Output parameter(s) :
     *poutipl : lcrzo_ipl set with a.b.c.d
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_init
   Description :
     Initialise une adresse sous la forme lcrzo_ipl
   Parametre(s) d'entree :
     a,b,c,d : octets de l'adresse IP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *poutipl : lcrzo_ipl affecte par a.b.c.d
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_init(lcrzo_uint8 a, 
		   lcrzo_uint8 b,
		   lcrzo_uint8 c,
		   lcrzo_uint8 d, 
		   lcrzo_ipl *poutipl);

/*en Name : lcrzo_ipl_init_ipl
   Description :
     Initialize an lcrzo_ipl address
   Input parameter(s) :
     inipl : input ipl
   Input/output parameter(s) :
   Output parameter(s) :
     *poutipl : lcrzo_ipl set with inipl
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_init_ipl
   Description :
     Initialise une adresse sous la forme lcrzo_ipl
   Parametre(s) d'entree :
     inipl : lcrzo_ipl d'entree
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *poutipl : lcrzo_ipl affecte par inipl
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_init_ipl(lcrzo_ipl inipl, 
		       lcrzo_ipl *poutipl);

/*en Name : lcrzo_ipl_initdefault
   Description :
     Initialize a default lcrzo_ipl address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with 0x00000000 (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_initdefault
   Description :
     Initialise une adresse lcrzo_ipl avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte par 0x00000000 (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_initdefault(lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_init_hs
   Description :
     Convert a lcrzo_hs to a lcrzo_ipl
   Input parameter(s) :
     hs : lcrzo_hs to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with hs value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipl_init_hs
   Description :
     Convertit un lcrzo_hs en lcrzo_ipl
   Parametre(s) d'entree :
     hs : lcrzo_hs a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte avec la valeur de hs 
   Valeurs de retour normales :
     LCRZO_EOK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipl_init_hs(lcrzo_consths hs, lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_init_ips
   Description :
     Convert a lcrzo_ips to a lcrzo_ipl
   Input parameter(s) :
     ips : lcrzo_ips to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with ips value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipl_init_ips
   Description :
     Convertit un lcrzo_ips en lcrzo_ipl
   Parametre(s) d'entree :
     ips : lcrzo_ips a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte avec la valeur de ips 
   Valeurs de retour normales :
     LCRZO_EOK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipl_init_ips(lcrzo_constips ips, lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_init_ipa
   Description :
     Convert a lcrzo_ipa to a lcrzo_ipl
   Input parameter(s) :
     ipa : lcrzo_ipa to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with ipa value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipl_init_ipa
   Description :
     Convertit un lcrzo_ipa en lcrzo_ipl
   Parametre(s) d'entree :
     ipa : lcrzo_ipa a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte avec la valeur de ipa 
   Valeurs de retour normales :
     LCRZO_EOK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipl_init_ipa(lcrzo_constipa ipa, lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_init_eths
   Description :
     Convert a lcrzo_eths to a lcrzo_ipl
   Input parameter(s) :
     eths : lcrzo_eths to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with eths value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipl_init_eths
   Description :
     Convertit un lcrzo_eths en lcrzo_ipl
   Parametre(s) d'entree :
     eths : lcrzo_eths a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte avec la valeur de eths 
   Valeurs de retour normales :
     LCRZO_EOK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipl_init_eths(lcrzo_consteths eths, lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_init_etha
   Description :
     Convert a lcrzo_etha to a lcrzo_ipl
   Input parameter(s) :
     etha : lcrzo_etha to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with etha value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_ipl_init_etha
   Description :
     Convertit un lcrzo_etha en lcrzo_ipl
   Parametre(s) d'entree :
     etha : lcrzo_etha a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte avec la valeur de etha 
   Valeurs de retour normales :
     LCRZO_EOK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_ipl_init_etha(lcrzo_constetha etha, lcrzo_ipl *pipl);

/*en Name : lcrzo_ipl_equal
   Description :
     Verify if two lcrzo_ipl are equal
   Input parameter(s) :
     ipl1 : lcrzo_ipl to compare with ipl2
     ipl2 : lcrzo_ipl to compare with ipl1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if ipl1 and ipl2 are different
     1 : if ipl1 and ipl2 are equal
*/
/*fr Nom : lcrzo_ipl_equal
   Description :
     Verifie l'egalite de deux lcrzo_ipl
   Parametre(s) d'entree :
     ipl1 : lcrzo_ipl a comparer avec ipl2
     ipl2 : lcrzo_ipl a comparer avec ipl1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si ipl1 et ipl2 sont differents
     1 : si ipl1 et ipl2 sont egaux
*/
lcrzo_bool lcrzo_ipl_equal(lcrzo_ipl ipl1, lcrzo_ipl ipl2); 

/*en Name : lcrzo_ipl_cmp
   Description :
     Compare two lcrzo_ipl
   Input parameter(s) :
     ipl1 : lcrzo_ipl to compare with ipl2
     ipl2 : lcrzo_ipl to compare with ipl1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : ipl1<ipl2
      0 : if ipl1 and ipl2 are equal
     +1 : ipl1>ipl2
*/
/*fr Nom : lcrzo_ipl_cmp
   Description :
     Compare deux lcrzo_ipl
   Parametre(s) d'entree :
     ipl1 : lcrzo_ipl a comparer avec ipl2
     ipl2 : lcrzo_ipl a comparer avec ipl1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : ipl1<ipl2
      0 : si ipl1 et ipl2 sont egaux
     +1 : ipl1>ipl2
*/
lcrzo_int8 lcrzo_ipl_cmp(lcrzo_ipl ipl1, lcrzo_ipl ipl2); 

/*en Name : lcrzo_ipl_init_stdin
   Description :
     Initialize an lcrzo_ipl with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulths : default lcrzo_hs to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     *pipl : lcrzo_ipl set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulths : lcrzo_hs par defaut a utiliser si l'utilisateur
                 n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipl : lcrzo_ipl affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipl_init_stdin(lcrzo_conststring message,
		    lcrzo_consths defaulths, 
		    lcrzo_ipl *pipl);

/*-------------------------------------------------------------*/
/*en this define permits to verify that there is no buffer overflow
   before setting a lcrzo_eths */
/*fr ce define permet de verifier qu'il n'y ait pas
   de buffer overflow dans une chaine lcrzo_eths*/
#define lcrzo_eths_verifbof(eths) if(strlen(eths)>LCRZO_ETHS_MAXBYTES)return(LCRZO_ERR_PAOVERFLOW)

/*en Name : lcrzo_eths_init
   Description :
     Initialize an lcrzo_eths address
   Input parameter(s) :
     ineths : string containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     outeths : lcrzo_eths set with ineths
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in outeths
*/
/*fr Nom : lcrzo_eths_init
   Description :
     Initialise une adresse sous la forme lcrzo_eths
   Parametre(s) d'entree :
     ineths : chaine contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outeths : lcrzo_eths affecte par ineths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans outeths 
*/
int lcrzo_eths_init(lcrzo_conststring ineths,
		    lcrzo_eths outeths);

/*en Name : lcrzo_eths_init_eths
   Description :
     Initialize an lcrzo_eths address
   Input parameter(s) :
     ineths : lcrzo_eths containing the host name
   Input/output parameter(s) :
   Output parameter(s) :
     outeths : lcrzo_eths set with ineths
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : all the data can't be put in outeths
*/
/*fr Nom : lcrzo_eths_init_eths
   Description :
     Initialise une adresse sous la forme lcrzo_eths
   Parametre(s) d'entree :
     ineths : lcrzo_eths contenant un nom de machine
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outeths : lcrzo_eths affecte par ineths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : toutes les donnees entrees n'ont pas
                                 ete mises dans outeths 
*/
int lcrzo_eths_init_eths(lcrzo_consteths ineths,
			 lcrzo_eths outeths);

/*en Name : lcrzo_eths_initdefault
   Description :
     Initialize a default lcrzo_eths address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with "00:00:00:00:00:00" (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_eths_initdefault
   Description :
     Initialise une adresse lcrzo_eths avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outeths : lcrzo_eths affecte par "00:00:00:00:00:00" (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_eths_initdefault(lcrzo_eths eths);

/*en Name : lcrzo_eths_init_hs
   Description :
     Convert a lcrzo_hs to a lcrzo_eths
   Input parameter(s) :
     hs : lcrzo_hs to convert
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with hs value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_eths_init_hs
   Description :
     Convertit un lcrzo_hs en lcrzo_eths
   Parametre(s) d'entree :
     hs : lcrzo_hs a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte avec la valeur de hs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_eths_init_hs(lcrzo_consths hs, lcrzo_eths eths);

/*en Name : lcrzo_eths_init_ips
   Description :
     Convert a lcrzo_ips to a lcrzo_eths
   Input parameter(s) :
     ips : lcrzo_ips to convert
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with ips value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_eths_init_ips
   Description :
     Convertit un lcrzo_ips en lcrzo_eths
   Parametre(s) d'entree :
     ips : lcrzo_ips a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte avec la valeur de ips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_eths_init_ips(lcrzo_constips ips, lcrzo_eths eths);

/*en Name : lcrzo_eths_init_ipa
   Description :
     Convert a lcrzo_ipa to a lcrzo_eths
   Input parameter(s) :
     ipa : lcrzo_ipa to convert
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with ipa value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_eths_init_ipa
   Description :
     Convertit un lcrzo_ipa en lcrzo_eths
   Parametre(s) d'entree :
     ipa : lcrzo_ipa a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte avec la valeur de ipa 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_eths_init_ipa(lcrzo_constipa ipa, lcrzo_eths eths); 

/*en Name : lcrzo_eths_init_ipl
   Description :
     Convert a lcrzo_ipl to a lcrzo_eths
   Input parameter(s) :
     ipl : lcrzo_ipl to convert
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with ipl value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_eths_init_ipl
   Description :
     Convertit un lcrzo_ipl en lcrzo_eths
   Parametre(s) d'entree :
     ipl : lcrzo_ipl a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte avec la valeur de ipl 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_eths_init_ipl(lcrzo_ipl ipl, lcrzo_eths eths); 

/*en Name : lcrzo_eths_init_etha
   Description :
     Convert a lcrzo_etha to a lcrzo_eths
   Input parameter(s) :
     etha : lcrzo_etha to convert
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with etha value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_eths_init_etha
   Description :
     Convertit un lcrzo_etha en lcrzo_eths
   Parametre(s) d'entree :
     etha : lcrzo_etha a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte avec la valeur de etha 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_eths_init_etha(lcrzo_constetha etha, lcrzo_eths eths); 

/*en Name : lcrzo_eths_equal
   Description :
     Verify if two lcrzo_eths are equal
   Input parameter(s) :
     eths1 : lcrzo_eths to compare with eths2
     eths2 : lcrzo_eths to compare with eths1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if eths1 and eths2 are different
     1 : if eths1 and eths2 are equal
*/
/*fr Nom : lcrzo_eths_equal
   Description :
     Verifie l'egalite de deux lcrzo_eths
   Parametre(s) d'entree :
     eths1 : lcrzo_eths a comparer avec eths2
     eths2 : lcrzo_eths a comparer avec eths1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si eths1 et eths2 sont differents
     1 : si eths1 et eths2 sont egaux
*/
lcrzo_bool lcrzo_eths_equal(lcrzo_consteths eths1, lcrzo_consteths eths2); 

/*en Name : lcrzo_eths_cmp
   Description :
     Compare two lcrzo_eths
   Input parameter(s) :
     eths1 : lcrzo_eths to compare with eths2
     eths2 : lcrzo_eths to compare with eths1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : eths1<eths2
      0 : if eths1 and eths2 are equal
     +1 : eths1>eths2
*/
/*fr Nom : lcrzo_eths_cmp
   Description :
     Compare deux lcrzo_eths
   Parametre(s) d'entree :
     eths1 : lcrzo_eths a comparer avec eths2
     eths2 : lcrzo_eths a comparer avec eths1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : eths1<eths2
      0 : si eths1 et eths2 sont egaux
     +1 : eths1>eths2
*/
lcrzo_int8 lcrzo_eths_cmp(lcrzo_consteths eths1, lcrzo_consteths eths2); 

/*en Name : lcrzo_eths_init_stdin
   Description :
     Initialize an lcrzo_eths with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulteths : default lcrzo_eths to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     eths : lcrzo_eths set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_eths_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulteths : lcrzo_eths par defaut a utiliser si l'utilisateur
                   n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     eths : lcrzo_eths affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_eths_init_stdin(lcrzo_conststring message,
			  lcrzo_consteths defaulteths,
			  lcrzo_eths eths);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_etha_init
   Description :
     Initialize an lcrzo_etha address
   Input parameter(s) :
     a,b,c,d,e,f : bytes of the Ethernet address
   Input/output parameter(s) :
   Output parameter(s) :
     outetha : lcrzo_etha set with a:b:c:d:e:f
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_init
   Description :
     Initialise une adresse sous la forme lcrzo_etha
   Parametre(s) d'entree :
     a,b,c,d,e,f : octets de l'adresse Ethernet
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outetha : lcrzo_etha affecte par a:b:c:d:e:f
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_init(lcrzo_uint8 a, 
		    lcrzo_uint8 b,
		    lcrzo_uint8 c,
		    lcrzo_uint8 d,
		    lcrzo_uint8 e, 
		    lcrzo_uint8 f,
		    lcrzo_etha outetha);

/*en Name : lcrzo_etha_init_etha
   Description :
     Initialize an lcrzo_etha address
   Input parameter(s) :
     inetha : input etha
   Input/output parameter(s) :
   Output parameter(s) :
     outetha : lcrzo_etha set with inetha
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_init_etha
   Description :
     Initialise une adresse sous la forme lcrzo_etha
   Parametre(s) d'entree :
     intetha : lcrzo_etha d'entree
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outetha : lcrzo_etha affecte par inetha
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_init_etha(lcrzo_constetha inetha,
			 lcrzo_etha outetha);

/*en Name : lcrzo_etha_initdefault
   Description :
     Initialize a default lcrzo_etha address
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with [0][0][0][0] (default value)
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_initdefault
   Description :
     Initialise une adresse lcrzo_etha avec la valeur par defaut
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     outetha : lcrzo_etha affecte par [0][0][0][0] (valeur par defaut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_initdefault(lcrzo_etha etha);

/*en Name : lcrzo_etha_init_hs
   Description :
     Convert a lcrzo_hs to a lcrzo_etha
   Input parameter(s) :
     hs : lcrzo_hs to convert
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with hs value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_etha_init_hs
   Description :
     Convertit un lcrzo_hs en lcrzo_etha
   Parametre(s) d'entree :
     hs : lcrzo_hs a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte avec la valeur de hs 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_etha_init_hs(lcrzo_consths hs, lcrzo_etha etha);

/*en Name : lcrzo_etha_init_ips
   Description :
     Convert a lcrzo_ips to a lcrzo_etha
   Input parameter(s) :
     ips : lcrzo_ips to convert
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with ips value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_etha_init_ips
   Description :
     Convertit un lcrzo_ips en lcrzo_etha
   Parametre(s) d'entree :
     ips : lcrzo_ips a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte avec la valeur de ips 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_etha_init_ips(lcrzo_constips ips, lcrzo_etha etha);

/*en Name : lcrzo_etha_init_ipa
   Description :
     Convert a lcrzo_ipa to a lcrzo_etha
   Input parameter(s) :
     ipa : lcrzo_ipa to convert
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with ipa value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_etha_init_ipa
   Description :
     Convertit un lcrzo_ipa en lcrzo_etha
   Parametre(s) d'entree :
     ipa : lcrzo_ipa a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte avec la valeur de ipa 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_etha_init_ipa(lcrzo_constipa ipa, lcrzo_etha etha);

/*en Name : lcrzo_etha_init_ipl
   Description :
     Convert a lcrzo_ipl to a lcrzo_etha
   Input parameter(s) :
     ipl : lcrzo_ipl to convert
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with ipl value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_etha_init_ipl
   Description :
     Convertit un lcrzo_ipl en lcrzo_etha
   Parametre(s) d'entree :
     ipl : lcrzo_ipl a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte avec la valeur de ipl 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_etha_init_ipl(lcrzo_ipl ipl, lcrzo_etha etha);

/*en Name : lcrzo_etha_init_eths
   Description :
     Convert a lcrzo_eths to a lcrzo_etha
   Input parameter(s) :
     eths : lcrzo_eths to convert
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with eths value
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : the address could not be resolved
*/
/*fr Nom : lcrzo_etha_init_eths
   Description :
     Convertit un lcrzo_eths en lcrzo_etha
   Parametre(s) d'entree :
     eths : lcrzo_eths a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte avec la valeur de eths 
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKUNRESOLVED : l'adresse n'a pas pu etre convertie
*/
int lcrzo_etha_init_eths(lcrzo_consteths eths, lcrzo_etha etha); 

/*en Name : lcrzo_etha_equal
   Description :
     Verify if two lcrzo_etha are equal
   Input parameter(s) :
     etha1 : lcrzo_etha to compare with etha2
     etha2 : lcrzo_etha to compare with etha1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     0 : if etha1 and etha2 are different
     1 : if etha1 and etha2 are equal
*/
/*fr Nom : lcrzo_etha_equal
   Description :
     Verifie l'egalite de deux lcrzo_etha
   Parametre(s) d'entree :
     etha1 : lcrzo_etha a comparer avec etha2
     etha2 : lcrzo_etha a comparer avec etha1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     0 : si etha1 et etha2 sont differents
     1 : si etha1 et etha2 sont egaux
*/
lcrzo_bool lcrzo_etha_equal(lcrzo_constetha etha1, lcrzo_constetha etha2);

/*en Name : lcrzo_etha_cmp
   Description :
     Compare two lcrzo_etha
   Input parameter(s) :
     etha1 : lcrzo_etha to compare with etha2
     etha2 : lcrzo_etha to compare with etha1
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     -1 : etha1<etha2
      0 : if etha1 and etha2 are equal
     +1 : etha1>etha2
*/
/*fr Nom : lcrzo_etha_cmp
   Description :
     Compare deux lcrzo_etha
   Parametre(s) d'entree :
     etha1 : lcrzo_etha a comparer avec etha2
     etha2 : lcrzo_etha a comparer avec etha1
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     -1 : etha1<etha2
      0 : si etha1 et etha2 sont egaux
     +1 : etha1>etha2
*/
lcrzo_int8 lcrzo_etha_cmp(lcrzo_constetha etha1, lcrzo_constetha etha2); 

/*en Name : lcrzo_etha_init_stdin
   Description :
     Initialize an lcrzo_etha with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaulteths : default lcrzo_eths to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     etha : lcrzo_etha set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaulteths : lcrzo_eths par defaut a utiliser si l'utilisateur
                   n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     etha : lcrzo_etha affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_init_stdin(lcrzo_conststring message,
		     lcrzo_consteths defaulteths,
		     lcrzo_etha etha);


/*-------------------------------------------------------------*/
/*en definition of an Ethernet range*/
/*fr definition d'une plage d'adresses Ethernet*/
#define LCRZO_ETHRANGE_MAXBYTES  35
typedef char lcrzo_ethrange[LCRZO_ETHRANGE_MAXBYTES+1]; /*+1 for/pour '\0'*/
typedef const lcrzo_ethrange lcrzo_constethrange;

/*en examples of a lcrzo_ethrange :*/
/*fr exemples de lcrzo_ethrange :*/
/*  lcrzo_ethrange              :     ethinf        ethsup
  a:b:c:d:e:f                   :   a:b:c:d:e:f   a:b:c:d:e:f
  1:2:3:4:5:6-a:b:c:d:e:f       :   1:2:3:4:5:6   a:b:c:d:e:f
  1:2:3:4:5:0/FF:FF:FF:FF:FF:0  :   1:2:3:4:5:0   1:2:3:4:5:FF
  1:2:3:4:5:0/40                :   1:2:3:4:5:0   1:2:3:4:5:FF
  1:2:3:4:5/FF:FF:FF:FF:FF:0    :   1:2:3:4:5:0   1:2:3:4:5:FF
  1:2:3:4:5/40                  :   1:2:3:4:5:0   1:2:3:4:5:FF
  1:2:3:4:5:0%FF:FF:FF:FF:FF:0  :   1:2:3:4:5:1   1:2:3:4:5:FE
  1:2:3:4:5:0%40                :   1:2:3:4:5:1   1:2:3:4:5:FE
  1:2:3:4:5%FF:FF:FF:FF:FF:0    :   1:2:3:4:5:1   1:2:3:4:5:FE
  1:2:3:4:5%40                  :   1:2:3:4:5:1   1:2:3:4:5:FE
*/
/*en note : '%' has the same meaning as '/', except that the 
  broadcast adresses are excluded from range.*/
/*fr note : '%' a la meme signification que '/', sauf que les adresses
  broadcast sont excluses de l'intervalle.*/

/*en Name : lcrzo_etha_init_ethrange
   Description :
     Conversion from a lcrzo_ethrange to two lcrzo_etha
   Input parameter(s) :
     range : lcrzo_ethrange to convert
   Input/output parameter(s) :
   Output parameter(s) :
     ethainf : inferior value of the range
     ethasup : superior value of the range
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_etha_init_ethrange
   Description :
     Conversion de plage d'adresses ethernet en deux
     adresses ethernet
   Parametre(s) d'entree :
     range : lcrzo_ethrange a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ethainf : valeur inferieure de la plage
     ethasup : valeur superieure de la plage
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_etha_init_ethrange(lcrzo_constethrange range,
			     lcrzo_etha ethainf,
			     lcrzo_etha ethasup);

/*en Name : lcrzo_ethrange_init_etha
   Description :
     Conversion from two lcrzo_etha to a lcrzo_ethrange
   Input parameter(s) :
     ethainf : inferior value of the range
     ethasup : superior value of the range
   Input/output parameter(s) :
   Output parameter(s) :
     range : lcrzo_ethrange ethainf-ethasup
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ethrange_init_etha
   Description :
     Conversion de deux adresses ethernet en plage 
     d'adresses ethernet
   Parametre(s) d'entree :
     ethainf : valeur inferieure de la plage
     ethasup : valeur superieure de la plage
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     range : lcrzo_ethrange ethainf-ethasup
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ethrange_init_etha(lcrzo_etha ethainf,
			     lcrzo_etha ethasup,
			     lcrzo_ethrange range);

/*en Name : lcrzo_ethrange_init_stdin
   Description :
     Initialize an lcrzo_ethrange with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaultrange : default lcrzo_ethrange to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     ethrange : lcrzo_ethrange set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ethrange_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultrange : lcrzo_ethrange par defaut a utiliser si l'utilisateur
                    n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     ethrange : lcrzo_ethrange affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ethrange_init_stdin(lcrzo_conststring message,
			      lcrzo_conststring defaultrange,
			      lcrzo_ethrange ethrange);

/*-------------------------------------------------------------*/
/*en definition of an IP range */
/*fr definition d'une plage d'adresses IP */
#define LCRZO_IPRANGE_MAXBYTES   31
typedef char  lcrzo_iprange[LCRZO_IPRANGE_MAXBYTES+1]; /* +1 for/pour '\0' */
typedef const lcrzo_iprange lcrzo_constiprange;

/*en examples of a lcrzo_iprange : */
/*fr exemples de lcrzo_iprange : */
/*  lcrzo_iprange               :     iplinf     iplsup
  1.2.3.4                       :     1.2.3.4     1.2.3.4
  1.2.3.4-5.6.7.8               :     1.2.3.4     5.6.7.8
  1.2.3.0/255.255.255.0         :     1.2.3.0   1.2.3.255
  1.2.3.0/24                    :     1.2.3.0   1.2.3.255
  1.2.3/255.255.255.0           :     1.2.3.0   1.2.3.255
  1.2.3/24                      :     1.2.3.0   1.2.3.255
  1.2.3.0%255.255.255.O         :     1.2.3.1   1.2.3.254
  1.2.3.0%24                    :     1.2.3.1   1.2.3.254
  1.2.3%255.255.255.O           :     1.2.3.1   1.2.3.254
  1.2.3%24                      :     1.2.3.1   1.2.3.254
*/
/*en note : '%' has the same meaning as '/', except that the 
  broadcast adresses are excluded from range.*/
/*fr note : '%' a la meme signification que '/', sauf que les adresses
  broadcast sont excluses de l'intervalle.*/

/*en Name : lcrzo_ipl_init_iprange
   Description :
     Conversion from a lcrzo_iprange to two IP addresses
   Input parameter(s) :
     range : lcrzo_iprange to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *piplinf : inferior value of the range
     *piplsup : superior value of the range
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipl_init_iprange
   Description :
     Conversion de plage d'adresses IP en deux adresses IP
   Parametre(s) d'entree :
     range : lcrzo_iprange a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *piplinf : valeur inferieure de la plage
     *piplsup : valeur superieure de la plage
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
/*en
  It is possible to use a hostname range as input. In this case the
  separator is '=' (because '-' is a valid hostname separator). For
  example : */
/*fr
  Il est possible d'employer un intervalle de nom de machines comme
  entree. Dans ce cas, le separateur est '=' (car '-' est un
  separateur valide de hostname). Par exemple : */
/*  www.server.com
    hostname1=hostname2
    1.2.3.4=hostname2
    hostname1=5.6.7.8
*/
int lcrzo_ipl_init_iprange(lcrzo_conststring range,
			   lcrzo_ipl *piplinf,
			   lcrzo_ipl *piplsup);

/*en Name : lcrzo_iprange_init_ipl
   Description :
     Conversion from two lcrzo_ipl to a lcrzo_iprange
   Input parameter(s) :
     iplinf : inferior value of the range
     iplsup : superior value of the range
   Input/output parameter(s) :
   Output parameter(s) :
     range : lcrzo_iprange iplinf-iplsup
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_iprange_init_ipl
   Description :
     Conversion de deux adresses IP en plage 
     d'adresses IP
   Parametre(s) d'entree :
     iplinf : valeur inferieure de la plage
     iplsup : valeur superieure de la plage
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     range : lcrzo_iprange iplinf-iplsup
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_iprange_init_ipl(lcrzo_ipl iplinf,
			   lcrzo_ipl iplsup,
			   lcrzo_iprange range);

/*en Name : lcrzo_iprange_init_stdin
   Description :
     Initialize an lcrzo_iprange with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaultrange : default lcrzo_iprange to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     iprange : lcrzo_iprange set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_iprange_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultrange : lcrzo_iprange par defaut a utiliser si l'utilisateur
                    n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     iprange : lcrzo_iprange affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_iprange_init_stdin(lcrzo_conststring message,
			     lcrzo_conststring defaultrange,
			     lcrzo_iprange iprange);

/*-------------------------------------------------------------*/
/*en definition of a port range */
/*fr definition d'une plage de ports */
#define LCRZO_PORTRANGE_MAXBYTES 11
typedef char  lcrzo_portrange[LCRZO_PORTRANGE_MAXBYTES+1]; /*+1for/pour'\0'*/
typedef const lcrzo_portrange lcrzo_constportrange;

/*en examples of a lcrzo_iprange : */
/*fr exemples de lcrzo_iprange : */
/*lcrzo_portrange : portinf  portsup
  23              :    23        23
  23-25           :    23        25
  23-             :    23     65535
  0-21            :     0        21
  1-21            :     1        21
  -21             :     1        21
  -               :     1     65535
*/
/*en note : the default inferior port is 1 and not 0.*/
/*fr note : le port inferieur par defaut est 1 et non 0. En effet,
  un port '0' n'a que peu d'interet.*/

/*en Name : lcrzo_port_init_portrange
   Description :
     Conversion from a lcrzo_portrange to two ports
   Input parameter(s) :
     range : lcrzo_portrange to convert
   Input/output parameter(s) :
   Output parameter(s) :
     *pportinf : inferior value of the range
     *pportsup : superior value of the range
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_port_init_portrange
   Description :
     Conversion de plage de ports  en deux ports
   Parametre(s) d'entree :
     range : lcrzo_portrange a convertir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pportinf : valeur inferieure de la plage
     *pportsup : valeur superieure de la plage
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_port_init_portrange(lcrzo_constportrange range,
			      lcrzo_uint16 *pportinf,
			      lcrzo_uint16 *pportsup);

/*en Name : lcrzo_portrange_init_port
   Description :
     Conversion from two ports to a lcrzo_portrange
   Input parameter(s) :
     portinf : inferior value of the range
     portsup : superior value of the range
   Input/output parameter(s) :
   Output parameter(s) :
     range : lcrzo_portrange portinf-portsup
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_portrange_init_port
   Description :
     Conversion de deux ports en plage de ports
     d'adresses PORT
   Parametre(s) d'entree :
     portinf : valeur inferieure de la plage
     portsup : valeur superieure de la plage
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     range : lcrzo_portrange portinf-portsup
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_portrange_init_port(lcrzo_uint16 portinf,
			      lcrzo_uint16 portsup,
			      lcrzo_portrange range);


/*en Name : lcrzo_portrange_init_stdin
   Description :
     Initialize an lcrzo_portrange with the address entered by the 
     user through the keyboard
   Input parameter(s) :
     message : message to print before
     defaultrange : default lcrzo_portrange to use if user enters nothing
   Input/output parameter(s) :
   Output parameter(s) :
     portrange : lcrzo_portrange set with user's string
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_portrange_init_stdin
   Description :
     Initialise une adresse par une valeur entree au clavier
   Parametre(s) d'entree :
     message : message a afficher avant de demander a entrer
     defaultrange : lcrzo_portrange par defaut a utiliser si l'utilisateur
                    n'entre rien
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     portrange : lcrzo_portrange affecte par la valeur entree au clavier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_portrange_init_stdin(lcrzo_conststring message,
			       lcrzo_conststring defaultrange,
			       lcrzo_portrange portrange);
