/*en    Laurent Constantin's network library (lcrzo)
                 lcrzo_wait module

  Functions waiting for data to be read.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_wait

  Les fonctions presentes dans ce module permettent d'attendre
  que des donnees soient pretes a etre lues.
*/

/*-------------------------------------------------------------*/
/*en definition of a lcrzo_wait. You should not use its
   fields directly. */
/*fr definition d'un lcrzo_wait. Vous ne devriez pas utiliser
   directement ses champs. */
typedef struct 
{ int source;
  void *psource;
  int action;
  lcrzo_int32 wantedfixedsize;
} lcrzo_wait;

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_init_process_end
   Description :
     Initialize a lcrzo_wait : wait for the end of a process.
   Input parameter(s) :
     *pprocess : process to wait for
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pprocess
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_process_end
   Description :
     Initialise un lcrzo_wait : attend la fin d'un processus.
   Parametre(s) d'entree :
     *pprocess : processus a attendre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pprocess
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_process_end(lcrzo_process *pprocess, 
				lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_init_file_read
   Description :
     Initialize a lcrzo_wait : wait for data from the file.
   Input parameter(s) :
     *pfile : lcrzo_file where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_file_read
   Description :
     Initialise un lcrzo_wait : attend des donnees sur le fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_file_read(lcrzo_file *pfile, 
			      lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_file_read_fixed
   Description :
     Initialize a lcrzo_wait : wait for fixed data from the file.
   Input parameter(s) :
     *pfile : lcrzo_file where to wait for data
     wantedfixedsize : wanted size
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_file_read_fixed
   Description :
     Initialise un lcrzo_wait : attend des donnees fixes sur le fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file sur lequel attendre des donnees
     wantedfixedsize : taille des donnees voulues
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_file_read_fixed(lcrzo_file *pfile,
				    lcrzo_int32 wantedfixedsize,
				    lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_file_read_line
   Description :
     Initialize a lcrzo_wait : wait for a line from the file.
   Input parameter(s) :
     *pfile : lcrzo_file where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_file_read_line
   Description :
     Initialise un lcrzo_wait : attend une ligne sur le fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_file_read_line(lcrzo_file *pfile,
				   lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_file_end
   Description :
     Initialize a lcrzo_wait : wait for end of file.
   Input parameter(s) :
     *pfile : lcrzo_file where to wait for end
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pfile
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_file_end
   Description :
     Initialise un lcrzo_wait : attend la fin du fichier.
   Parametre(s) d'entree :
     *pfile : lcrzo_file sur lequel attendre la fin
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pfile
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_file_end(lcrzo_file *pfile,
			     lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_init_ipc_read
   Description :
     Initialize a lcrzo_wait : wait for data from the ipc.
   Input parameter(s) :
     *pipc : lcrzo_ipc where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pipc
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_ipc_read
   Description :
     Initialise un lcrzo_wait : attend des donnees sur le fichier.
   Parametre(s) d'entree :
     *pipc : lcrzo_ipc sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_ipc_read(lcrzo_ipc *pipc,
			     lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_ipc_read_fixed
   Description :
     Initialize a lcrzo_wait : wait for fixed data from the ipc.
   Input parameter(s) :
     *pipc : lcrzo_ipc where to wait for data
     wantedfixedsize : wanted size
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pipc
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_ipc_read_fixed
   Description :
     Initialise un lcrzo_wait : attend des donnees fixes sur le fichier.
   Parametre(s) d'entree :
     *pipc : lcrzo_ipc sur lequel attendre des donnees
     wantedfixedsize : taille des donnees voulues
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_ipc_read_fixed(lcrzo_ipc *pipc, 
				   lcrzo_int32 wantedfixedsize, 
				   lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_ipc_read_line
   Description :
     Initialize a lcrzo_wait : wait for a line from the ipc.
   Input parameter(s) :
     *pipc : lcrzo_ipc where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pipc
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_ipc_read_line
   Description :
     Initialise un lcrzo_wait : attend une ligne sur le fichier.
   Parametre(s) d'entree :
     *pipc : lcrzo_ipc sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_ipc_read_line(lcrzo_ipc *pipc,
				  lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_ipc_end
   Description :
     Initialize a lcrzo_wait : wait for end of ipc.
   Input parameter(s) :
     *pipc : lcrzo_ipc where to wait for end
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *pipc
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_ipc_end
   Description :
     Initialise un lcrzo_wait : attend la fin du fichier.
   Parametre(s) d'entree :
     *pipc : lcrzo_ipc sur lequel attendre la fin
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *pipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_ipc_end(lcrzo_ipc *pipc, 
			    lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_init_sniff_read
   Description :
     Initialize a lcrzo_wait : wait for data from the sniff.
   Input parameter(s) :
     *psniff : lcrzo_sniff where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *psniff
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_sniff_read
   Description :
     Initialise un lcrzo_wait : attend des donnees sur le sniff.
   Parametre(s) d'entree :
     *psniff : lcrzo_sniff sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *psniff
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_sniff_read(lcrzo_sniff *psniff, 
                               lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_init_sock_read
   Description :
     Initialize a lcrzo_wait : wait for data from the sock.
   Input parameter(s) :
     *psock : lcrzo_sock where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *psock
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_sock_read
   Description :
     Initialise un lcrzo_wait : attend des donnees sur le fichier.
   Parametre(s) d'entree :
     *psock : lcrzo_sock sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *psock
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_sock_read(lcrzo_sock *psock, 
			      lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_sock_read_fixed
   Description :
     Initialize a lcrzo_wait : wait for fixed data from the sock.
   Input parameter(s) :
     *psock : lcrzo_sock where to wait for data
     wantedfixedsize : wanted size
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *psock
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_sock_read_fixed
   Description :
     Initialise un lcrzo_wait : attend des donnees fixes sur le fichier.
   Parametre(s) d'entree :
     *psock : lcrzo_sock sur lequel attendre des donnees
     wantedfixedsize : taille des donnees voulues
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *psock
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_sock_read_fixed(lcrzo_sock *psock,
				    lcrzo_int32 wantedfixedsize,
				    lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_sock_read_line
   Description :
     Initialize a lcrzo_wait : wait for a line from the sock.
   Input parameter(s) :
     *psock : lcrzo_sock where to wait for data
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *psock
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_sock_read_line
   Description :
     Initialise un lcrzo_wait : attend une ligne sur le fichier.
   Parametre(s) d'entree :
     *psock : lcrzo_sock sur lequel attendre des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *psock
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_sock_read_line(lcrzo_sock *psock,
				   lcrzo_wait *pwait);

/*en Name : lcrzo_wait_init_sock_end
   Description :
     Initialize a lcrzo_wait : wait for end of sock.
   Input parameter(s) :
     *psock : lcrzo_sock where to wait for end
   Input/output parameter(s) :
   Output parameter(s) :
     *pwait : lcrzo_wait associated to *psock
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_init_sock_end
   Description :
     Initialise un lcrzo_wait : attend la fin du fichier.
   Parametre(s) d'entree :
     *psock : lcrzo_sock sur lequel attendre la fin
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pwait : lcrzo_wait associe a *psock
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_init_sock_end(lcrzo_sock *psock,
			     lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_close
   Description :
     Close a lcrzo_wait.
   Input parameter(s) :
     *pwait : lcrzo_wait to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_close
   Description :
     Ferme un lcrzo_wait.
   Parametre(s) d'entree :
     *pwait : lcrzo_wait a fermer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_close(lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Use LCRZO_WAIT_NODELAY to wait 0 seconds */
/*fr Employez LCRZO_WAIT_NODELAY pour attendre 0 secondes */
#define LCRZO_WAIT_NODELAY   lcrzo__time_init_wait_nodelay()
/*en Use LCRZO_WAIT_INFINITE to wait indefinitively */
/*fr Employez LCRZO_WAIT_INFINITE pour attendre indefiniment */
#define LCRZO_WAIT_INFINITE  lcrzo__time_init_wait_infinite()
/*en Do not use the 2 following functions, but the defines above */
/*fr N'utilisez pas ces fonctions, mais les defines ci-dessus */
lcrzo_time * lcrzo__time_init_wait_nodelay(void);
lcrzo_time * lcrzo__time_init_wait_infinite(void);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_wait1
   Description :
     Wait for 1 event.
   Input parameter(s) :
     *pwait1 : lcrzo_wait to wait for
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pevent1 : an event occured on *pwait1
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_wait1
   Description :
     Attend 1 evenement.
   Parametre(s) d'entree :
     *pwait1 : lcrzo_wait sur lequel attendre
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE) 
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pvent1 : un evenement est arrive sur *pwait1
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_wait1(lcrzo_wait *pwait1, 
                     lcrzo_time *pmaxwaitingtime, 
                     lcrzo_bool *pevent1);

/*en Name : lcrzo_wait_wait2
   Description :
     Wait for 1 event amongst 2 lcrzo_wait.
   Input parameter(s) :
     *pwait1 : lcrzo_wait to wait for
     *pwait2 : lcrzo_wait to wait for
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pevent1 : an event occured on *pwait1
     *pevent2 : an event occured on *pwait2
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_wait2
   Description :
     Attend 1 evenement sur 2 lcrzo_wait.
   Parametre(s) d'entree :
     *pwait1 : lcrzo_wait sur lequel attendre
     *pwait2 : lcrzo_wait sur lequel attendre
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE) 
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pvent1 : un evenement est arrive sur *pwait1
     *pvent2 : un evenement est arrive sur *pwait2
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_wait2(lcrzo_wait *pwait1, 
                     lcrzo_wait *pwait2, 
                     lcrzo_time *pmaxwaitingtime, 
                     lcrzo_bool *pevent1, 
                     lcrzo_bool *pevent2);

/*en Name : lcrzo_wait_wait2_all
   Description :
     Wait for 2 events.
   Input parameter(s) :
     *pwait1 : lcrzo_wait to wait for
     *pwait2 : lcrzo_wait to wait for
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
   Output parameter(s) :
     *pevent : an event occured on *pwait1 and on *pwait2
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_wait2_all
   Description :
     Attend 2 evenements.
   Parametre(s) d'entree :
     *pwait1 : lcrzo_wait sur lequel attendre
     *pwait2 : lcrzo_wait sur lequel attendre
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE) 
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pvent : un evenement est arrive sur *pwait1 et *pwait2
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_wait2_all(lcrzo_wait *pwait1, 
                         lcrzo_wait *pwait2, 
                         lcrzo_time *pmaxwaitingtime, 
                         lcrzo_bool *pevent);

/*en Idem with 3, 4 or 5 events */
/*fr Idem avec 3, 4 ou 5 evenements */
int lcrzo_wait_wait3(lcrzo_wait *pwait1, 
                     lcrzo_wait *pwait2, 
                     lcrzo_wait *pwait3, 
                     lcrzo_time *pmaxwaitingtime, 
                     lcrzo_bool *pevent1, 
                     lcrzo_bool *pevent2, 
                     lcrzo_bool *pevent3);
int lcrzo_wait_wait3_all(lcrzo_wait *pwait1, 
                         lcrzo_wait *pwait2, 
                         lcrzo_wait *pwait3, 
                         lcrzo_time *pmaxwaitingtime, 
                         lcrzo_bool *pevent);
int lcrzo_wait_wait4(lcrzo_wait *pwait1, 
                     lcrzo_wait *pwait2, 
                     lcrzo_wait *pwait3, 
                     lcrzo_wait *pwait4, 
                     lcrzo_time *pmaxwaitingtime, 
                     lcrzo_bool *pevent1, 
                     lcrzo_bool *pevent2, 
                     lcrzo_bool *pevent3, 
                     lcrzo_bool *pevent4);
int lcrzo_wait_wait4_all(lcrzo_wait *pwait1, 
                         lcrzo_wait *pwait2, 
                         lcrzo_wait *pwait3, 
                         lcrzo_wait *pwait4, 
                         lcrzo_time *pmaxwaitingtime, 
                         lcrzo_bool *pevent);
int lcrzo_wait_wait5(lcrzo_wait *pwait1, 
                     lcrzo_wait *pwait2, 
                     lcrzo_wait *pwait3, 
                     lcrzo_wait *pwait4, 
                     lcrzo_wait *pwait5, 
                     lcrzo_time *pmaxwaitingtime,
                     lcrzo_bool *pevent1, 
                     lcrzo_bool *pevent2, 
                     lcrzo_bool *pevent3, 
                     lcrzo_bool *pevent4, 
                     lcrzo_bool *pevent5);
int lcrzo_wait_wait5_all(lcrzo_wait *pwait1, 
                         lcrzo_wait *pwait2, 
                         lcrzo_wait *pwait3, 
                         lcrzo_wait *pwait4, 
                         lcrzo_wait *pwait5, 
                         lcrzo_time *pmaxwaitingtime,
                         lcrzo_bool *pevent);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_list_init
   Description :
     Initialize a list a events. It can be use to wait for more 
     than 5 event.
   Input parameter(s) :
   Input/output parameter(s) :
     *pwaitlist : list initialized
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_list_init
   Description :
     Initialise une liste d'evenements. Cela sert pour attendre
     sur plus de 5 evenements.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *pwaitlist : liste initialisee
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_list_init(lcrzo_list *pwaitlist);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_list_add
   Description :
     Add an event to the waiting list.
   Input parameter(s) :
     itemindex : index of the item to add
     *pwait : lcrzo_wait to add
   Input/output parameter(s) :
     *pwaitlist : list where to add items
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_list_add
   Description :
     Ajoute un evenement a une liste d'evenements.
   Parametre(s) d'entree :
     itemindex : index de l'element a ajouter
     *pwait : lcrzo_wait a ajouter
   Parametre(s) d'entree-sortie :
     *pwaitlist : liste d'evenements ou ajouter
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_list_add(lcrzo_list *pwaitlist,
                        lcrzo_int32 itemindex,
                        lcrzo_wait *pwait);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_list_del
   Description :
     Remove an event to the waiting list.
   Input parameter(s) :
     itemindex : index of the item to remove
   Input/output parameter(s) :
     *pwaitlist : list where to remove items
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_list_del
   Description :
     Enleve un evenement a une liste d'evenements.
   Parametre(s) d'entree :
     itemindex : index de l'element a enlever
   Parametre(s) d'entree-sortie :
     *pwaitlist : liste d'evenements ou enlever
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_list_del(lcrzo_list *pwaitlist,
                        lcrzo_int32 itemindex);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_list_wait
   Description :
     Wait for an event in the waiting list.
   Input parameter(s) :
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
     *pwaitlist : list where to wait for
   Output parameter(s) :
     *pevent : true if an event occured
     *pfirsteventindex : index of the first event
     *plistofalleventindexes : list of all events indexes
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_list_wait
   Description :
     Attend un evenement d'une liste d'evenements.
   Parametre(s) d'entree :
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
     *pwaitlist : liste d'evenements ou attendre
   Parametre(s) de sortie :
     *pevent : vrai si un evenement est arrive
     *pfirsteventindex : index du premier evenement
     *plistofalleventindexes : liste de tous les index d'evenements
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_list_wait(lcrzo_list *pwaitlist,
                         lcrzo_time *pmaxwaitingtime, 
                         lcrzo_bool *pevent, 
                         lcrzo_int32 *pfirsteventindex,
                         lcrzo_list *plistofalleventindexes);

/*-------------------------------------------------------------*/
/*en Name : lcrzo_wait_list_wait_all
   Description :
     Wait for all events in the waiting list.
   Input parameter(s) :
     *pmaxwaitingtime : maximum waiting time (can be a lcrzo_time,
                        LCRZO_WAIT_NODELAY or LCRZO_WAIT_INFINITE)
   Input/output parameter(s) :
     *pwaitlist : list where to wait for
   Output parameter(s) :
     *pevent : true if all events occured
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_wait_list_wait_all
   Description :
     Attend tous les evenements d'une liste d'evenements.
   Parametre(s) d'entree :
     *pmaxwaitingtime : temps maximum d'attente (peut etre un
                        lcrzo_time, LCRZO_WAIT_NODELAY ou
                        LCRZO_WAIT_INFINITE)
   Parametre(s) d'entree-sortie :
     *pwaitlist : liste d'evenements ou attendre
   Parametre(s) de sortie :
     *pevent : vrai si tous les evenements sont arrives
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_wait_list_wait_all(lcrzo_list *pwaitlist,
                             lcrzo_time *pmaxwaitingtime, 
                             lcrzo_bool *pevent);

