class MaxSizeError < StandardError; end

class Worksheet < BIFFWriter

   RowMax = 65536
   ColMax = 256
   StrMax = 255
   Buffer = 4096

   attr_reader :name, :data, :xf_index
   attr_accessor :index, :colinfo, :selection, :offset

   def initialize(name,index=0,activesheet=0,firstsheet=0,url_format=nil)
      super(name,index)

      @name        = name
      @index       = index
      @activesheet = activesheet
      @firstsheet  = firstsheet
      @url_format  = url_format

      @offset = 0
      @dim_rowmin = RowMax + 1
      @dim_rowmax = 0
      @dim_colmin = ColMax + 1
      @dim_colmax = 0

      @colinfo = []
      @selection = [0,0]
   end

   def close
      store_dimensions

      unless @colinfo.empty?
         while @colinfo.length > 0
            store_colinfo(@colinfo.pop)
         end
         store_defcol
      end

      store_bof(0x0010)

      store_window2
      store_selection(*@selection)
      store_eof
   end

   def data
      if (defined? @data) && (@data.length > 0)
         tmp = @data
         @data = ""
         return tmp
      end
      return nil
   end

   def activate
      @activesheet = @index
   end

   def set_first_sheet
      @firstsheet = @index
   end

   def column(*range)
      @colinfo = *range
   end

   def selection(*values)
      @selection = *values
   end

   def store_dimensions
      record   = 0x0000
      length   = 0x000A
      reserved = 0x0000

      header = [record,length].pack("vv")
      fields = [@dim_rowmin,@dim_rowmax,@dim_colmin,@dim_colmax,reserved]
      data = fields.pack("vvvvv")
      prepend(header,data)
   end

   def store_window2
      record  = 0x023E
      length  = 0x000A

      grbit   = 0x00B6
      rwTop   = 0x0000
      colLeft = 0x0000
      rgbHdr  = 0x00000000

      if @activesheet == @index
        grbit = 0x06B6
      end

      header = [record,length].pack("vv")
      data   = [grbit, rwTop, colLeft, rgbHdr].pack("vvvV")

      append(header, data)
   end

   def store_defcol
      record   = 0x0055
      length   = 0x0002

      colwidth = 0x0008

      header = [record,length].pack("vv")
      data   = [colwidth].pack("v")

      prepend(header,data)
   end

   def store_colinfo(first=0,last=0,coldx=8.43,ixfe=0x0F,grbit=0)
      record   = 0x007D
      length   = 0x000B

      coldx += 0.72
      coldx *= 256
      reserved = 0x00

      if ixfe.kind_of?(Format)
         ixfe = ixfe.xf_index
      end

      header = [record,length].pack("vv")
      data   = [first,last,coldx,ixfe,grbit,reserved].pack("vvvvvC")

      prepend(header,data)
   end

   # I think this may have problems
   def store_selection(row=0,col=0,row_last=0,col_last=0)
      record = 0x001D
      length = 0x000F

      pnn     = 3
      rwAct   = row
      colAct  = col
      irefAct = 0
      cref    = 1

      rwFirst  = row
      colFirst = col

      if row_last != 0
         rwLast = row_last
      else
         rwLast = rwFirst
      end

      if col_last != 0
         colLast = col_last
      else
         colLast = colFirst
      end

      if rwFirst > rwLast
        rwFirst,rwLast = rwLast,rwFirst
      end

      if colFirst > colLast
        colFirst,colLast = colLast,colFirst
      end

      header = [record,length].pack("vv")
      fields = [pnn,rwAct,colAct,irefAct,cref,rwFirst,rwLast,colFirst,colLast]
      data   = fields.pack("CvvvvvvCC")

      append(header,data)
   end

   def write(row,col,data=nil,format=nil)
      if data.nil?
         write_blank(row,col,format)
      elsif data.kind_of?(Array)
         write_row(row,col,data,format)
      elsif data.kind_of?(Numeric)
         write_number(row,col,data,format)
      else
         write_string(row,col,data,format)
      end
   end

   def write_row(row,col,tokens=nil,format=nil)
      tokens.each{|token|
         if token.kind_of?(Array)
            write_column(row,col,token,format)
         else
            write(row,col,token,format)
         end
         col += 1
      }
   end

   def write_column(row,col,tokens=nil,format=nil)
      tokens.each{|token|
         if token.kind_of?(Array)
            write_row(row,col,token,format)
         else
            write(row,col,token,format)
         end
         row += 1
      }
   end

   def write_number(row,col,num,format)
      record  = 0x0203
      length  = 0x000E
      
      if format.nil?
         xf = 0x0F
      else
         xf = format.xf_index
      end

      raise MaxSizeError if row >= RowMax
      raise MaxSizeError if col >= ColMax

      @dim_rowmin = row if row < @dim_rowmin
      @dim_rowmax = row if row > @dim_rowmax
      @dim_colmin = col if col < @dim_colmin
      @dim_colmax = col if col > @dim_colmax

      header    = [record,length].pack("vv")
      data      = [row,col,xf].pack("vvv")
      xl_double = [num].pack("d")

      if @big_endian == true
         xl_double.reverse!
      end

      append(header,data,xl_double)

      return 0
   end

   def write_string(row,col,str,format)
      record = 0x0204
      length = 0x0008 + str.length

      if format.nil?
         xf = 0x0F
      else
         xf = format.xf_index
      end

      strlen = str.length

      raise MaxSizeError if row >= RowMax
      raise MaxSizeError if col >= ColMax
      raise MaxSizeError if strlen > StrMax

      @dim_rowmin = row if row < @dim_rowmin
      @dim_rowmax = row if row > @dim_rowmax
      @dim_colmin = col if col < @dim_colmin
      @dim_colmax = col if col > @dim_colmax

      if strlen > StrMax
         str = str[0..StrMax-1]
         length = 0x0008 + StrMax
         strlen = StrMax
      end

      header = [record,length].pack("vv")
      data   = [row,col,xf,strlen].pack("vvvv")

      append(header,data,str)
   end

   def write_blank(row,col,format)
      record = 0x0201
      length = 0x0006

      if format.nil?
         xf = 0x0F
      else
         xf = format.xf_index
      end

      raise MaxSizeError if row >= RowMax
      raise MaxSizeError if col >= ColMax

      @dim_rowmin = row if row < @dim_rowmin
      @dim_rowmax = row if row > @dim_rowmax
      @dim_colmin = col if col < @dim_colmin
      @dim_colmax = col if col > @dim_colmax

      header = [record,length].pack("vv")
      data = [row,col,xf].pack("vvv")

      append(header + data)
      return 0
   end

   def write_url(row,col,url,string=url,format=nil)
      record = 0x01B8
      length = 0x0034 + 2 * (1+url.length)
      
      write_string(row,col,str,format)

      header = [record,length].pack("vv")
      data   = [row,row,col,col].pack("vvvv")

      unknown = "D0C9EA79F9BACE118C8200AA004BA90B02000000"
      unknown += "03000000E0C9EA79F9BACE118C8200AA004BA90B"

      stream = [unknown].pack("H*")

      url = url.split('').join("\0")
      url += "\0\0\0"

      len = url.length
      url_len = [len].pack("V")

      append(header + data)
      append(stream)
      append(url_len)
      append(url)
   end

end
=begin
= Differences between Worksheet.pm and worksheet.rb
---write_url
   I made this a public method to be called directly by the user if they want
   to write a url string.  A variable number of arguments made it a pain to
   integrate into the 'write' method.
=end
