/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_COUNTER_H
#define QWT_COUNTER_H

#include <qwidget.h>
#include "qwt_drange.h"

class QwtArrowButton;
class QLabel;

/*!
  \brief The Counter Widget

  A Counter consists of a label displaying a number and
  one ore more (up to three) push buttons on each side
  of the label which can be used to increment or decrement
  the counter's value.

  A Counter has a range from a minimum value to a maximum value
  and a step size. The range can be specified using
  QwtDblRange::setRange().
  The counter's value is an integer multiple of the step size.
  The number of steps by which a button increments or decrements
  the value can be specified using @QwtCounter::setIncSteps@.
  The number of buttons can be changed with
  setNumButtons().

  Example:
\code
#include "../include/qwt_counter.h>

QwtCounter *cnt;

cnt = new QwtCounter(parent, name);

cnt->setRange(0.0, 100.0, 1.0);             // From 0.0 to 100, step 1.0
cnt->setNumButtons(2);                      // Two buttons each side
cnt->setIncSteps(QwtCounter::Button1, 1);   // Button 1 increments 1 step
cnt->setIncSteps(QwtCounter::Button2, 20);  // Button 2 increments 20 steps

connect(cnt, SIGNAL(valueChanged(double)), my_class, SLOT(newValue(double)));
\endcode
 */
class QwtCounter : public QWidget, public QwtDblRange
{
  Q_OBJECT
public:
  enum Button { Button1, Button2, Button3, ButtonCnt };
private:
  int tmrID;
  int d_tmrIntv;
  int d_tick;
  int d_editable;
  
  QwtArrowButton *btnL[ButtonCnt];
  QwtArrowButton *btnR[ButtonCnt];
  QLabel *lblValue;
  
  int d_incValue;
  int d_Inc[ButtonCnt];
  int d_nButtons;
  
  void valueChange();
public:
  QwtCounter(QWidget *parent = 0, char *name = 0);
  ~QwtCounter();
  
  void setNumButtons(int n);
  void setIncSteps(QwtCounter::Button btn, int nSteps);
  
  int incSteps(QwtCounter::Button btn) const;
  /*!
    \return The number of buttons on each side of the widget.
   */
  int numButtons() const { return d_nButtons; } 
  
  void setEnabled(bool tf);
  QSize sizeHint() const;
protected:
  void fontChange(const QFont &f);
  void resizeEvent(QResizeEvent *e);
  void timerEvent(QTimerEvent *e);
private slots:
  void btnL1Dn();
  void btnL2Dn();
  void btnL3Dn();
  void btnR1Dn();
  void btnR2Dn();
  void btnR3Dn();
  void btnUp();
signals:
  /*!
    This signal is emitted when a button has been released
    \param value The new value
   */
  void valueChanged (double value);
  /*!
    This signal is emitted when the counter's value has changed
    \param value The new value
   */
  void buttonReleased (double value);  
};

#endif

