/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_CLRFLTR_H
#define QWT_CLRFLTR_H

#include <qcolor.h>


/*!
  \brief A base class for color filters

  A color filter is a class which can be used to modify colors
  in a specific way. A QwtColorFilter has an @QwtColorFilter::operator()@
  which takes a reference to a QColor as its argument and returns the
  result of the modification. The QwtColorFilter class itself does not
  modify colors, but is intended as a base class for user-defined color
  filters. Building a color filter is quite easy: Derive a class from
  QwtColorFilter and redefine the operator() member. Qwt has several
  pre-defined color filters (see below).
  
  Color filters are used by QwtPlot::print().
*/  
class QwtColorFilter
{
public:
  //! Dtor
  virtual ~QwtColorFilter() {}
  /*!
    \brief Filter function
    
    The filter function can be redefined by derived classes.
    \return The default implementation returns its argument.
    \param c modified color
  */
  virtual QColor operator() (const QColor& c) const { return c; }
};


//! This color filter can be used to darken colors
class QwtFltrDim : public QwtColorFilter
{
private:
  int d_dim;
public:
  QwtFltrDim (int dim = 100);
  QColor operator()(const QColor &c) const;
  /*! \return dim factor */
  int dimFactor() const { return d_dim; }
};

//! Maps colors into a grayscale with optional inversion.
class QwtFltrGray : public QwtColorFilter
{
private:
  bool d_inv;
public:
  /*!
    \brief CTOR
    \param inv Invert the result if \c TRUE
  */
  QwtFltrGray (bool inv = FALSE) { d_inv = inv; }
  QColor operator() (const QColor &c) const;
};

/*!
  \brief Inverts the RGB values of colors.
*/
class QwtFltrInv : public QwtColorFilter
{
public:
  QColor operator()(const QColor &c) const;
};


extern QwtColorFilter NoColorFilter;

#endif
