/*-*- c++ -*-******************************************************************
 * Qwt Widget Library 
 * Copyright (C) 1997   Josef Wilgen
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef QWT_SCLDIV_H
#define QWT_SCLDIV_H

#include <qglobal.h>
#include <qarray.h>

/*!
  \brief A class for building scale divisions
  
  The QwtScaleDiv class can build
  linear and logarithmic scale divisions for specified
  intervals. It uses an adjustable algorithm to
  generate the major and minor step widths automatically.
  A scale division has a minimum value, a maximum value,
  a vector of major marks, and a vector of minor marks.

  \warning QwtScaleDiv uses implicit sharing for the mark vectors.
  \par Example
  To build a logarithmic scale division from 0.01 to 1000
  and print out the major and minor marks. \verbatim
#include "../include/qwt_scldiv.h>
#include <iostream.h>

main()
{
  int i,k;
  QwtScaleDiv sd;

  sd.rebuild(0.01, 100, 10, 10, TRUE, 0.0);

  k=0;
  for (i=0;i<sd.majCnt();i++)
  {
    while(k < sd.minCnt())
    {
      if (sd.minMark(k) < sd.majMark(i))
      {
        cout << " - " << sd.minMark(i) << "\n";
	k++;
      }
      else
        break;
    }
    cout << "-- " << sd.majMark(i) << "\n";
  }
  while(k < sd.minCnt())
  {
    cout << " - " << sd.minMark(i) << "\n";
    k++;
  }
}\endverbatim
*/
class QwtScaleDiv
{
private:
    
    double d_lBound;
    double d_hBound;
    double d_majStep;
    bool d_log;


    QArray<double> d_majMarks;
    QArray<double> d_minMarks;

    void copy(const QwtScaleDiv &s);
    
public:

    QwtScaleDiv ();
    virtual ~QwtScaleDiv();
    QwtScaleDiv(const QwtScaleDiv& s);

    QwtScaleDiv& operator= (const QwtScaleDiv &s);
    int operator== (const QwtScaleDiv &s) const;
    int operator!= (const QwtScaleDiv &s) const;
    
    double lBound() const { return d_lBound; }
    double hBound() const { return d_hBound; }
    int minCnt() const { return d_minMarks.size(); }
    int majCnt() const { return d_majMarks.size(); }
    bool logScale() const { return d_log; }
    double majMark(int i) const { return d_majMarks[i]; }
    double minMark(int i) const { return d_minMarks[i]; }
    double majStep() const { return d_majStep; }
    void reset();

private:

    bool buildLinDiv(int maxMajMark, int maxMinMark, double step = 0.0);
    bool buildLogDiv(int maxMajMark, int maxMinMark, double step = 0.0);

public:

    bool rebuild(double lBound, double hBound, int maxMaj, int maxMin,
			 bool log, double step = 0.0, bool ascend = TRUE);
    
};

#endif
