//////////////////////////////////////////////////////////////////////////////
// Copyright (C) 1993 - 2000 California Institute of Technology             //
//                                                                          //
// Read the COPYING and README files, or contact 'avida@alife.org',         //
// before continuing.  SOME RESTRICTIONS MAY APPLY TO USE OF THIS FILE.     //
//////////////////////////////////////////////////////////////////////////////

#ifndef LINEAGE_HH
#define LINEAGE_HH

#ifndef DEFS_HH
#include "../defs.hh"
#endif

#ifndef GENOTYPE_HH
#include "genotype.hh"
#endif

#include <assert.h>

#include <map>

// struct to compare to genotypes
struct gt_gentype
{
  bool operator()(const cGenotype* g1, const cGenotype* g2) const
  {
    return g1->GetID() > g2->GetID();
  }
};

class cLineage {
private:
  static int m_max_id;
  int m_id;
  int m_parent_id;
  int m_update_born;
  int m_num_CPUs;
  int m_total_CPUs;
  int m_total_genotypes;
  double m_start_fitness;
  mutable double m_max_fitness;
  mutable double m_ave_fitness;
  double m_generation_born;

  map<const cGenotype*, int, gt_gentype> m_genotype_map;
  bool m_threshold;
  mutable bool m_ave_fitness_changed;
  
  void CalcCurrentFitness() const;
public:
  /**
   * Creates a new lineage with a given start fitness and parent id.
   * The first genotype has to be added with AddGenotype later.
   *
   * @param start_fitness The initial fitness of the lineage.
   * @param parent_id The id of the parent's lineage.
   * @param id The requested id for this lineage. If negative, a generated
   * value is used. Attention: if you request lineage id's, it is your
   * responsibility not to request two identical ones for different lineages
   * (the class @ref cLineageControl does that correctly).
   **/
  cLineage(double start_fitness, int parent_id, int id = -1);
  ~cLineage();

  /**
   * Adds one instance of the given genotype to the lineage.
   **/
  void AddCreature( cGenotype *genotype );

  /**
   * Removes on instance of the given genotype from the lineage.
   *
   * @return True if the removal of this creature potentially creates a new
   * best lineage, otherwise false.
   **/
  bool RemoveCreature( cGenotype *genotype );
  void SetThreshold() { m_threshold = true; }

  /**
   * @return The id of the lineage.
   **/
  int GetID() const { return m_id; }

  /**
   * @return The id of the parent lineage.
   **/
  int GetParentID() const { return m_parent_id; }

  /**
   * @return An integer that is one larger than the maximum id used so far.
   **/
  int GetMaxID() const { return m_max_id; }

  /**
   * @return The update at which the lineage was created.
   **/
  int GetUpdateBorn() const { return m_update_born; }

  /**
   * @return The generation at which the lineage was created.
   **/
  double GetGenerationBorn() const { return m_generation_born; }

  /**
   * @return The initial fitness of the lineage.
   **/
  double GetStartFitness() const { return m_start_fitness; }

  /**
   * @return The current maximum fitness of the lineage.
   **/
  double GetMaxFitness() const { return m_max_fitness; }

  /**
   * @return The average fitness of the lineage.
   **/
  double GetAveFitness() const { 
    if ( m_ave_fitness_changed )
      CalcCurrentFitness();
    return m_ave_fitness; }

  /**
   * @return The current number of genotypes in the lineage.
   **/
  int GetNumGenotypes() const { return m_genotype_map.size(); }

  /**
   * @return The current number of creatures in the lineage.
   **/
  int GetNumCreatures() const { return m_num_CPUs; }

  /**
   * Counts the number of creatures in the lineage.
   *
   * Should always equal GetNumCreatures(), otherwise something
   * is wrong.
   **/
  int CountNumCreatures() const;

  /**
   * @return The total number of creatures that have ever existed
   * in this lineage.
   **/
  int GetTotalCreatures() const { return m_total_CPUs; }

  /**
   * @return The total number of genotypes that have ever been part
   * of this lineage.
   **/
  int GetTotalGenotypes() const { return m_total_genotypes; }

  bool GetThreshold() const { return m_threshold; }
};

#endif
