/* QualifierChoice.java
 *
 * created: Tue Sep  7 1999
 *
 * This file is part of Artemis
 *
 * Copyright (C) 1999  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/QualifierChoice.java,v 1.4 2000/02/25 16:31:05 kmr Exp $
 */

package diana.components;

import diana.*;

import uk.ac.sanger.pathogens.StringVector;
import uk.ac.sanger.pathogens.embl.Key;
import uk.ac.sanger.pathogens.embl.InvalidKeyException;
import uk.ac.sanger.pathogens.embl.EntryInformation;

import java.awt.*;
import java.awt.event.*;

/**
 *  This is a Choice component that shows only the qualifier names for a given
 *  key.
 *
 *  @author Kim Rutherford
 *  @version $Id: QualifierChoice.java,v 1.4 2000/02/25 16:31:05 kmr Exp $
 **/

public class QualifierChoice extends Choice {
  /**
   *  Create a new QualifierChoice component for the given Key with note as
   *  the default qualifier in the list.
   *  @param entry_information The object to get the list of possible
   *    qualifiers from.
   **/
  public QualifierChoice (final EntryInformation entry_information,
                          final Key key) {
    this (entry_information, key, "note");
  }
  
  /**
   *  Create a new QualifierChoice component for the given Key with the given
   *  qualifier as the default.
   *  @param entry_information The object to get the list of possible
   *    qualifiers from.
   **/
  public QualifierChoice (final EntryInformation entry_information,
                          final Key key, final String default_qualifier) {
    this.entry_information = entry_information;
    this.key = key;

    if (entry_information.isValidQualifier (key, default_qualifier)) {
      this.default_qualifier = default_qualifier;
    } else {
      this.default_qualifier = null;
    }
    
//      final Font font = Options.getOptions ().getFont ();

//      setFont (font);

    update ();
  }

  /**
   *  Change the qualifiers shown in this component to be those of the given
   *  Key.
   **/
  public void setKey (final Key key) {
    if (this.key != key) {
      this.key = key;

      update ();
    }
  }

  /**
   *  Select the given qualifier_name.
   **/
  public void setQualifierName (final String qualifier_name) {
    final int index = indexOf (qualifier_name);

    if (index == -1) {
      // add the key
      add (qualifier_name);
      select (qualifier_name);
    } else {
      select (index);
    }
  }

  /**
   *  Return the index in the Choice component of the given qualifier_name.
   **/
  private int indexOf (final String qualifier_name) {
    for (int i = 0 ; i < getItemCount () ; ++i) {
      if (getItem (i).equals (qualifier_name)) {
        return i;
      }
    }
    return -1;
  }

  /**
   *  Update the Choice to refect the current Key.
   **/
  private void update () {
    while (getItemCount () > 0) {
      remove (0);
    }

    StringVector qualifier_names =
      entry_information.getValidQualifierNames (key);

    if (qualifier_names == null) {
      qualifier_names = new StringVector ("note");
    } else {
      qualifier_names.sort ();
    }

    for (int i = 0 ; i < qualifier_names.size () ; ++i) {
      add (qualifier_names.elementAt (i));
    }

    if (default_qualifier == null) {
      final StringVector names =
        entry_information.getValidQualifierNames (key);
      setQualifierName (names.elementAt (0));
    } else {
      setQualifierName (default_qualifier);
    }
  }

  /**
   *  The Key that was passed to the constructor.
   **/
  private Key key = null;

  /**
   *  The qualifier name that was passed to the constructor.
   **/
  private String default_qualifier = null;

  /**
   *  The EntryInformation object that was passed to the constructor.
   **/
  private EntryInformation entry_information;
}

