
#include <p2c/p2c.h>

/* "MOSIS", an automatic mosis request maker,
   Copyright (C) 1985, 1990 California Institute of Technology.
   Original author: Massimo Sivilotti
   Unix Port Maintainer: John Lazzaro
   Maintainers's address: lazzaro@hobiecat.cs.caltech.edu;
                          CB 425 CU Boulder/Boulder CO 91125.


This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation (Version 1, 1989).

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


/*   To customize for a particular installation, search for the
     string CUSTOM, and change printf's to reflect the MOSIS contract,
     account name, ect. 

     John Lazzaro, MOSIS program maintainer                            */


/*
   This program accepts a WOL CIF file, and adds all the packaging
   required to submit a project to MOSIS.
*/

/*
   Modifications for general layer conversion: JohnW 10/17/87

   New technologies may be added, assuming there is a one-to-one
   mapping of layer names, by modifying the procedure <initialize_techs>.

   In noninteractive mode the input technology is ASSUMED to be
   the value of <default_wol_tech>, unless one is specified with the
   I switch on the command line. This input technology is used for both
   the case when conversion is requested and without conversion.
   In interactive mode the input technology is prompted for.
*/


#ifndef CITINFOMOD_H
#include <p2c/citinfomod.h>
#endif

#ifndef NEWCI_H
#include <p2c/newci.h>
#endif

#ifndef NEWASM_H
#include <p2c/newasm.h>
#endif


#define max_sym         6000   /* maximum number of symbols in symbol table*/

#define version           "#1.02"
#define Compilation_date  "Feb 7, 1990 11:43 pm"

#define max_x_size      7928   /*um*/
#define max_y_size      9225   /*um*/

#define default_output  "NOCHANGE"
#define default_out_tech  "SCG"
#define default_wol_tech  "SCG"

#define max_layers      15
#define max_techs       10

typedef struct tech_spec {
  Char name[9];
  Char layers[max_layers][9];
  long num_layers;
} tech_spec;


Static Char fname[256], infname[256], fullinfname[256], outfname[256],
	    tmpoutfname[256], arpa_addr[256], quantity_request[256],
	    full_name[256], proj_name[256], bond_same_as[256],
	    technology_name[256], input_technology[256];
Static FILE *outfile, *ciffile;
Static double lambda;
Static long n, padno;
Static short pads[max_sym];
Static Char in_name[20][256], out_name[20][256];
Static boolean no_prompt;   /* silent operation */
Static boolean found_errorboxes, inside_symbol_def, top_level_call;

Static newci_parserec parsetab[11];

Static tech_spec tech_specs[max_techs];
Static long num_techs, num_layers;
Static boolean TinyChip;


Static long max(long a, long b)
{
  if (a > b)
    return a;
  else
    return b;
}


Static long min(long a, long b)
{
  if (a < b)
    return a;
  else
    return b;
}


Static void get_size(Char *s)
{
  long junk, dx, dy, cx, cy;
  Char buff[256];
  long a, b, lastx, lasty;
  Char *TEMP;

  if (ciffile != NULL)
    ciffile = freopen(fullinfname, "r", ciffile);
  else
    ciffile = fopen(fullinfname, "r");

  if (ciffile == NULL)
    _EscIO(FileNotFound);

  a = 1;
  b = 1;
  lastx = 1;
  lasty = 1;
  fgets(buff, 256, ciffile);
  TEMP = strchr(buff, '\n');
  if (TEMP != NULL)
    *TEMP = 0;
  do {
    if (strpos2(buff, "DS", 1) == 1) {
      sscanf(buff + 2, "%ld%ld%ld%ln", &junk, &a, &b, &n);
      n += 3;
      fgets(buff, 256, ciffile);
      TEMP = strchr(buff, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
    } else if (strpos2(buff, "(MBB", 1) == 1) {
      fgets(buff, 256, ciffile);
      TEMP = strchr(buff, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
      /* extract dimensions from the MBB */
      /* debugging only --  writeln (buff); */
      sscanf(buff + 1, "%ld%ld%ld%ld%ln", &dx, &dy, &cx, &cy, &n);
      n += 2;
      lastx = (long)((double)(dx * a) / b / 100);
      lasty = (long)((double)(dy * a) / b / 100);
      do {
	fgets(buff, 256, ciffile);
	TEMP = strchr(buff, '\n');
	if (TEMP != NULL)
	  *TEMP = 0;
      } while (!((strpos2(buff, "DS", 1) == 1) |
		 (strpos2(buff, "E", 1) == 1) | P_eof(ciffile)));
    } else {
      fgets(buff, 256, ciffile);
      TEMP = strchr(buff, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
    }
  } while (!((strpos2(buff, "E", 1) == 1) | P_eof(ciffile)));
  if (ciffile != NULL)
    fclose(ciffile);
  ciffile = NULL;
  s[0] = '\0';

  lambda = a / 100.0;
  printf("\n    Lambda = %0.2f microns.\n", lambda);
  printf("    Project size = %ld x %ld microns\n", lastx, lasty);

  sprintf(s, "%ld x %ld", lastx, lasty);
  junk = strlen(s) + 1;
  if (!((max(lastx, lasty) > max(max_x_size, max_y_size)) |
	(min(lastx, lasty) > min(max_x_size, max_y_size))))
    return;
/* p2c: mosis.text, line 128: Note: Characters >= 128 encountered [281] */
/* p2c: mosis.text, line 128:
 * Note: WRITE statement contains color/attribute characters [203] */
  printf("\214WARNING:  project too big!!!\210\n");
  printf("    Project size = %ld x %ld microns\n",
	 max(lastx, lasty), min(lastx, lasty));
  printf("    Max. size    = %ld x %ld microns\n",
	 max(max_x_size, max_y_size), min(max_x_size, max_y_size));
}


/* Local variables for write_CIF_checksum: */
struct LOC_write_CIF_checksum {
  long chksum;   /* Checksum accumulates here.*/
  long nbyte;   /* Byte count accumulates here.*/
  boolean sepflg;   /* Boolean for prevdataus separator.*/
} ;

Local void sum(Char ch, struct LOC_write_CIF_checksum *LINK)
{
  /* Compute checksum 1 char at a time.*/
  /* Get rid of unwanted bits.*/
  if ((ch & (~127)) != 0)
    ch = (Char)((uchar)ch - 128);

  if (ch > ' ') {   /* Is this a printing character? */
    LINK->chksum += ch;
    LINK->nbyte++;
    LINK->sepflg = false;
    return;
  }

  if (ch == '\0' || LINK->sepflg != false)
    return;
  LINK->chksum += 32;   /* Process the first */
  LINK->nbyte++;   /* separator in a row.*/
  LINK->sepflg = true;
}


Static void write_CIF_checksum(void)
{
  /* shamelessly adapted from MOSIS code */
  struct LOC_write_CIF_checksum V;
  Char c;   /* Gets a character from the file, or EOF.*/


  V.sepflg = true;   /* Set the separator flag.*/
  V.chksum = 32;   /* Initial checksum value.*/
  V.nbyte = 1;   /* Initial byte count.*/

  if (ciffile != NULL)
    ciffile = freopen(tmpoutfname, "r", ciffile);
  else
    ciffile = fopen(tmpoutfname, "r");
  if (ciffile == NULL)
    _EscIO(FileNotFound);
  while (!P_eof(ciffile)) {  /* Compute checksum.*/
    while (!P_eoln(ciffile)) {
      c = getc(ciffile);
      if (c == '\n')
	c = ' ';
      sum(c, &V);
    }
    fscanf(ciffile, "%*[^\n]");
    getc(ciffile);
    sum(' ', &V);   /* Treat new-line as a space.*/
  }


  /* Process the implied trailing separator.*/

  if (V.sepflg == false) {
    V.chksum += 32;
    V.nbyte++;
  }

  fprintf(outfile, "LAYOUT-CHECKSUM: %12ld %12ld\n", V.chksum, V.nbyte);
  printf("LAYOUT-checksum = %12ld\n", V.chksum);
  printf("Byte-count   = %12ld\n", V.nbyte);

  if (ciffile != NULL)
    fclose(ciffile);
  ciffile = NULL;
}



Static void get_pads(long *padno)
{
  /* scan the file to find overglass cuts larger than 50 x 50 um */
  double a, b;
  boolean done, defining_cell;
  Char buff[256];
  long current_cell, called_cell, top_cell, i, junk;
  Char *TEMP;
  Char *STR1;

  for (i = 0; i < max_sym; i++)   /* let all cells have no pads */
    pads[i] = 0;

  *padno = 0;
  done = false;
  defining_cell = false;
  if (ciffile != NULL)
    ciffile = freopen(fullinfname, "r", ciffile);
  else
    ciffile = fopen(fullinfname, "r");
  if (ciffile == NULL)
    _EscIO(FileNotFound);
  while ((!done) & (!P_eof(ciffile))) {
    fgets(buff, 256, ciffile);
    TEMP = strchr(buff, '\n');
    if (TEMP != NULL)
      *TEMP = 0;
    if ((strpos2(buff, "LCG", 1) == 1) | (strpos2(buff, "LCOG", 1) == 1)) {
      fgets(buff, 256, ciffile);
      TEMP = strchr(buff, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
      while (buff[0] == 'B') {
	sscanf(buff + 1, "%lg%lg%ln", &a, &b, &junk);
	junk += 2;
	a = a * lambda / 2;
	b = b * lambda / 2;
	/* writeln ('  found a = ',a,',  b = ',b); */
	if (a > 50 && b > 50) {
	  pads[current_cell - 1]++;
	  /* writeln ('  Found a pad in cell: ',current_cell:0); */
	}
	fgets(buff, 256, ciffile);
	TEMP = strchr(buff, '\n');
	if (TEMP != NULL)
	  *TEMP = 0;
      }
      /* writeln ('  found a LCG box');   */
    }
    if (strpos2(buff, "DS", 1) == 1) {
      current_cell = strtol(buff + 2, &STR1, 10);
      junk = STR1 - buff + 1;
      /* writeln ('currently processing cell: ',current_cell:0); */
      defining_cell = true;
    }
    if (strpos2(buff, "DF", 1) == 1)
      defining_cell = false;
    if (strpos2(buff, "C", 1) == 1) {
      if (defining_cell) {
	called_cell = strtol(buff + 1, &STR1, 10);
	junk = STR1 - buff + 1;
	if (called_cell > max_sym || current_cell > max_sym)
	  printf("FATAL ERROR:  MOSIS symbol table too small\n");
	pads[current_cell - 1] += pads[called_cell - 1];
	/*  writeln ('  Cell: ',current_cell:0,' calls cell: ',called_cell:0,
	         ' (',pads[called_Cell]:0,' pads)');   */
      } else {  /* must be top level call */
	top_cell = strtol(buff + 1, &STR1, 10);
	junk = STR1 - buff + 1;
	*padno = pads[top_cell - 1];
      }
    }
    if (strpos2(buff, "E", 1) == 1)
      done = true;
  }
  if (ciffile != NULL)
    fclose(ciffile);
  ciffile = NULL;

  if (*padno == 0)   /* try guessing at pads */
    *padno = pads[current_cell - 1];
  /* left at top level call*/

  printf("    Pads found = %ld\n", *padno);

}


Static void initialize_techs(void)
{
  long i;

  num_techs = 6;

  i = 1;
  strcpy(tech_specs[i - 1].name, "CBPM2");
  tech_specs[i - 1].num_layers = 10;
  strcpy(tech_specs[i - 1].layers[0], "LCM;");
  strcpy(tech_specs[i - 1].layers[1], "LCP;");
  strcpy(tech_specs[i - 1].layers[2], "LCD;");
  strcpy(tech_specs[i - 1].layers[3], "LCC;");
  strcpy(tech_specs[i - 1].layers[4], "LCC;");
  strcpy(tech_specs[i - 1].layers[5], "LCM2;");
  strcpy(tech_specs[i - 1].layers[6], "LCV;");
  strcpy(tech_specs[i - 1].layers[7], "LCS;");
  strcpy(tech_specs[i - 1].layers[8], "LCW;");
  strcpy(tech_specs[i - 1].layers[9], "LCG;");

  i = 2;
  strcpy(tech_specs[i - 1].name, "SCG");
  tech_specs[i - 1].num_layers = 10;
  strcpy(tech_specs[i - 1].layers[0], "LCMF;");
  strcpy(tech_specs[i - 1].layers[1], "LCPG;");
  strcpy(tech_specs[i - 1].layers[2], "LCAA;");
  strcpy(tech_specs[i - 1].layers[3], "LCCP;");
  strcpy(tech_specs[i - 1].layers[4], "LCCA;");
  strcpy(tech_specs[i - 1].layers[5], "LCMS;");
  strcpy(tech_specs[i - 1].layers[6], "LCVA;");
  strcpy(tech_specs[i - 1].layers[7], "LCSG;");
  strcpy(tech_specs[i - 1].layers[8], "LCWG;");
  strcpy(tech_specs[i - 1].layers[9], "LCOG;");

  i = 3;
  strcpy(tech_specs[i - 1].name, "SCP");
  tech_specs[i - 1].num_layers = 10;
  strcpy(tech_specs[i - 1].layers[0], "LCMF;");
  strcpy(tech_specs[i - 1].layers[1], "LCPG;");
  strcpy(tech_specs[i - 1].layers[2], "LCAA;");
  strcpy(tech_specs[i - 1].layers[3], "LCCP;");
  strcpy(tech_specs[i - 1].layers[4], "LCCA;");
  strcpy(tech_specs[i - 1].layers[5], "LCMS;");
  strcpy(tech_specs[i - 1].layers[6], "LCVA;");
  strcpy(tech_specs[i - 1].layers[7], "LCSP;");
  strcpy(tech_specs[i - 1].layers[8], "LCWP;");
  strcpy(tech_specs[i - 1].layers[9], "LCOG;");

  i = 4;
  strcpy(tech_specs[i - 1].name, "SCN");
  tech_specs[i - 1].num_layers = 10;
  strcpy(tech_specs[i - 1].layers[0], "LCMF;");
  strcpy(tech_specs[i - 1].layers[1], "LCPG;");
  strcpy(tech_specs[i - 1].layers[2], "LCAA;");
  strcpy(tech_specs[i - 1].layers[3], "LCCP;");
  strcpy(tech_specs[i - 1].layers[4], "LCCA;");
  strcpy(tech_specs[i - 1].layers[5], "LCMS;");
  strcpy(tech_specs[i - 1].layers[6], "LCVA;");
  strcpy(tech_specs[i - 1].layers[7], "LCSN;");
  strcpy(tech_specs[i - 1].layers[8], "LCWN;");
  strcpy(tech_specs[i - 1].layers[9], "LCOG;");

  i = 5;
  strcpy(tech_specs[i - 1].name, "SCGE");
  tech_specs[i - 1].num_layers = 12;
  strcpy(tech_specs[i - 1].layers[0], "LCMF;");
  strcpy(tech_specs[i - 1].layers[1], "LCPG;");
  strcpy(tech_specs[i - 1].layers[2], "LCAA;");
  strcpy(tech_specs[i - 1].layers[3], "LCCP;");
  strcpy(tech_specs[i - 1].layers[4], "LCCA;");
  strcpy(tech_specs[i - 1].layers[5], "LCMS;");
  strcpy(tech_specs[i - 1].layers[6], "LCVA;");
  strcpy(tech_specs[i - 1].layers[7], "LCSG;");
  strcpy(tech_specs[i - 1].layers[8], "LCWG;");
  strcpy(tech_specs[i - 1].layers[9], "LCOG;");
  strcpy(tech_specs[i - 1].layers[10], "LCEL;");
  strcpy(tech_specs[i - 1].layers[11], "LCCE;");

  i = 6;
  strcpy(tech_specs[i - 1].name, "SCGA");
  tech_specs[i - 1].num_layers = 14;
  strcpy(tech_specs[i - 1].layers[0], "LCMF;");
  strcpy(tech_specs[i - 1].layers[1], "LCPG;");
  strcpy(tech_specs[i - 1].layers[2], "LCAA;");
  strcpy(tech_specs[i - 1].layers[3], "LCCP;");
  strcpy(tech_specs[i - 1].layers[4], "LCCA;");
  strcpy(tech_specs[i - 1].layers[5], "LCMS;");
  strcpy(tech_specs[i - 1].layers[6], "LCVA;");
  strcpy(tech_specs[i - 1].layers[7], "LCSG;");
  strcpy(tech_specs[i - 1].layers[8], "LCWG;");
  strcpy(tech_specs[i - 1].layers[9], "LCOG;");
  strcpy(tech_specs[i - 1].layers[10], "LCEL;");
  strcpy(tech_specs[i - 1].layers[11], "LCCE;");
  strcpy(tech_specs[i - 1].layers[12], "LCBA;");
  strcpy(tech_specs[i - 1].layers[13], "LCCD;");

}


Static Char *available_techs(Char *Result)
{
  Char s[256];
  long l, i, FORLIM;

  s[0] = '\0';
  l = 1;
  strcpy(s + l - 1, tech_specs[0].name);
  l = strlen(s) + 1;
  FORLIM = num_techs;
  for (i = 1; i < FORLIM; i++) {
    sprintf(s + l - 1, "|%s", tech_specs[i].name);
    l = strlen(s) + 1;
  }
  return strcpy(Result, s);
}


Static void set_input_tech_tab(Char *intech)
{
  long i, j, FORLIM;

  i = 0;
  do {
    i++;
  } while (strcmp(tech_specs[i - 1].name, intech) && i <= num_techs);
  if (i > num_techs) {
    printf("sorry - no technology called %s, using %s\n",
	   intech, default_wol_tech);
    strcpy(input_technology, default_wol_tech);
    set_input_tech_tab(default_wol_tech);
    return;
  }
  FORLIM = tech_specs[i - 1].num_layers;
  for (j = 0; j < FORLIM; j++)
    strcpy(in_name[j], tech_specs[i - 1].layers[j]);
  num_layers = tech_specs[i - 1].num_layers;
}


Static void set_output_tech_tab(Char *outtech)
{
  long i, j, FORLIM;

  i = 0;
  do {
    i++;
  } while (strcmp(tech_specs[i - 1].name, outtech) && i <= num_techs);
  if (i > num_techs) {
    printf("sorry - no technology called %s, using %s\n",
	   outtech, default_out_tech);
    strcpy(technology_name, default_out_tech);
    set_output_tech_tab(default_out_tech);
  } else {
    FORLIM = tech_specs[i - 1].num_layers;
    for (j = 0; j < FORLIM; j++)
      strcpy(out_name[j], tech_specs[i - 1].layers[j]);
  }
  if (tech_specs[i - 1].num_layers == num_layers)
    return;
  printf("warning - input and output translation tables of different sizes\n");
  if (tech_specs[i - 1].num_layers > num_layers)
    printf("        - more output layers than input layers\n");
  else
    printf("        - more input layers than output layers\n");
  if (tech_specs[i - 1].num_layers < num_layers)
    num_layers = tech_specs[i - 1].num_layers;
}


Static void replace(Char *source, Char *target, Char *replace_with)
{
  long position;
  Char STR1[256];

  if (strpos2(source, target, 1) == 0)
    return;
  position = strpos2(source, target, 1);
  strcpy(source + position - 1, source + position + strlen(target) - 1);
  sprintf(STR1, "%s%s", replace_with, source + position - 1);
  strcpy(source + position - 1, STR1);
}


Static void fixup_layers(Char *source)
{
  long i;

  i = 0;
  do {
    i++;
  } while (strpos2(source, in_name[i - 1], 1) == 0 && i <= num_layers);
  if (i <= num_layers)
    replace(source, in_name[i - 1], out_name[i - 1]);
  else
    printf("warning: unknown layer %s\n", source);
}


/****************************************************************************/
/*    This procedure must validate the members of Carver's group, for
      accounting purposes.
*/

/* CUSTOM -- change the names in this procedure so only cool people
   can use the MOSIS account                                                */

Static boolean CarversGroup(Char *s)
{
  boolean tmp;

  tmp = (strcicmp(s, "CARVER") == 0) | (strcicmp(s, "MASS") == 0) |
      (strcicmp(s, "STEVE") == 0) | (strcicmp(s, "DICK") == 0) | (strcicmp(s,
	  "JIM") == 0) | (strcicmp(s, "SYLVIE") == 0) |
      (strcicmp(s, "LAZZARO") == 0) | (strcicmp(s, "MICHAEL") == 0) |
      (strcicmp(s, "MISHA") == 0) | (strcicmp(s, "MARYANN") == 0) |
      (strcicmp(s, "TOBI") == 0) | (strcicmp(s, "LEMON") == 0) |
      (strcicmp(s, "BENSON") == 0) | (strcicmp(s, "HARRIS") == 0) |
      (strcicmp(s, "JLUO") == 0) | (strcicmp(s, "DAKERNS") == 0) |
      (strcicmp(s, "ANDY") == 0) | (strcicmp(s, "GRACET") == 0) |
      (strcicmp(s, "LLOYD") == 0);
/* p2c: mosis.text, line 483: Note:
 * Line breaker spent 0.0+2.00 seconds, 5000 tries on line 560 [251] */
  /*
          OR     strequal(s,'TANNER')
          OR     strequal(s,'JOHNW')
          OR     strequal(s,'CHRIS')
  */
  return tmp;
}


Static void create_translated_file(void)
{
  FILE *outfile;
  Char fname[256];
  Char *TEMP;

  outfile = NULL;
  if (ciffile != NULL)
    ciffile = freopen(fullinfname, "r", ciffile);
  else
    ciffile = fopen(fullinfname, "r");
  if (ciffile == NULL)
    _EscIO(FileNotFound);
  if (outfile != NULL)
    outfile = freopen(tmpoutfname, "w", outfile);
  else
    outfile = fopen(tmpoutfname, "w");
  if (outfile == NULL)
    _EscIO(FileNotFound);

  found_errorboxes = false;
  inside_symbol_def = false;
  top_level_call = false;
/*  if (strcmp(technology_name, "NOCHANGE") && 
      strcmp(input_technology, technology_name)) { */
    if (1) {
    do {
      fgets(fname, 256, ciffile);
      TEMP = strchr(fname, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
      if (*fname != '\0') {
	if (!strcmp(fname, "LERR;")) {   /* do not pass any ERROR stuff */
	  do {
	    fgets(fname, 256, ciffile);
	    TEMP = strchr(fname, '\n');
	    if (TEMP != NULL)
	      *TEMP = 0;
	    found_errorboxes = true;
	  } while (fname[0] != 'L' || !strcmp(fname, "LERR;"));
	}
	if (fname[0] == 'L')
	  fixup_layers(fname);
	if (fname[0] == 'D') {
	  if (fname[1] == 'S')
	    inside_symbol_def = true;
	  else if (fname[1] == 'F')
	    inside_symbol_def = false;
	}
	if (fname[0] == 'C' && !inside_symbol_def)
	  top_level_call = true;
      }
      fprintf(outfile, "%s\n", fname);
    } while (!P_eof(ciffile));
  } else {  /* just copy verbatim */
    printf("No layer translation required. Output technology = %s\n",
	   technology_name);
    do {
      fgets(fname, 256, ciffile);
      TEMP = strchr(fname, '\n');
      if (TEMP != NULL)
	*TEMP = 0;
      if (!strcmp(fname, "LERR;")) {   /* do not pass any ERROR stuff */
	do {
	  fgets(fname, 256, ciffile);
	  TEMP = strchr(fname, '\n');
	  if (TEMP != NULL)
	    *TEMP = 0;
	  found_errorboxes = true;
	} while (fname[0] != 'L' || !strcmp(fname, "LERR;"));
      }
      if (fname[0] == 'D') {
	if (fname[1] == 'S')
	  inside_symbol_def = true;
	else if (fname[1] == 'F')
	  inside_symbol_def = false;
      }
      if (fname[0] == 'C' && !inside_symbol_def)
	top_level_call = true;
      fprintf(outfile, "%s\n", fname);
    } while (!P_eof(ciffile));
  }
  if (found_errorboxes)
    printf("WARNING:  boxes on ERROR layer (from DRC) deleted in .mosis file\n");
  if (!top_level_call)
    printf("ERROR:  no top level call.  MOSIS will not fabricate this!\n");
  if (ciffile != NULL)
    fclose(ciffile);
  ciffile = NULL;
  if (outfile != NULL)
    fclose(outfile);
  outfile = NULL;
  if (outfile != NULL)
    fclose(outfile);
}


Static void copy_translated_file(void)
{
  Char buffer[256];
  FILE *infile;
  Char *TEMP;

  infile = NULL;
  if (infile != NULL)
    infile = freopen(tmpoutfname, "r", infile);
  else
    infile = fopen(tmpoutfname, "r");
  if (infile == NULL)
    _EscIO(FileNotFound);
  do {
    fgets(buffer, 256, infile);
    TEMP = strchr(buffer, '\n');
    if (TEMP != NULL)
      *TEMP = 0;
    fprintf(outfile, "%s\n", buffer);
  } while (!P_eof(infile));
  if (infile != NULL)
    fclose(infile);
/* p2c: mosis.text, line 552:
 * Note: File is being closed with PURGE option [186] */
  infile = NULL;
  if (infile != NULL)
    fclose(infile);
}


main(int argc, Char *argv[])
{
  Char STR1[256], STR3[256];
  Char STR4[256];

  PASCAL_MAIN(argc, argv);
  ciffile = NULL;
  outfile = NULL;
  initialize_techs();
  TinyChip = false;

  /* set up command line switches */
  strcpy(parsetab[0].switch_, "mM");   /* Mailing address */
  parsetab[0].kind = 'S';
  strcpy(parsetab[1].switch_, "fF");   /* Full name */
  parsetab[1].kind = 'S';
  strcpy(parsetab[2].switch_, "nN");   /* project name */
  parsetab[2].kind = 'S';
  strcpy(parsetab[3].switch_, "dD");   /* 1 line description */
  parsetab[3].kind = 'S';
  strcpy(parsetab[4].switch_, "tT");   /* technology */
  parsetab[4].kind = 'S';
  strcpy(parsetab[5].switch_, "qQ");   /* quantity request */
  parsetab[5].kind = 'I';
  strcpy(parsetab[6].switch_, "bB");   /* bond same-as */
  parsetab[6].kind = 'S';
  strcpy(parsetab[7].switch_, "iI");   /* input technology */
  parsetab[7].kind = 'S';
  newci_parseswitch(parsetab, 8, infname);
  if (P_argc != 1) {
    if (*infname != '\0' || P_argc != 2 || !strcmp(infname, "?")) {
      fprintf(stderr,
	"Usage:  MOSIS -m<mail addr> -f<full name> -n<proj. name> -d<description>\n");
      fprintf(stderr,
	"              -t<%s> -i<%s> -q<quantity> -b<bond as> filename[.cif]\n",
	available_techs(STR1), available_techs(STR3));
      _Escape(0);
    }
    strcpy(infname, P_argv[1]);
  } else
    *infname = '\0';

  no_prompt = (P_argc != 1);


  printf("MOSIS Mail Generator.   Version %s.  Compiled: %s\n",
	 version, Compilation_date);
  if (*infname == '\0') {
    printf("\nInput (CIF) file name? ");
    gets(infname);
  }
  strcpy(fullinfname, infname);
  if (strpos2(fullinfname, ".", 1) == 0)
    strcat(fullinfname, ".cif");
  printf("   Reading from file: %s\n", fullinfname);
  if (ciffile != NULL)
    ciffile = freopen(fullinfname, "r", ciffile);
  else
    ciffile = fopen(fullinfname, "r");
  if (ciffile == NULL)
    _EscIO(FileNotFound);
  if (ciffile != NULL) {
    /* just test opening it */
    fclose(ciffile);
  }
  ciffile = NULL;

  sprintf(outfname, "%.*s.mosis",
	  strpos2(fullinfname, ".", 1) - 1, fullinfname);
  sprintf(tmpoutfname, "%.*s.motmp",
	  strpos2(fullinfname, ".", 1) - 1, fullinfname);
  if (!no_prompt) {
    printf("Output file name [%s] ? ", outfname);
    gets(fname);
    if (*fname != '\0')
      strcpy(outfname, fname);
  }
  if (strpos2(outfname, ".", 1) == 0)
    strcat(outfname, ".mosis");
  if (outfile != NULL)
    outfile = freopen(outfname, "w", outfile);
  else
    outfile = fopen(outfname, "w");
  if (outfile == NULL)
    _EscIO(FileNotFound);
  if (!no_prompt)
    printf("   Writing to file: %s\n", outfname);


  /* FULL NAME */
  if (parsetab[1].used == 0) {
    if (no_prompt)
      strcpy(full_name, P_citinfo->username);
    else {
      printf("Enter your full name[%s]: ", P_citinfo->username);
      gets(full_name);
      if (*full_name == '\0')
	strcpy(full_name, P_citinfo->username);
    }
  }
  else
    strcpy(full_name, parsetab[1].UU.s);

  strupper(full_name, full_name);   /* from NEWASM */

  fprintf(outfile, "REQUEST:           NEW-PROJECT\n");

/* CUSTOM -- here is the account info both for a research account and a
   class account. Good if you have NSF support for your VLSI course. */

  if (CarversGroup(full_name)) {
    fprintf(outfile, "  D-NAME:          BAR\n");
    fprintf(outfile, "  AFFILIATION:     WINTERPARK\n");
    fprintf(outfile, "  ACCOUNT:         SPAM-SPAM-SPAM\n");
    fprintf(outfile, "  D-PASSWORD:      FOO\n");
    fprintf(outfile, "  PHONE:           (000)000-0000\n");
  }
  else {
    printf("This chip is being submitted as a CLASS PROJECT CHIP\n");
    printf("If it is a research project chip, do not mail it to MOSIS!\n");
    fprintf(outfile, "	D-NAME:          FOO              \n");
    fprintf(outfile, "	ACCOUNT:         SPAM-SPAM-SPAM \n");
    fprintf(outfile, "	D-PASSWORD:      BAR            \n");
    fprintf(outfile, "	PHONE:           (666)666-6666        \n");
  }

  /* CUSTOM -- YOUR MAILING ADDRESS */
  if (parsetab[0].used == 0) {
    if (no_prompt) {
      /* arpa_addr := citinfo^.username+'@YOURMACHINE.LOCATION.EDU';
        if citinfo^.username = 'STAFF' then arpa_addr := ''; */ 
      *arpa_addr = '\0';   /* by default, no independent return mail */
    } else {
      printf("Enter your mailing address (login@arpa.site) or CR for none: ");
      gets(arpa_addr);
    }
  } else
    strcpy(arpa_addr, parsetab[0].UU.s);

  fprintf(outfile, "  NET-ADDRESS:     FOO@BAR.SPAM.EDU, FOO@BAR.SPAM.EDU");

  if (*arpa_addr != '\0')
    fprintf(outfile, ", %s\n", arpa_addr);
  else
    putc('\n', outfile);

  fprintf(outfile, "  MAILING-ADDRESS: SCHOOL FROM HELL\n");
  fprintf(outfile, "                   STAFFMAN / %s\n", full_name);
  fprintf(outfile, "                   FUNKY ADDRESS\n");
  fprintf(outfile, "                   ANYTOWN, ANYWHERE, 66666\n");
  fprintf(outfile, "                   \n");


  /* PROJECT NAME */
  if (!parsetab[2].used) {
    if (no_prompt)
      strcpy(proj_name, infname);
    else {
      printf("Enter your project name [%.*s]: ",
	     strpos2(fullinfname, ".", 1) - 1, fullinfname);
      gets(proj_name);
      if (*proj_name == '\0')
	sprintf(proj_name, "%.*s", strpos2(fullinfname, ".", 1) - 1, fullinfname);
    }
  } 
  else
    strcpy(proj_name, parsetab[2].UU.s);

  if (strpos2(full_name, " ", 1) != 0) {
    sprintf(STR4, "%.*s", strpos2(full_name, " ", 1) - 1, full_name);
    strcpy(full_name, STR4);
  }

  if (*full_name == '\0' && strcmp(P_citinfo->username, "UNKNOWN"))
    strcpy(full_name, P_citinfo->username);

  fprintf(outfile, "  P-NAME:          %s_%s\n", proj_name, full_name);
  fprintf(outfile, "  P-PASSWORD:      %s_%s\n", proj_name, full_name);

  /* ONE LINE DESCRIPTION */
  if (parsetab[3].used == 0) {
    if (no_prompt)
      sprintf(fname, "%s_%s", proj_name, full_name);
    else {
      printf("Enter a 1-line description of your project\n");
      gets(fname);
    }
  } else
    strcpy(fname, parsetab[3].UU.s);

  fprintf(outfile, "  DESCRIPTION:     %s\n", fname);

  /* TECHNOLOGY */
  if (parsetab[4].used == 0) {
    if (no_prompt)
      strcpy(technology_name, default_output);
    else {
      printf("Output technology (%s) or NOCHANGE [%s] ? ",
	     available_techs(STR1), default_output);
      gets(technology_name);
    }
  } else
    strcpy(technology_name, parsetab[4].UU.s);

  if (*technology_name == '\0')
    strcpy(technology_name, default_output);

  strupper(technology_name, technology_name);

  if (strpos2(technology_name, "TINY-", 1) == 1) {
    TinyChip = true;
    replace(technology_name, "TINY-", "");
  }

  /* if not in interactive mode and a layer conversion requested then
      assume input is default_wol_tech if no I switch*/

  if (!parsetab[7].used) {
    if (no_prompt)
      strcpy(input_technology, default_wol_tech);
    else {
      printf("Input Technology (%s) [%s] ? ",
	     available_techs(STR1), default_wol_tech);
      gets(input_technology);
    }
  }
  else
    strcpy(input_technology, parsetab[7].UU.s);

  if (*input_technology == '\0')
    strcpy(input_technology, default_wol_tech);

  strupper(input_technology, input_technology);

  if (strpos2(input_technology, "TINY-", 1) == 1) {
    printf("Please do not specify TINY- as an input technology!!!\n");
    replace(input_technology, "TINY-", "");
  }

  if (strcmp(technology_name, "NOCHANGE")) {
    set_input_tech_tab(input_technology);
    set_output_tech_tab(technology_name);
    printf("   converting technology from %s to %s\n", input_technology, technology_name);
  } 
  else
    printf("   no technology conversion begin performed -- technology is %s\n", input_technology);

  if (TinyChip)
    printf("   TinyChip has been requested\n");
  if (!strcmp(technology_name, "NOCHANGE"))
    fprintf(outfile, "  TECHNOLOGY:      %s\n", input_technology);
  else {
    if (TinyChip)
      fprintf(outfile, "  TECHNOLOGY:      TINY-%s\n", technology_name);
    else
      fprintf(outfile, "  TECHNOLOGY:      %s\n", technology_name);
  }


  /* QUANTITY REQUESTED */
  if (parsetab[5].used == 0) {
    if (!no_prompt) {
      printf("Quantity requested:  (<cr> for default) ");
      gets(quantity_request);
      if (*quantity_request != '\0')
	fprintf(outfile, " QUANTITY: %s\n", quantity_request);
      /*
                        writeln (outfile, '  SPECIAL-HANDLING:    Please send me ',
                                 quantity_request,' bonded parts.  Thank-you.');
      */
    }
  } else
    fprintf(outfile, " QUANTITY: %ld\n", parsetab[5].UU.i);
  /*
               writeln (outfile, '  SPECIAL-HANDLING:    Please send me ',
                        parsetab[6].i:0,' bonded parts.  Thank-you.');
  */
  /* BOND SAME AS */
  *bond_same_as = '\0';
  if (parsetab[6].used == 0) {
    if (no_prompt)
      *bond_same_as = '\0';
    else {
      printf("Bond same as:  (<cr> or <project-id> or <fab-id>) ");
      gets(bond_same_as);
    }
  } else
    strcpy(bond_same_as, parsetab[6].UU.s);

  get_size(fname);
  get_pads(&padno);

  fprintf(outfile, "  PADS:            %ld\n", padno);
  fprintf(outfile, "  LAMBDA:          %0.2f\n", lambda);
  fprintf(outfile, "REQUEST:           FABRICATE\n");
  fprintf(outfile, "  ID:              *\n");
  fprintf(outfile, "  P-PASSWORD:      %s_%s\n", proj_name, full_name);

  if (*bond_same_as != '\0')
    fprintf(outfile, "  BOND-SAME-AS:    %s\n", bond_same_as);

  fprintf(outfile, "  SIZE:            %s\n", fname);

  create_translated_file();

  putc('\n', outfile);
  write_CIF_checksum();
  fprintf(outfile, "\nLAYOUT-FORMAT:    CIF\n\n");
  fprintf(outfile, "\nLAYOUT:\n\n");

  copy_translated_file();

  fprintf(outfile, "\nREQUEST:  END\n\n");
  if (outfile != NULL)
    fclose(outfile);
  outfile = NULL;
  if (outfile != NULL)
    fclose(outfile);
  if (ciffile != NULL)
    fclose(ciffile);
  exit(0);
}

