/* FileDialogEntrySource.java
 *
 * created: Thu Jun  8 2000
 *
 * This file is part of Artemis
 *
 * Copyright (C) 2000  Genome Research Limited
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Header: /nfs/disk222/yeastpub/Repository/powmap/diana/components/FileDialogEntrySource.java,v 1.3 2000/06/30 13:49:18 kmr Exp $
 */

package diana.components;

import diana.*;
import diana.sequence.*;

import uk.ac.sanger.pathogens.*;
import uk.ac.sanger.pathogens.embl.EntryInformation;
import uk.ac.sanger.pathogens.embl.DocumentEntryFactory;
import uk.ac.sanger.pathogens.embl.SimpleEntryInformation;
import uk.ac.sanger.pathogens.embl.EntryInformationException;

import java.awt.*;
import java.io.*;

/**
 *  This is an EntrySource that reads Entry objects from the local filesystem.
 *
 *  @author Kim Rutherford <kmr@sanger.ac.uk>
 *  @version $Id: FileDialogEntrySource.java,v 1.3 2000/06/30 13:49:18 kmr Exp $
 **/

public class FileDialogEntrySource
    implements EntrySource {
  /**
   *  Create a new FileDialogEntrySource.
   *  @param frame The component that created this FileDialogEntrySource.
   *    (Used for requesters.)
   *  @param listener InputStreamProgressEvent objects will be sent to this
   *    listener as progress on reading is made.
   **/
  public FileDialogEntrySource (final Frame frame,
                                final InputStreamProgressListener listener) {
    this.frame = frame;
    this.stream_progress_listener = listener;
  }

  /**
   *  Get an Entry object from this source (by reading from a file).
   *  @param bases The Bases object to pass to the Entry constructor.
   *  @exception OutOfRangeException Thrown if one of the features in
   *    the Entry is out of range of the Bases object.
   *  @return null if and only if the user cancels the read.
   **/
  public Entry getEntry (final Bases bases)
      throws OutOfRangeException {
    try {
      return getEntryInternal (bases);
    } catch (NoSequenceException e) {
      throw new Error ("internal error - unexpected exception: " + e);
    }
  }

  /**
   *  Get an Entry object from this source (by reading from a file).
   *  @exception OutOfRangeException Thrown if one of the features in
   *    the Entry is out of range of the Bases object.
   *  @exception NoSequenceException Thrown if the entry that we read has no
   *    sequence.
   *  @return null if and only if the user cancels the read.
   **/
  public Entry getEntry ()
      throws OutOfRangeException, NoSequenceException {
    return getEntryInternal (null);
  }

  /**
   *  Returns true if and only if this EntrySource always returns "full"
   *  entries.  ie. entries that contain features and sequence.  Entries that
   *  are read from a file may contain just features so in this class this
   *  method returns false.
   **/
  public boolean isFullEntrySource () {
    return false;
  }
  

////  change ArtemisMain.readArgsAndOptions () to use this?:

//    /**
//     *  Get an Entry object from this source by name (by reading from a file).
//     *  @exception OutOfRangeException Thrown if one of the features in
//     *    embl_entry is out of range of the Bases object.
//     *  @exception NoSequenceException Thrown if the entry that we read has no
//     *    sequence.
//     *  @return null if and only if there is no Entry with that name.
//     **/
//    public Entry getEntryByName (final String entry_file_name)
//        throws OutOfRangeException, NoSequenceException, IOException {
//      final Document new_document =
//        new FileProgressDocument (new File (entry_file_name),
//                                  getInputStreamProgressListener ());

//      final EntryInformation new_entry_information =
//        new SimpleEntryInformation (Options.getArtemisEntryInformation ());

//      boolean seen_error = false;

//      while (true) {
//        try {
//          final uk.ac.sanger.pathogens.embl.Entry new_entry =
//            DocumentEntryFactory.makeDocumentEntry (new_entry_information,
//                                                    new_document);

//          return new Entry (new_entry);
//        } catch (EntryInformationException e) {

//          if (!seen_error) {
//            final String message =
//              "warning while reading " + entry_file_name + " - " +
//              e.getMessage ();

//            System.err.println (message);

//            new MessageDialog (frame, message);

//            seen_error = true;
//          }

//          EntryFileDialog.handleOpenException (new_entry_information, e);

//          // go around the loop again
//        }
//      }
//    }

  /**
   *  Make a new Entry.
   *  @param bases The Bases object to pass to the Entry constructor.  If null
   *    a new Bases object will be created.
   *  @exception OutOfRangeException Thrown if one of the features in
   *    the Entry is out of range of the Bases object.
   *  @exception NoSequenceException Thrown if bases is null and the Entry that
   *    we read has no sequence.
   **/
  public Entry makeEntry (final Bases bases,
                          final uk.ac.sanger.pathogens.embl.Entry embl_entry)
      throws OutOfRangeException, NoSequenceException {
    if (bases == null) {
      return new Entry (embl_entry);
    } else {
      return new Entry (bases, embl_entry);
    }
  }

  /**
   *  Return the InputStreamProgressListener that was passed to the
   *  constructor.
   **/
  public InputStreamProgressListener getInputStreamProgressListener () {
    return stream_progress_listener;
  }

  /**
   *  Return the name of this source (for display to the user in menus).
   **/
  public String getSourceName () {
    return "Filesystem";
  }

  /**
   *  Implementation of getEntry ().
   *  @param bases The Bases object to pass to the Entry constructor.
   *  @exception OutOfRangeException Thrown if one of the features in
   *    Entry is out of range of the Bases object.
   *  @exception NoSequenceException Thrown if bases is null and the entry that
   *    we read has no sequence.
   *  @return null if and only if the user cancels the read.
   **/
  private Entry getEntryInternal (final Bases bases)
      throws OutOfRangeException, NoSequenceException {
    final EntryInformation new_entry_information =
      new SimpleEntryInformation (Options.getArtemisEntryInformation ());

    final EntryFileDialog dialog = new EntryFileDialog (frame);

    uk.ac.sanger.pathogens.embl.Entry new_embl_entry =
      dialog.getEntry (new_entry_information, stream_progress_listener);

    if (new_embl_entry == null) {
      return null;
    }

    return makeEntry (bases, new_embl_entry);
  }

  /**
   *  The component that created this FileEntrySource.  (Used for requesters.)
   **/
  final Frame frame;

  /**
   *  InputStreamProgressEvents are sent to this object.
   **/
  private final InputStreamProgressListener stream_progress_listener;
}
