
MODULE Main;

(* The m3tohtmlf program receives a Modula-3 file name as command line
   argument and formats the named file in HTML to stdout, according to
   the typesetting rules below. If no argument is provided, the standard
   input is used.

   All lines up to and including the first blank line
   are skipped.  (This makes it easy to omit the copyright notice
   from the typeset version, for example.)

   The portions of the file that are not in comments will be typeset
   verbatim, in typewriter font, obeying line breaks and indentation.
   Any characters that have special significance to HTML will be quoted.

   The portions of the file that are in comments will be treated
   differently depending on the position of the comment.  There are 
   three cases:

   Comments that do not begin in the first column are typeset in a
   fixed-width font obeying line breaks and indentation, just like 
   program text. In the following example, OUT will be processed
   like the rest of the line:

|    PROCEDURE P(VAR (*OUT*) x: INTEGER);

   Comments that begin in the first column and are not preceded
   by a blank line are indented and typeset in slanted roman text,
   except that double-quoted expressions are typeset in slanted
   typewriter font.  The purpose of this is to allow a brief comment
   after a procedure declaration, indicating what the procedure does.
   For example, here is the specification of "Path.LineTo":

|    PROCEDURE LineTo(path: T; READONLY p: Point.T);
|    (* Extend "path" with a linear segment that starts
|       at its current point and ends at "p". *)

   It is recommended that these slanted comments be kept brief;
   additional details can be provided in longer type 3 comments. 

   Comments that begin in the first column and are preceded by a
   blank line are typeset in a roman font and justified.
   Items enclosed in double-quotes are set in unslanted 
   typewriter font; for example, program identifiers 
   should be double-quoted this way.  Line breaks and
   indentation have no effect in these comments.

   Sometimes you want a portion of a comment to be treated verbatim, like
   program text, for example, when including an example program or a
   table.  Therefore, any line that starts with "| " (vertical bar 
   followed by space) is treated like program text, typesetting it 
   verbatim in typewriter font.

   Comments of type 2 and 3 may contain HTML tags, for example 
   to annotate section headings or emphasize text.
   
   Lines in type-three comments that begin with "| " are treated just
   like program text with one exception: you can slip in words in roman
   font by enclosing them in back-quotes.
       
|    |   ELSIF s `was generated by` LineTo THEN

*)

IMPORT Wx, Buf, MarkUp;

IMPORT Stdio, Rd, Wr, Params, Text, Thread, ASCII;

TYPE
  FileType = {Interface, GenericInterface, Module, GenericModule, Unknown};

<*FATAL Thread.Alerted*>
<*FATAL Wr.Failure*>

PROCEDURE FindNameAndType(b: Buf.T; VAR name: TEXT; VAR type: FileType) =
  VAR
    position: CARDINAL := 0;
    length: CARDINAL := NUMBER(b^);
    t: TEXT;
    generic: BOOLEAN := FALSE;
  BEGIN
    t := GetName(b,position,length);
    IF Text.Equal(t,"GENERIC") THEN
      t := GetName(b,position,length);
      generic := TRUE;
    END;
    IF Text.Equal(t,"MODULE") THEN 
      IF generic THEN type := FileType.GenericModule; 
      ELSE type := FileType.Module;
      END;
    ELSIF Text.Equal(t,"INTERFACE") THEN
      IF generic THEN type := FileType.GenericInterface; 
      ELSE type := FileType.Interface;
      END;
    ELSE
      type := FileType.Unknown;
      RETURN;
    END;
    name := GetName(b,position,length);
  END FindNameAndType;

PROCEDURE GetName(b: Buf.T; VAR position: CARDINAL; length: CARDINAL): TEXT =
  VAR
    pos: CARDINAL;
  BEGIN
    SkipComments(b,position,length);
    pos := position;
    WHILE position < length AND b[position] IN ASCII.AlphaNumerics DO 
      INC(position);
    END;
    RETURN Text.FromChars(SUBARRAY(b^,pos,position - pos));
  END GetName;

PROCEDURE SkipComments(b: Buf.T; VAR position: CARDINAL; length: CARDINAL) =
  VAR
    inComment := FALSE;
  BEGIN
    WHILE position < (length - 1) DO
      IF inComment THEN
        IF b[position] = '*' AND b[position + 1] = ')' THEN
          INC(position);
          inComment := FALSE;
        END;
      ELSE
        IF b[position] = '(' AND b[position + 1] = '*' THEN
          INC(position);
          inComment := TRUE;
        ELSIF NOT b[position] IN ASCII.Spaces THEN RETURN;
        END;
      END;
      INC(position);
    END;
  END SkipComments;

CONST
  BufSize = 8000;

VAR
  buf: Buf.T;
  file, unitName: TEXT;
  wx := Wx.New();
  wr := Stdio.stdout;
  fileType: FileType;
  readBuffer: ARRAY [0..BufSize - 1] OF CHAR;
  nb: CARDINAL;

BEGIN
  TRY
    IF Params.Count < 2 THEN 
      LOOP
        nb := Rd.GetSub(Stdio.stdin, readBuffer);
        IF nb = 0 THEN EXIT; END;
        IF nb = BufSize THEN Wx.PutStr(wx,readBuffer); 
        ELSE Wx.PutStr(wx,SUBARRAY(readBuffer,0,nb));
        END;
      END;
      buf := Buf.FromText(Wx.ToText(wx));
    ELSE
      file := Params.Get(1);
      buf := Buf.FromFile (file);
      IF buf = NIL THEN RAISE Rd.Failure(NIL); END;
    END;

    FindNameAndType(buf, unitName, fileType);

    MarkUp.Annotate(buf,wx,FALSE);

    IF fileType = FileType.Interface OR 
        fileType = FileType.GenericInterface THEN
      Wr.PutText(wr,"<TITLE> The " & unitName & " interface </TITLE>\n");
      Wr.PutText(wr,"<H1> The " & unitName & " interface </H1>\n");
    ELSIF fileType = FileType.Module OR fileType = FileType.GenericModule THEN
      Wr.PutText(wr,"<TITLE> The " & unitName & " module </TITLE>\n");
      Wr.PutText(wr,"<H1> The " & unitName & " module </H1>\n");
    END;

    Wr.PutText(wr,Wx.ToText(wx));
  EXCEPT
  | Rd.Failure => 
      Wr.PutText(Stdio.stderr, "? can't read file " & file & " \n");
  | Wr.Failure =>
      Wr.PutText(Stdio.stderr, "? can't write file\n");
  END
END Main.
