/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "ui_local.h"

//=============================================================================
//
// TEAM CONFIG MENU
//
//=============================================================================

static menuframework_s	s_team_config_menu;

static int currentTEAM = TEAM_RED;
menucommon_t *hasForcedModelMenuItem;
menucommon_t *hasForcedColorMenuItem;


cvar_t *color;
cvar_t *model;
cvar_t *skin;

int ForceTeamNumber( char *cvarname ) {
	if( !trap_Cvar_VariableString( cvarname ) )
		return 0;

	if( !Q_stricmp( trap_Cvar_VariableString(cvarname), "players" ) )
		return TEAM_PLAYERS;

	if( !Q_stricmp( trap_Cvar_VariableString(cvarname), "RED" ) )
		return TEAM_RED;

	if( !Q_stricmp( trap_Cvar_VariableString(cvarname), "BLUE" ) )
		return TEAM_BLUE;

	if( !Q_stricmp( trap_Cvar_VariableString(cvarname), "GREEN" ) )
		return TEAM_GREEN;
	
	if( !Q_stricmp( trap_Cvar_VariableString(cvarname), "YELLOW" ) )
		return TEAM_YELLOW;

	return 0;
}



void UpdateTeamCvars( void ) {
	switch( currentTEAM ) {
		case TEAM_RED:
			model = trap_Cvar_Get( "cg_teamREDmodel", "", CVAR_ARCHIVE|CVAR_USERINFO );
			skin = trap_Cvar_Get( "cg_teamREDskin", "", CVAR_ARCHIVE|CVAR_USERINFO );
			color = trap_Cvar_Get( "cg_teamREDcolor", "", CVAR_ARCHIVE|CVAR_USERINFO );
			break;
		case TEAM_BLUE:
			model = trap_Cvar_Get( "cg_teamBLUEmodel", "", CVAR_ARCHIVE|CVAR_USERINFO );
			skin = trap_Cvar_Get( "cg_teamBLUEskin", "", CVAR_ARCHIVE|CVAR_USERINFO );
			color = trap_Cvar_Get( "cg_teamBLUEcolor", "", CVAR_ARCHIVE|CVAR_USERINFO );
			break;
		case TEAM_GREEN:
			model = trap_Cvar_Get( "cg_teamGREENmodel", "", CVAR_ARCHIVE|CVAR_USERINFO );
			skin = trap_Cvar_Get( "cg_teamGREENskin", "", CVAR_ARCHIVE|CVAR_USERINFO );
			color = trap_Cvar_Get( "cg_teamGREENcolor", "", CVAR_ARCHIVE|CVAR_USERINFO );
			break;
		case TEAM_YELLOW:
			model = trap_Cvar_Get( "cg_teamYELLOWmodel", "", CVAR_ARCHIVE|CVAR_USERINFO );
			skin = trap_Cvar_Get( "cg_teamYELLOWskin", "", CVAR_ARCHIVE|CVAR_USERINFO );
			color = trap_Cvar_Get( "cg_teamYELLOWcolor", "", CVAR_ARCHIVE|CVAR_USERINFO );
			break;
		case TEAM_PLAYERS:
		default:
			model = trap_Cvar_Get( "cg_teamPLAYERSmodel", "", CVAR_ARCHIVE|CVAR_USERINFO );
			skin = trap_Cvar_Get( "cg_teamPLAYERSskin", "", CVAR_ARCHIVE|CVAR_USERINFO );
			color = trap_Cvar_Get( "cg_teamPLAYERScolor", "", CVAR_ARCHIVE|CVAR_USERINFO );
			break;
	}
}



//=================
//M_TeamConfig_ApplyChanges
//=================
void M_TeamConfig_ApplyChanges( void ) {
	menucommon_t *playerbox = UI_MenuItemByName( "m_TeamConfig_model" );
	menucommon_t *menuitem = UI_MenuItemByName( "m_TeamConfig_enemyteam" );

	UpdateTeamCvars();
	if( hasForcedModelMenuItem->curvalue ) {
		trap_Cvar_Set( model->name, playerbox->itemnames[playerbox->curvalue] );
		trap_Cvar_Set( skin->name, "default" );
	} else {
		trap_Cvar_Set( model->name, "" );
		trap_Cvar_Set( skin->name, "" );
	}

	if( hasForcedColorMenuItem->curvalue ) {
		trap_Cvar_Set( color->name, va( "%i %i %i", (int)playerColor[0], (int)playerColor[1], (int)playerColor[2] ) );
	} else {
		trap_Cvar_Set( color->name, "" );
	}

	if( menuitem ) {
		switch( menuitem->curvalue ) {
			case TEAM_PLAYERS:
				trap_Cvar_Set( "cg_forceEnemyTeam", "PLAYERS" );
				break;
			case TEAM_RED:
				trap_Cvar_Set( "cg_forceEnemyTeam", "RED" );
				break;
			case TEAM_BLUE:
				trap_Cvar_Set( "cg_forceEnemyTeam", "BLUE" );
				break;
			case TEAM_GREEN:
				trap_Cvar_Set( "cg_forceEnemyTeam", "GREEN" );
				break;
			case TEAM_YELLOW:
				trap_Cvar_Set( "cg_forceEnemyTeam", "YELLOW" );
				break;
			case TEAM_SPECTATOR:
			default:
				trap_Cvar_Set( "cg_forceEnemyTeam", "" );
				break;
		}
	}

	menuitem = UI_MenuItemByName( "m_TeamConfig_myteam" );
	if( menuitem ) {
		switch( menuitem->curvalue ) {
			case TEAM_PLAYERS:
				trap_Cvar_Set( "cg_forceMyTeam", "PLAYERS" );
				break;
			case TEAM_RED:
				trap_Cvar_Set( "cg_forceMyTeam", "RED" );
				break;
			case TEAM_BLUE:
				trap_Cvar_Set( "cg_forceMyTeam", "BLUE" );
				break;
			case TEAM_GREEN:
				trap_Cvar_Set( "cg_forceMyTeam", "GREEN" );
				break;
			case TEAM_YELLOW:
				trap_Cvar_Set( "cg_forceMyTeam", "YELLOW" );
				break;
			case TEAM_SPECTATOR:
			default:
				trap_Cvar_Set( "cg_forceMyTeam", "" );
				break;
		}
	}
}

void M_GetTeamColor( void ) {
	int rgbcolor;
	menucommon_t *menuitem;
	
	rgbcolor = COM_ReadColorRGBString( color->string );
	if( rgbcolor == -1 ) {
		rgbcolor = COM_ReadColorRGBString( color->dvalue );
	}
	if( rgbcolor != -1 ) {
		Vector4Set( playerColor, COLOR_R(rgbcolor), COLOR_G(rgbcolor), COLOR_B(rgbcolor), 255 ); 
	} else {
		Vector4Set( playerColor, 255, 255, 255, 255 ); // start white
	}
	color->modified = qfalse;

	// update the bars
	menuitem = UI_MenuItemByName( "m_TeamConfig_colorred" );
	menuitem->curvalue = playerColor[0];
	menuitem = UI_MenuItemByName( "m_TeamConfig_colorgreen" );
	menuitem->curvalue = playerColor[1];
	menuitem = UI_MenuItemByName( "m_TeamConfig_colorblue" );
	menuitem->curvalue = playerColor[2];
}



qboolean M_TeamHasModel( void ) {
	UpdateTeamCvars();

	if( model->string[0] )
		return qtrue;

	return qfalse;
}

void M_GetTeamModel( void ) {
	menucommon_t *menuitem;
	int currentdirectoryindex = 0;
	int currentskinindex = 0;
	if( currentTEAM < TEAM_PLAYERS )
		currentTEAM = TEAM_PLAYERS;
	if( currentTEAM >= GS_MAX_TEAMS )
		currentTEAM = GS_MAX_TEAMS - 1;

	model->modified = qfalse;

	// find the skin and model matching our user settings (if any)
	UI_FindIndexForModelAndSkin( model->string, skin->string, &currentdirectoryindex, &currentskinindex );
	menuitem = UI_MenuItemByName( "m_TeamConfig_model" );
	menuitem->curvalue = currentdirectoryindex;
}





static void ForceAModelCallback( menucommon_t *menuitem ) {
	UpdateTeamCvars();
	if( model->string[0] ) { // had it
		if( !menuitem->curvalue )
			trap_Cvar_Set( model->name, "" );
		
	} else { // didn't have it
		if( menuitem->curvalue ) {
			trap_Cvar_Set( model->name, "default" );
		}
	}
}

static void ForceAColorCallback( menucommon_t *menuitem ) {
	UpdateTeamCvars();
	if( color->string[0] ) { // had it
		if( !menuitem->curvalue )
			trap_Cvar_Set( color->name, "" );

	} else { // didn't have it
		if( menuitem->curvalue ) {
			trap_Cvar_Set( color->name, va( "%i %i %i", (int)playerColor[0], (int)playerColor[1], (int)playerColor[2] ) );
		}
	}
}

static void ChangeTeamCallback( menucommon_t *menuitem ) {

	M_TeamConfig_ApplyChanges();
	currentTEAM = menuitem->curvalue + TEAM_PLAYERS;
	UpdateTeamCvars();

	// see if this team has a forced model set up
	hasForcedModelMenuItem->curvalue = ( model->string[0] != 0 );
	hasForcedColorMenuItem->curvalue = ( color->string[0] != 0 );

	M_GetTeamColor();
	M_GetTeamModel();
}

static void M_TeamConfig_Close( menucommon_t *menuitem ) {
	M_genericBackFunc( menuitem );
}

static void M_TeamConfig_SaveAndClose( menucommon_t *menuitem ) {
	M_TeamConfig_ApplyChanges();
	M_TeamConfig_Close( menuitem );
}

static void TeamModelCallback( menucommon_t *menuitem )
{
	// todo, update the skin itemnames list
}

//=================
//TeamConfig_MenuInit
//=================
qboolean TeamConfig_MenuInit( void )
{
	menucommon_t	*menuitem;
	int currentdirectoryindex = 0;
	int x = -150;
	int y = -100;
	int yoffset = 0;

	static char *force_team_names[] =
	{
		"with their real aspect",
			"as if they were the PLAYERS team",
			"as if they were the RED team",
			"as if they were the BLUE team",
			"as if they were the GREEN team",
			"as if they were the YELLOW team",
			0
	};

	static char *team_names[] =
	{
		"PLAYERS",
			"RED",
			"BLUE",
			"GREEN",
			"YELLOW",
			0
	};

	if( playermodelsItemsList.numItems == 0 )
		return qfalse;

	s_team_config_menu.x = uis.vidWidth / 2; 
	s_team_config_menu.y = uis.vidHeight / 2;
	s_team_config_menu.nitems = 0;

	currentTEAM = TEAM_RED;
	UpdateTeamCvars();

	// team
	menuitem = UI_InitMenuItem( "m_TeamConfig_team", "Team", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, ChangeTeamCallback );
	UI_SetupSpinControl( menuitem, team_names, TEAM_RED - TEAM_PLAYERS );
	Menu_AddItem( &s_team_config_menu, menuitem );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_forcemodel", "Force a team model", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, ForceAModelCallback );
	UI_SetupSpinControl( menuitem, noyes_names, ( model->string[0] != 0 ) );
	Menu_AddItem( &s_team_config_menu, menuitem );
	hasForcedModelMenuItem = menuitem;

	yoffset += trap_SCR_strHeight( menuitem->font );

	// model
	menuitem = UI_InitMenuItem( "m_TeamConfig_model", "model", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, TeamModelCallback );
	UI_SetupSpinControl( menuitem, playermodelsItemsList.item_names, currentdirectoryindex );
	Menu_AddItem( &s_team_config_menu, menuitem );

	yoffset += trap_SCR_strHeight( menuitem->font );

	M_GetTeamModel();

	menuitem = UI_InitMenuItem( "m_TeamConfig_forcecolor", "Force a team color", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, ForceAColorCallback );
	UI_SetupSpinControl( menuitem, noyes_names, ( color->string[0] != 0 ) );
	Menu_AddItem( &s_team_config_menu, menuitem );
	hasForcedColorMenuItem = menuitem;

	yoffset += trap_SCR_strHeight( menuitem->font );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_colorred", "red", x, y+yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, UI_ColorRedCallback );
	Menu_AddItem( &s_team_config_menu, menuitem );
	UI_SetupSlider( menuitem, 12, playerColor[0], 0, 255 );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_colorgreen", "green", x, y+yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, UI_ColorGreenCallback );
	Menu_AddItem( &s_team_config_menu, menuitem );
	UI_SetupSlider( menuitem, 12, playerColor[1], 0, 255 );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_colorblue", "blue", x, y+yoffset, MTYPE_SLIDER, ALIGN_RIGHT_TOP, uis.fontSystemSmall, UI_ColorBlueCallback );
	Menu_AddItem( &s_team_config_menu, menuitem );
	UI_SetupSlider( menuitem, 12, playerColor[2], 0, 255 );

	M_GetTeamColor();

	yoffset += trap_SCR_strHeight( menuitem->font );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_enemyteam", "Show my enemies", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, M_TeamConfig_ApplyChanges );
	UI_SetupSpinControl( menuitem, force_team_names, ForceTeamNumber("cg_forceEnemyTeam") );
	Menu_AddItem( &s_team_config_menu, menuitem );

	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_myteam", "Show my teammates", x, y+yoffset, MTYPE_SPINCONTROL, ALIGN_RIGHT_TOP, uis.fontSystemSmall, M_TeamConfig_ApplyChanges );
	UI_SetupSpinControl( menuitem, force_team_names, ForceTeamNumber("cg_forceMyTeam") );
	Menu_AddItem( &s_team_config_menu, menuitem );

	yoffset += trap_SCR_strHeight( menuitem->font );
	yoffset += trap_SCR_strHeight( menuitem->font );

	menuitem = UI_InitMenuItem( "m_TeamConfig_back", "back", x - 16, y+yoffset, MTYPE_ACTION, ALIGN_RIGHT_TOP, uis.fontSystemBig, M_TeamConfig_SaveAndClose );
	Menu_AddItem( &s_team_config_menu, menuitem );

	Menu_Init( &s_team_config_menu );

	return qtrue;
}

//=================
//TeamConfig_MenuDraw - skelmod
//=================
void TeamConfig_MenuDraw( void )
{
	menucommon_t *playerbox;
	m_listitem_t *item;
	playermodelinfo_s *playermodel;
	int x, y, width, height, i;
	vec4_t tmpcolor;

	Menu_Draw( &s_team_config_menu ); // draw menu on back

	// go for the model

	// if the color cvar is modified from outside of the ui (console), update the color vector
	if( color && color->modified ) {
		M_GetTeamColor();
	}

	x = (uis.vidWidth / 2)+64;
	y = (uis.vidHeight/2 - 128);//128;
	width = 256;
	height = 256;

	// now on the model.
	// First of all, see if we have to draw any model
	if( M_TeamHasModel() ) {
		if( model && model->modified ) {
			M_GetTeamModel();
		}

		playerbox = UI_MenuItemByName( "m_TeamConfig_model" );
		if( playerbox && model->string[0] ) {
			item = UI_FindItemInScrollListWithId( &playermodelsItemsList, playerbox->curvalue ); 
			if( item && item->data ) {
				playermodel = (playermodelinfo_s*)item->data;

				// draw player model
				
				UI_DrawPlayerModel( playermodel->directory, "default", playerColor, x, y, width, height );
			}
		}

	} else {
		if( hasForcedColorMenuItem->curvalue )
		{ 
			// draw colored square
			for( i = 0; i<3; i++ )
				tmpcolor[i] = playerColor[i] * (1.0/255.0);
			tmpcolor[3] = 1;
			trap_R_DrawStretchPic( x+64, y+64, width-128, height-128, 0, 0, 1, 1, tmpcolor, uis.whiteShader );
		}
	}
}


const char *TeamConfig_MenuKey( int key )
{
	menucommon_t *item;

	item = Menu_ItemAtCursor( &s_team_config_menu );

	if ( key == K_ESCAPE || ( (key == K_MOUSE2) && (item->type != MTYPE_SPINCONTROL) &&
		(item->type != MTYPE_SLIDER)) )
	{
		M_TeamConfig_SaveAndClose( NULL );
		return menu_out_sound;
	}

	return Default_MenuKey( &s_team_config_menu, key );
}

const char *TeamConfig_MenuCharEvent( int key )
{
	return Default_MenuCharEvent( &s_team_config_menu, key );
}

void M_Menu_TeamConfig_f( void )
{
	if( !TeamConfig_MenuInit() )
	{
		Menu_SetStatusBar( &s_team_config_menu, "No valid player models found" );
		return;
	}

	Menu_SetStatusBar( &s_team_config_menu, NULL );
	M_PushMenu( &s_team_config_menu, TeamConfig_MenuDraw, TeamConfig_MenuKey, TeamConfig_MenuCharEvent );
}
