//--------------------------------------------------------------------------------
//
//	File:		AppleUSBAudioEngine.h
//
//	Contains:	Support for the USB Audio Class Stream Interface.
//
//	Technology:	Mac OS X
//
//--------------------------------------------------------------------------------

#ifndef _APPLEUSBAUDIOENGINE_H
#define _APPLEUSBAUDIOENGINE_H

#include <IOKit/IOBufferMemoryDescriptor.h>
#include <IOKit/audio/IOAudioEngine.h>
#include <IOKit/audio/IOAudioStream.h>
#include <IOKit/usb/IOUSBPipe.h>
#include <IOKit/usb/IOUSBDevice.h>
#include <IOKit/usb/IOUSBInterface.h>

#include "USBAudioObject.h"
#include "AppleiSubEngine.h"
#include "AppleUSBAudioClip.h"

class AppleUSBAudioDevice;

//	-----------------------------------------------------------------

#define	kSampleRate_44100				44100
#define kDefaultSamplingRate			kSampleRate_44100
#define	kBitDepth_16bits				16
#define	kChannelDepth_MONO				1
#define	kChannelDepth_STEREO			2

#define kMinimumFrameOffset				1

// Don't make NUM_FRAME_LISTS * NUM_FRAMES_PER_LIST smaller than 375ms or else the HAL will have problems
#define NUM_FRAME_LISTS					80
#if FIXEDSIZEPACKETS
#define NUM_FRAMES_PER_LIST				5
#else
#define NUM_FRAMES_PER_LIST				64
#endif
#define NUM_FRAME_LISTS_TO_QUEUE		10

#define MAX_WRITE_RETRIES				10

class IOSyncer;
class AppleUSBAudioEngine;

typedef struct FrameListWriteInfo {
    AppleUSBAudioEngine *				audioEngine;
    UInt32								frameListNum;
    UInt32								retryCount;
} FrameListWriteInfo;

class AppleUSBAudioEngine : public IOAudioEngine {
    friend class AppleUSBAudioDevice;

    OSDeclareDefaultStructors (AppleUSBAudioEngine);

public:
    virtual bool init (OSDictionary *properties);
    virtual void free ();
    virtual bool initHardware (IOService *provider);
    virtual bool start (IOService *provider);
    virtual void stop (IOService *provider);
#if DOUBLEPARENT
	virtual bool requestTerminate (IOService * provider, IOOptionBits options);
#endif
    virtual bool terminate (IOOptionBits options = 0);
    virtual IOReturn message (UInt32 type, IOService * provider, void * arg);

    virtual IOReturn performAudioEngineStart ();
    virtual IOReturn performAudioEngineStop ();
    
    virtual IOAudioEngineState _setState (IOAudioEngineState newState);

    static void readHandler (AppleUSBAudioEngine * self, void * frameListIndex, IOReturn result, IOUSBIsocFrame * pFrames);
	static void sampleRateHandler (void * target, void * parameter, IOReturn result, IOUSBIsocFrame * pFrames);
    static void writeHandler (AppleUSBAudioEngine * self, void * parameter, IOReturn result, IOUSBIsocFrame * pFrames);

protected:
	UInt64					theFirstFrame;
	UInt64					nextSynchReadFrame;
	IOUSBIsocFrame *		theFrames;
	IOUSBIsocFrame			theSampleRateFrame;
	IOUSBIsocCompletion *	usbCompletion;
	IOUSBIsocCompletion		sampleRateCompletion;
	IOSyncer *				signal;
	AppleUSBAudioDevice *	usbAudioDevice;
	IOUSBInterface *		streamInterface;
	IOUSBInterface *		previousInterface;
	IOUSBPipe *				thePipe;
	IOUSBPipe *				theAssociatedPipe;
	IOMemoryDescriptor **	soundBuffer;
	IOMemoryDescriptor *	neededSampleRate;
	FrameListWriteInfo *	pendingFrameListWriteInfo;
#if RETRY_WRITES
	IOSyncer *				retrySync;
#endif
	IONotifier *			streamNotifier;
	OSObject *				interfaceVendor;
	OSObject *				interfaceProduct;
	OSObject *				deviceReleaseNumber;
	OSObject *				configurationValue;
	OSObject *				interfaceNumber;
	void *					readBuffer;
#if TIMERSTREAM
	IOThread				timerStreamThread;
#endif
	thread_call_t			pendingFrameListCall;
	IOAudioSampleRate		curSampleRate;
	UInt32					previous;
	UInt32					previousClippedToFrame;
	UInt32					theMaxPacket;
	UInt32					frameListSize;
	UInt32					readFrameListSize;
	UInt32					bufferOffset;
	UInt32					aveSampleRateBuf;		// only needs to be 3 bytes for the 10.14 value, but this is more convenient
	UInt32					averageSampleRate;
	UInt32					sum;					// This is the number of samples we have sent in the previous frames in this period
	UInt32					sampleNum;				// The number of the frame that we are currently sending, it goes between 1 and 1000
	UInt32					numFrameLists;
	UInt32					numFramesPerList;
	UInt32					numFrameListsToQueue;
	UInt32					currentFrameList;
	UInt32					bufferSize;
	UInt32					bytesPerSampleFrame;
	UInt16					samplesThisFrame;
	UInt8					ourInterfaceNumber;
	UInt8					shouldStop;
	UInt8					alternateInterfaceID;
	UInt8					refreshInterval;
	UInt8					framesUntilRefresh;
	UInt8					direction;
	Boolean					usbStreamRunning;
	Boolean					notifyDeviceOfStop;
#if TIMERSTREAM
	Boolean					didStop;
#endif
	Boolean					startingEngine;
	Boolean					startiSub;
	IOAudioStream *			mainStream;

	// For iSub
	IOMemoryDescriptor *	iSubBufferMemory;
	UInt32					iSubLoopCount;
	SInt32					iSubBufferOffset;		// Where we are writing to in the iSub buffer
	IOService *				ourProvider;
	IONotifier *			iSubEngineNotifier;
	AppleiSubEngine *		iSubEngine;
	AppleiSubEngine *		oldiSubEngine;
	float *					lowFreqSamples;
	float *					highFreqSamples;
	PreviousValues			filterState;
	Boolean					needToSync;

	static inline IOFixed IOUFixedDivide(UInt32 a, UInt32 b)
	{
		return (IOFixed)((((UInt64) a) << 16) / ((UInt64) b));
	}

	static inline UInt32 IOUFixedMultiply(UInt32 a, UInt32 b)
	{
		return (UInt32)((((UInt64) a) * ((UInt64) b)) >> 16);
	}

	void	GetDeviceInfo (void);
	IOReturn	PrepareFrameLists (UInt32 numFrameLists, UInt8 sampleSize, UInt8 numChannels);
	IOReturn	SetSampleRate (USBAudioConfigObject *usbAudio, UInt32 sampleRate);
	IOReturn	AddAvailableFormatsFromDevice (USBAudioConfigObject *usbAudio);
	UInt32		CalculateNumSamplesPerFrameList (UInt32 sampleRate, UInt32 numFrames, UInt32 theNumFrameLists = 1);
	IOReturn	CheckForAssociatedEndpoint (USBAudioConfigObject *usbAudio);

    static bool audioDevicePublished (AppleUSBAudioEngine *audioEngine, void *ref, IOService *newService);
    static bool	iSubEnginePublished (AppleUSBAudioEngine * usbAudioEngineObject, void * refCon, IOService * newService);

	virtual OSString * getGlobalUniqueID ();
    virtual IOReturn readFrameList (UInt32 frameListNum);
    virtual IOReturn writeFrameList (UInt32 frameListNum);

    virtual IOReturn startUSBStream ();
    virtual IOReturn stopUSBStream ();

#if TIMERSTREAM
    virtual void startTimerStream ();
    virtual void stopTimerStream ();

    static IOReturn startTimerStreamIfNecessaryAction (OSObject *owner, void *arg1, void *arg2, void *arg3, void *arg4);
    virtual bool startTimerStreamIfNecessary ();
#endif

    static void retryWriteFrameList (void *arg);

	static IOReturn iSubCloseAction (OSObject *owner, void *arg1, void *arg2, void *arg3, void *arg4);
	static IOReturn iSubOpenAction (OSObject *owner, void *arg1, void *arg2, void *arg3, void *arg4);
	static void iSubTeardown (AppleUSBAudioEngine *usbAudioEngine, thread_call_t iSubTeardownThreadCall);
	virtual void iSubTeardownConnection (void);

#if TIMERSTREAM
    virtual bool isUnattachedStreamRegistered (void);
    virtual void registerUnattachedStream (void);
    virtual void unregisterUnattachedStream (void);
    static bool interfacePublished (AppleUSBAudioEngine *audioEngine, void *ref, IOService *newService);

    static IOReturn reinitWithInterfaceAction (OSObject *owner, void *arg1, void *arg2, void *arg3, void *arg4);
    virtual bool reinitWithInterface (IOUSBInterface *interface);

    static void timerStreamMain (void *audioEngine);
#endif

    virtual UInt32 getCurrentSampleFrame (void);

    virtual IOAudioStreamDirection getDirection (void);
    virtual void *getSampleBuffer (void);
	virtual UInt32 getSampleBufferSize (void);

	virtual void resetClipPosition (IOAudioStream *audioStream, UInt32 clipSampleFrame);
    virtual IOReturn clipOutputSamples (const void *mixBuf, void *sampleBuf, UInt32 firstSampleFrame, UInt32 numSampleFrames, const IOAudioStreamFormat *streamFormat, IOAudioStream *audioStream);
	virtual IOReturn convertInputSamples (const void *sampleBuf, void *destBuf, UInt32 firstSampleFrame, UInt32 numSampleFrames, const IOAudioStreamFormat *streamFormat, IOAudioStream *audioStream);

    virtual IOReturn performFormatChange (IOAudioStream *audioStream, const IOAudioStreamFormat *newFormat, const IOAudioSampleRate *newSampleRate);
	void CalculateSamplesPerFrame (UInt32 sampleRate, UInt16 * averageFrameSize, UInt16 * additionalSampleFrameFreq);
};

#endif /* _APPLEUSBAUDIOENGINE_H */
