//--------------------------------------------------------------------------------
//
//	File:		AppleUSBAudioDevice.cpp
//
//	Contains:	Support for the USB Audio Class Control Interface.
//			This includes support for exporting device controls
//			to the Audio HAL such as Volume, Bass, Treble and
//			Mute.
//
//			Future support will include parsing of the device
//			topology and exporting of all appropriate device
//			control functions through the Audio HAL.
//
//	Technology:	OS X
//
//--------------------------------------------------------------------------------

#include "AppleUSBAudioDevice.h"

#define super IOAudioDevice

OSDefineMetaClassAndStructors (AppleUSBAudioDevice, super)

bool AppleUSBAudioDevice::init (OSDictionary * properties)  {
	bool		resultCode;

	debugIOLog ("+AppleUSBAudioDevice[%p]::init ()", this);
	resultCode = FALSE;		// Assume failure

	// get the IOAudioDevice generic initialization
	FailIf (FALSE == super::init (properties), Exit);

	resultCode = TRUE;

Exit:
	debugIOLog ("-AppleUSBAudioDevice[%p]::init ()", this);
	return resultCode;
}

void AppleUSBAudioDevice::free () {
    debugIOLog ("+AppleUSBAudioDevice[%p]::free ()", this);

    if (interfaceLock) {
        IORecursiveLockFree (interfaceLock);
        interfaceLock = NULL;
    }

    if (usbAudio) {
        usbAudio->release ();
        usbAudio = NULL;
    }

    super::free ();
    debugIOLog ("-AppleUSBAudioDevice[%p]::free ()", this);
}

bool AppleUSBAudioDevice::requestTerminate (IOService * provider, IOOptionBits options) {
	debugIOLog ("+AppleUSBAudioDevice[%p]::requestTerminate (%p, %x)", this, provider, options);
	debugIOLog ("-AppleUSBAudioDevice[%p]::requestTerminate (%p, %x)", this, provider, options);
	return TRUE;		// it is OK to terminate us
}

bool AppleUSBAudioDevice::ControlsStreamNumber (UInt8 streamNumber) {
	UInt8 *							streamNumbers;
	UInt8							numStreams;
	UInt8							index;
	bool							doesControl;

	doesControl = FALSE;

	if (usbAudio) {
		usbAudio->GetControlledStreamNumbers (&streamNumbers, &numStreams);
		for (index = 0; index < numStreams; index++) {
			debugIOLog ("Checking stream %d against controled stream %d", streamNumber, streamNumbers[index]);
			if (streamNumber == streamNumbers[index]) {
				doesControl = TRUE;
				break;				// get out of for loop
			}
		}
	}

	return doesControl;
}

bool AppleUSBAudioDevice::start (IOService * provider) {
	bool								result;

    debugIOLog ("+ AppleUSBAudioDevice[%p]::start (%p)", this, provider);
	result = FALSE;

	controlInterface = OSDynamicCast (IOUSBInterface, provider);
	FailIf (FALSE == controlInterface->open (this), Exit);

	mInitHardwareThread = thread_call_allocate ((thread_call_func_t)AppleUSBAudioDevice::initHardwareThread, (thread_call_param_t)this);
	FailIf (NULL == mInitHardwareThread, Exit);

    debugIOLog ("- AppleUSBAudioDevice[%p]::start (%p)", this, provider);

	result = super::start (provider);				// Causes our initHardware routine to be called.

Exit:
	return result;
}

bool AppleUSBAudioDevice::initHardware (IOService * provider) {
	bool								result;

	result = FALSE;

	FailIf (NULL == mInitHardwareThread, Exit);
	thread_call_enter1 (mInitHardwareThread, (void *)provider);

	result = TRUE;

Exit:
	return result;
}

void AppleUSBAudioDevice::initHardwareThread (AppleUSBAudioDevice * aua, void * provider) {
	IOCommandGate *						cg;
	IOReturn							result;

	FailIf (NULL == aua, Exit);
//	FailIf (TRUE == aua->mTerminating, Exit);	

	cg = aua->getCommandGate ();
	if (cg) {
		result = cg->runAction (aua->initHardwareThreadAction, provider);
	}

Exit:
	return;
}

IOReturn AppleUSBAudioDevice::initHardwareThreadAction (OSObject * owner, void * provider, void * arg2, void * arg3, void * arg4) {
	AppleUSBAudioDevice *				aua;
	IOReturn							result;

	result = kIOReturnError;

	aua = (AppleUSBAudioDevice *)owner;
	FailIf (NULL == aua, Exit);

	result = aua->protectedInitHardware ((IOService *)provider);

Exit:
	return result;
}

IOReturn AppleUSBAudioDevice::protectedInitHardware (IOService * provider) {
	char							string[kStringBufferSize];
	UInt8							stringIndex;
	IOReturn						err;
    Boolean							resultCode;
	UInt8 *							streamNumbers;
	UInt8							numStreams;

	debugIOLog ("+AppleUSBAudioDevice[%p]::start (%p)", this, provider);

	resultCode = FALSE;

	debugIOLog ("There are %d configurations on this device", controlInterface->GetDevice()->GetNumConfigurations ());
	debugIOLog ("Our control interface number is %d", controlInterface->GetInterfaceNumber ());
	usbAudio = USBAudioConfigObject::create (controlInterface->GetDevice()->GetFullConfigurationDescriptor (0), controlInterface->GetInterfaceNumber ());
	FailIf (NULL == usbAudio, Exit);

	// Check to make sure that the control interface we loaded against has audio streaming interfaces and not just MIDI.
	usbAudio->GetControlledStreamNumbers (&streamNumbers, &numStreams);
	debugIOLog ("Num streams controlled = %d", numStreams);
	debugIOLog ("GetNumStreamInterfaces = %d", usbAudio->GetNumStreamInterfaces ());
	FailIf (0 == numStreams, Exit);

	// If this is an iSub, we need to not go any further because we don't support it in this driver
	// This will cause the driver to not load on any device that has _only_ a low frequency effect output terminal
	FailIf (usbAudio->GetNumOutputTerminals (0, 0) == 1 && usbAudio->GetIndexedOutputTerminalType (0, 0, 0) == OUTPUT_LOW_FREQUENCY_EFFECTS_SPEAKER, Exit);

	err = kIOReturnError;
	string[0] = 0;
	stringIndex = controlInterface->GetInterfaceStringIndex ();
	if (0 != stringIndex) {
		err = controlInterface->GetDevice()->GetStringDescriptor (stringIndex, string, kStringBufferSize);
	} else {
	stringIndex = controlInterface->GetDevice()->GetProductStringIndex ();
	if (0 != stringIndex) {
		err = controlInterface->GetDevice()->GetStringDescriptor (stringIndex, string, kStringBufferSize);
	}
	}

	if (0 == string[0] || kIOReturnSuccess != err) {
		strcpy (string, "Unknown USB Audio Device");
	}

	setDeviceName (string);

	err = kIOReturnError;
	string[0] = 0;
	stringIndex = controlInterface->GetDevice()->GetManufacturerStringIndex ();
	if (0 != stringIndex) {
		err = controlInterface->GetDevice()->GetStringDescriptor (stringIndex, string, kStringBufferSize);
	}

	if (0 == string[0] || kIOReturnSuccess != err) {
		strcpy (string, "Unknown Manufacturer");
	}

	setManufacturerName (string);
	setDeviceTransportType (kIOAudioDeviceTransportTypeUSB);

	interfaceLock = IORecursiveLockAlloc ();
	FailIf (NULL == interfaceLock, Exit);

	resultCode = super::initHardware (provider);

Exit:
	debugIOLog ("-AppleUSBAudioDevice[%p]::start (%p)", this, provider);

	return TRUE;
}
/*
void AppleUSBAudioDevice::streamStopped (IOAudioEngine *audioEngine) {
    // This must be called from within the workLoop
    // The AppleUSBAudioEngine does so from _setState()
    // which has the same requirement
    debugIOLog ("+AppleUSBAudioDevice::streamStopped ()");
    
    if (allStreamsStopped()) {
        terminate();
    }
    
    debugIOLog ("-AppleUSBAudioDevice::streamStopped ()");
}

bool AppleUSBAudioDevice::allStreamsStopped () {
    OSCollectionIterator *			audioEngineIterator;
    IOAudioEngine *					audioEngine;
    bool							result;

    debugIOLog ("+AppleUSBAudioDevice[%p]::allStreamsStopped ()", this);
    result = TRUE;

    FailIf (NULL == audioEngines, Exit);

    // Need to do this on the IOWorkLoop
    
    audioEngineIterator = OSCollectionIterator::withCollection (audioEngines);

    if (audioEngineIterator) {
        while (audioEngine = (IOAudioEngine *)audioEngineIterator->getNextObject ()) {
            if (OSDynamicCast (AppleUSBAudioEngine, audioEngine)) {	// We only care that the AppleUSBAudioEngines are stopped
                if (audioEngine->getState () != kIOAudioEngineStopped) {
                    result = FALSE;
                    break;
                }
            }
        }
        audioEngineIterator->release ();
    }

Exit:
    debugIOLog ("-AppleUSBAudioDevice[%p]::allStreamsStopped ()", this);
    return result;
}
*/
IOReturn AppleUSBAudioDevice::performPowerStateChange (IOAudioDevicePowerState oldPowerState, IOAudioDevicePowerState newPowerState, UInt32 *microSecsUntilComplete) {
	IOReturn						result;

	debugIOLog ("+AppleUSBAudioDevice[%p]::performPowerStateChange (%d, %d, %p)", this, oldPowerState, newPowerState, microSecsUntilComplete);

	result = super::performPowerStateChange (oldPowerState, newPowerState, microSecsUntilComplete);

	if (oldPowerState == kIOAudioDeviceSleep) {
		debugIOLog ("Waking from sleep - flushing controls to the device.");
		flushAudioControls ();
	}

	return result;
}

void AppleUSBAudioDevice::stop (IOService *provider) {
	bool			shouldStop;

	debugIOLog ("+AppleUSBAudioDevice[%p]::stop (%p) - audioEngines = %p - rc=%d", this, provider, audioEngines, getRetainCount());

	shouldStop = TRUE;

	if (shouldStop) {
		performStop (provider);
	}

	debugIOLog("-AppleUSBAudioDevice[%p]::stop ()", this);
}

// Return FALSE if you don't want PRAM updated on a volume change, TRUE if you want it updated.
// Only update PRAM if we're on a Cube and the speakers are Cube, SoundSticks, or Mirconas (somethings).
Boolean AppleUSBAudioDevice::ShouldUpdatePRAM (void) {
	const IORegistryPlane *			usbPlane;
	IORegistryEntry *				usbRegEntry;
	OSObject *						obj;
	OSNumber *						number;
	UInt16							productID;
	UInt16							vendorID;
	Boolean							speakersGood;
	Boolean							connectionGood;
	Boolean							result;

	// Assume failure
	result = FALSE;
	speakersGood = FALSE;
	connectionGood = FALSE;

	// Make sure they're speakers that can support boot beep
	vendorID = controlInterface->GetDevice()->GetVendorID ();
	debugIOLog ("+ ShouldUpdatePRAM\nspeaker's vendorID = 0x%x", vendorID);
	if (kIOUSBVendorIDAppleComputer == vendorID || kIOUSBVendorIDHaronKardon == vendorID || kIOUSBVendorMicronas == vendorID) {
		speakersGood = TRUE;
	}
	debugIOLog ("speakersGood = %d", speakersGood);

	// They have to be plugged into a root hub or a hub in monitor that can support boot beep
	if (TRUE == speakersGood) {
		usbPlane = getPlane (kIOUSBPlane);
		FailIf (NULL == usbPlane, Exit);

		usbRegEntry = controlInterface->GetDevice()->getParentEntry (usbPlane);
		FailIf (NULL == usbRegEntry, Exit);

		obj = usbRegEntry->getProperty (kUSBVendorID);
		number = OSDynamicCast (OSNumber, obj);
		FailIf (NULL == number, Exit);

		vendorID = number->unsigned32BitValue ();
		debugIOLog ("hub's vendorID = 0x%x", vendorID);

		if (kIOUSBVendorIDAppleComputer == vendorID) {
			obj = usbRegEntry->getProperty (kUSBDevicePropertyLocationID);
			number = OSDynamicCast (OSNumber, obj);
			FailIf (NULL == number, Exit);

			if (OSDynamicCast (IOUSBRootHubDevice, usbRegEntry)) {
				// It's connected to the root hub
				connectionGood = TRUE;
				debugIOLog ("Directly connected to the root hub");
			} else {
				obj = usbRegEntry->getProperty (kUSBProductID);
				number = OSDynamicCast (OSNumber, obj);
				FailIf (NULL == number, Exit);

				productID = number->unsigned32BitValue ();
				debugIOLog ("hub's productID = 0x%x", productID);

				if (kStudioDisplay15CRT == productID || kStudioDisplay17CRT == productID || kCinemaDisplay == productID || kStudioDisplay17FP == productID) {
					// It's connected to a good monitor
					connectionGood = TRUE;
					debugIOLog ("Connected to a capable monitor");
				}
			}
		}
	}
	debugIOLog ("connectionGood = %d", connectionGood);

	// And there CANNOT be a "sound" node in the device tree so that OF will boot beep through them
	if (TRUE == connectionGood && FALSE == FindSoundNode ()) {
		result = TRUE;
	}

Exit:
	debugIOLog ("- ShouldUpdatePRAM result = %d", result);
	return result;
}

Boolean AppleUSBAudioDevice::FindSoundNode (void) {
	const IORegistryPlane *			dtPlane;
	IORegistryEntry *				regEntry;
	IORegistryIterator *			iterator;
	Boolean							found;
	Boolean							done;
	const char *					name;

	found = FALSE;

	dtPlane = IORegistryEntry::getPlane (kIODeviceTreePlane);
	FailIf (NULL == dtPlane, Exit);

	iterator = IORegistryIterator::iterateOver (dtPlane, kIORegistryIterateRecursively);
	FailIf (NULL == iterator, Exit);

	done = FALSE;
	regEntry = iterator->getNextObject ();
	while (NULL != regEntry && FALSE == done) {
		name = regEntry->getName ();
		if (0 == strcmp (name, "mac-io")) {
			// This is where we want to start the search
			iterator->release ();		// release the current iterator and make a new one rooted at "mac-io"
			iterator = IORegistryIterator::iterateOver (regEntry, dtPlane);
			done = TRUE;
		}
		regEntry = iterator->getNextObject ();
	}

	// Now the real search begins...
	regEntry = iterator->getNextObject ();
	while (NULL != regEntry && FALSE == found) {
		name = regEntry->getName ();
		if (0 == strcmp (name, "sound")) {
			found = TRUE;
		}
		regEntry = iterator->getNextObject ();
	}

	iterator->release ();

Exit:
	return found;
}

void AppleUSBAudioDevice::performStop (IOService *provider) {

    debugIOLog("+AppleUSBAudioDevice[%p]::performStop (%p)", this, provider);

	super::stop (provider);  // call the IOAudioDevice generic stop routine

    if (controlInterface) {
        controlInterface->close (this);
        controlInterface = NULL;
    }

    debugIOLog ("-AppleUSBAudioDevice[%p]::performStop (%p)", this, provider);
}

bool AppleUSBAudioDevice::terminate (IOOptionBits options) {
	bool							shouldTerminate;
	bool							result;

	shouldTerminate = TRUE;
	result = TRUE;
	debugIOLog ("+AppleUSBAudioDevice[%p]::terminate () - rc=%d", this, getRetainCount ());

	if (shouldTerminate) {
		result = super::terminate (options);
	}

	debugIOLog ("-AppleUSBAudioDevice[%p]::terminate () - rc=%d", this, getRetainCount ());

	return result;
}

bool AppleUSBAudioDevice::finalize (IOOptionBits options) {
	bool result;
	debugIOLog ("+AppleUSBAudioDevice[%p]::finalize (%p) - rc=%d", this, options, getRetainCount ());

	result = super::finalize (options);

	debugIOLog ("-AppleUSBAudioDevice[%p]::finalize (%p) - rc=%d", this, options, getRetainCount ());

	return result;
}

IOReturn AppleUSBAudioDevice::message (UInt32 type, IOService * provider, void * arg) {
	debugIOLog ("+AppleUSBAudioDevice[%p]::message (0x%x, %p) - rc=%d", this, type, provider, getRetainCount ());

	switch (type) {
		case kIOMessageServiceIsTerminated:
		case kIOMessageServiceIsRequestingClose:
			if (controlInterface != NULL && controlInterface == provider) {
				controlInterface->close (this);
				controlInterface = NULL;
			}
		default:
			;
	}

	debugIOLog ("-AppleUSBAudioDevice[%p]::message (0x%x, %p) - rc=%d", this, type, provider, getRetainCount ());
	return kIOReturnSuccess;
}

IOReturn AppleUSBAudioDevice::createControlsForInterface (IOAudioEngine *audioEngine, UInt8 interfaceNum, UInt8 altInterfaceNum) {
    AppleUSBAudioEngine *				usbAudioEngine;
    UInt8								channelNum;
    AppleUSBAudioLevelControl *			speakerControl;
    AppleUSBAudioLevelControl *			micControl;
	AppleUSBAudioLevelControl *			pramControl;
    AppleUSBAudioMuteControl *			muteControl;
    IOReturn							result;
	UInt16								terminalType;
	UInt8								numTerminals;
	UInt8								featureUnitID;
	UInt8								terminalIndex;
	UInt8								controlInterfaceNum;
	Boolean								shouldUpdatePRAM;

    debugIOLog ("+AppleUSBAudioDevice[%p]::createControlsForInterface %d %d", this, interfaceNum, altInterfaceNum);

    result = kIOReturnError;
	terminatingDriver = FALSE;
	FailIf (NULL == controlInterface, Exit);

    usbAudioEngine = OSDynamicCast (AppleUSBAudioEngine, audioEngine);
    FailIf (NULL == usbAudioEngine, Exit);

	controlInterfaceNum = controlInterface->GetInterfaceNumber ();

    if (usbAudioEngine->getDirection () == kIOAudioStreamDirectionOutput) {
        // It's an output device (e.g., a speaker).
		featureUnitID = 0;
		// look for a streaming input terminal that's connected to a non-streaming output terminal
		// for the moment we'll just look for a feature unit connected to a non-streaming output terminal
		numTerminals = usbAudio->GetNumOutputTerminals (controlInterfaceNum, 0);
		debugIOLog ("num output terminals = %d", numTerminals);
		for (terminalIndex = 0; terminalIndex < numTerminals; terminalIndex++) {
			debugIOLog ("terminalIndex = 0x%X", terminalIndex);
			terminalType = usbAudio->GetIndexedOutputTerminalType (controlInterfaceNum, 0, terminalIndex);
			debugIOLog ("terminalType = 0x%X", terminalType);
			if (terminalType != 0x0101) {	// Only look for output terminals that output audio (things we can play to)
				featureUnitID = usbAudio->GetFeatureUnitIDConnectedToOutputTerminal (controlInterfaceNum, 0, usbAudio->GetIndexedOutputTerminalID (controlInterfaceNum, 0, terminalIndex));
				debugIOLog ("featureUnitID = %d", featureUnitID);
				break;	// get out of for loop
			}
		}
		FailWithAction (0 == featureUnitID, result = kIOReturnSuccess, Exit);	// There isn't a feature unit connected to this input terminal
		// The interface was opened in AppleUSBAudioEngine::initHardware
		controlInterface->SetAlternateInterface (this, kRootAlternateSetting);

		shouldUpdatePRAM = ShouldUpdatePRAM ();
		if (shouldUpdatePRAM) {
			pramControl = AppleUSBAudioLevelControl::create (featureUnitID,
															controlInterfaceNum,
															0xff,
															1,
															shouldUpdatePRAM,
															(USBDeviceRequest)&deviceRequest,
															this,
															kIOAudioLevelControlSubTypeVolume,
															kIOAudioControlUsageOutput);
			if (NULL != pramControl) {
				audioEngine->addDefaultAudioControl (pramControl);
				pramControl->release ();
			}
		}

		for (channelNum = 0; channelNum <= usbAudio->GetNumChannels (interfaceNum, altInterfaceNum); channelNum++) {
			if (usbAudio->ChannelHasVolumeControl (controlInterfaceNum, 0, featureUnitID, channelNum)) {
				speakerControl = AppleUSBAudioLevelControl::create (featureUnitID,
																	controlInterfaceNum,
																	VOLUME_CONTROL,
																	channelNum,
																	shouldUpdatePRAM,
																	(USBDeviceRequest)&deviceRequest,
																	this,
																	kIOAudioLevelControlSubTypeVolume,
																	kIOAudioControlUsageOutput);

				if (NULL != speakerControl) {
					audioEngine->addDefaultAudioControl (speakerControl);
					speakerControl->release ();
				} else {
					debugIOLog ("++AppleUSBAudioDevice::createControlsForInterface () - error creating volume control for channelNum %d", channelNum);
				}
			}
		}
		muteControl = NULL;
		for (channelNum = 0; channelNum <= usbAudio->GetNumChannels (interfaceNum, altInterfaceNum); channelNum++) {
			if (usbAudio->ChannelHasMuteControl (controlInterfaceNum, 0, featureUnitID, channelNum)) {
				muteControl = AppleUSBAudioMuteControl::create (featureUnitID,
																controlInterfaceNum,
																channelNum,
																(USBDeviceRequest)&deviceRequest,
																this,
																kIOAudioControlUsageOutput);
				if (NULL != muteControl) {
					audioEngine->addDefaultAudioControl (muteControl);
					muteControl->release ();
				} else {
					debugIOLog ("++AppleUSBAudioDevice::createControlsForInterface () - error creating mute control for channelNum %d", channelNum);
				}
			}
		}
    } else {
        // It's an input device (e.g., a microphone).
		featureUnitID = 0;
		// look for a streaming input terminal that's connected to a non-streaming output terminal
		// for the moment we'll just look for a feature unit connected to a non-streaming output terminal
		numTerminals = usbAudio->GetNumOutputTerminals (controlInterfaceNum, 0);
		debugIOLog ("num output terminals = %d", numTerminals);
		for (terminalIndex = 0; terminalIndex < numTerminals; terminalIndex++) {
			debugIOLog ("terminalIndex = 0x%X", terminalIndex);
			terminalType = usbAudio->GetIndexedOutputTerminalType (controlInterfaceNum, 0, terminalIndex);
			debugIOLog ("terminalType = 0x%X", terminalType);
			if (terminalType == 0x0101) {	// Only look for output terminals that output digital audio data (things we can record from)
				featureUnitID = usbAudio->GetFeatureUnitIDConnectedToOutputTerminal (controlInterfaceNum, 0, usbAudio->GetIndexedOutputTerminalID (controlInterfaceNum, 0, terminalIndex));
				debugIOLog ("featureUnitID = %d", featureUnitID);
				break;	// get out of for loop
			}
		}
		FailWithAction (0 == featureUnitID, result = kIOReturnSuccess, Exit);	// There isn't a feature unit connected to this output terminal
		// The interface was opened in AppleUSBAudioEngine::initHardware
		controlInterface->SetAlternateInterface (this, kRootAlternateSetting);
		debugIOLog ("There are %d controls", usbAudio->GetNumChannels (interfaceNum, altInterfaceNum));
		for (channelNum = 0; channelNum <= usbAudio->GetNumChannels (interfaceNum, altInterfaceNum); channelNum++) {
			if (usbAudio->ChannelHasVolumeControl (controlInterfaceNum, 0, featureUnitID, channelNum)) {
				micControl = AppleUSBAudioLevelControl::create (featureUnitID,
																controlInterfaceNum,
																VOLUME_CONTROL,
																channelNum,
																FALSE,
																(USBDeviceRequest)&deviceRequest,
																this,
																kIOAudioLevelControlSubTypeVolume,
																kIOAudioControlUsageInput);

				if (NULL != micControl) {
					audioEngine->addDefaultAudioControl (micControl);
					micControl->release ();
				} else {
					debugIOLog ("++AppleUSBAudioDevice::createControlsForInterface () - error creating volume control for channelNum %d", channelNum);
				}
			} else {
				debugIOLog ("channel %d doesn't have a volume control", channelNum);
			}
		}
		muteControl = NULL;
		for (channelNum = 0; channelNum <= usbAudio->GetNumChannels (interfaceNum, altInterfaceNum); channelNum++) {
			if (usbAudio->ChannelHasMuteControl (controlInterfaceNum, 0, featureUnitID, channelNum)) {
				muteControl = AppleUSBAudioMuteControl::create (featureUnitID,
																controlInterfaceNum,
																channelNum,
																(USBDeviceRequest)&deviceRequest,
																this,
																kIOAudioControlUsageInput);
				if (NULL != muteControl) {
					audioEngine->addDefaultAudioControl (muteControl);
					muteControl->release ();
				} else {
					debugIOLog ("++AppleUSBAudioDevice::createControlsForInterface () - error creating mute control for channelNum %d", channelNum);
				}
			}
		}
    }

Exit:
	return result;
}

IOReturn AppleUSBAudioDevice::activateAudioEngine (IOAudioEngine *audioEngine, bool shouldStartAudioEngine, UInt8 interfaceNum, UInt8 altInterfaceNum) {
    IOReturn							result;

    debugIOLog ("+AppleUSBAudioDevice[%p]::activateAudioEngine (%p, %d) - rc=%d", this, audioEngine, shouldStartAudioEngine, getRetainCount());

    result = super::activateAudioEngine (audioEngine, shouldStartAudioEngine);

    debugIOLog ("-AppleUSBAudioDevice[%p]::activateAudioEngine (%p) - rc=%d", this, audioEngine, getRetainCount ());
    return result;
}

IOReturn AppleUSBAudioDevice::deviceRequest (IOUSBDevRequest *request, AppleUSBAudioDevice * self, IOUSBCompletion *completion) {
	IOReturn						result;

	debugIOLog ("+AppleUSBAudioDevice[%p]::deviceRequest (%p, %p)", self, request, completion);
	result = kIOReturnSuccess;
	if (self->controlInterface && FALSE == self->terminatingDriver) {
		FailIf (NULL == self->interfaceLock, Exit);

        IORecursiveLockLock (self->interfaceLock);
        result = self->controlInterface->DeviceRequest (request, completion);
        IORecursiveLockUnlock (self->interfaceLock);
    }

	debugIOLog ("-AppleUSBAudioDevice[%p]::deviceRequest (%p, %p)", self, request, completion);

Exit:
	return result;
}

bool AppleUSBAudioDevice::willTerminate (IOService * provider, IOOptionBits options) {
	debugIOLog ("+AppleUSBAudioDevice[%p]::willTerminate (%p)", this, provider);

	if (controlInterface == provider) {
		terminatingDriver = TRUE;
	}

	debugIOLog ("-AppleUSBAudioDevice[%p]::willTerminate", this);

	return super::willTerminate (provider, options);
}


#ifdef DEBUG

void AppleUSBAudioDevice::retain() const
{
//    debugIOLog("AppleUSBAudioDevice(%p)::retain() - rc=%d", this, getRetainCount());
    super::retain();
}

void AppleUSBAudioDevice::release() const
{
//    debugIOLog("AppleUSBAudioDevice(%p)::release() - rc=%d", this, getRetainCount());
	super::release();
}

#endif
