//--------------------------------------------------------------------------------
//
//	File:		AppleUSBAudioStream.h
//
//	Contains:	Support for the USB Audio Class Stream Interface.
//
//	Technology:	OS X
//
//--------------------------------------------------------------------------------

#ifndef _APPLEUSBAUDIOCOMMON_H
#define _APPLEUSBAUDIOCOMMON_H

#include <libkern/OSTypes.h>

#ifdef DEBUGLOGGING
	#define DEBUG_LEVEL 1
	#ifdef DEBUGLOG_TO_USBLOG
		#include <IOKit/usb/IOUSBLog.h>
	#endif
	#ifdef DEBUGLOG_TO_FIRELOG
		#include <IOKit/firewire/IOFireLog.h>
	#endif
#endif


//	-----------------------------------------------------------------
#define SoundAssertionMessage( cond, file, line, handler ) \
    "Sound assertion \"" #cond "\" failed in " #file " at line " #line " goto " #handler ""

#define SoundAssertionFailed( cond, file, line, handler ) \
	{debugIOLog( SoundAssertionMessage( cond, file, line, handler )); IOSleep(20);};

//	-----------------------------------------------------------------
#define	FailIf( cond, handler )										\
    if( cond ){														\
        SoundAssertionFailed( cond, __FILE__, __LINE__, handler )	\
        goto handler; 												\
    }

//	-----------------------------------------------------------------
#define	FailWithAction( cond, action, handler )						\
    if( cond ){														\
        SoundAssertionFailed( cond, __FILE__, __LINE__, handler )	\
            { action; }												\
        goto handler; 												\
    }

//	-----------------------------------------------------------------
#define FailMessage(cond, handler)									\
	if (cond) {														\
		SoundAssertionFailed(cond, __FILE__, __LINE__, handler)		\
		goto handler;												\
	}

//  -----------------------------------------------------------------
//
// System Logging or USB Prober Logging
//
//#define sleepTime 20
#define sleepTime 0

#ifdef DEBUGLOGGING /* { */
	#ifdef CONSOLELOGGING /* { */
		#define debugIOLog( message... ) \
			do {IOLog( #message "\n", message ); IOSleep(sleepTime);} while (0)
	#elif defined (DEBUGLOG_TO_KPRINTF) /* }{ */
			#define debugIOLog( message... ) \
				do { kprintf ( message ); kprintf ( "\n" ); } while (0)
	#elif defined (DEBUGLOG_TO_FIRELOG) /* }{ */
			#define debugIOLog( message... ) \
				do { FireLog( message ); FireLog( "\n" ); } while (0)
	#else /* }{ */
		#define debugIOLog( message... ) \
			do {USBLog( DEBUG_LEVEL_DEVELOPMENT, message );} while (0)
	#endif /* } */
#else /* }{ */
	#define debugIOLog( message... ) ;
#endif /* } */

// Following are a list of precompiler variables that affect the way that AppleUSBAudio executes, logs, and compiles.

// kUSBInputRecoveryTimeFraction represents the denominator (with 1 as the numerator) of the fractional multiple of a USB frame we must wait
// before data read via isoc in is available for converting. This must be greater than zero.
#define	kUSBInputRecoveryTimeFraction		4

// DEBUGLATENCY enables methods that allow the tracking of how long clipped samples stay in the output buffer prior to DMA
#define DEBUGLATENCY				FALSE

// PRIMEISOCINPUT queues kNumIsocFramesToPrime USB frames to be read and disregarded before attempting to stream samples to CoreAudio
#define PRIMEISOCINPUT				TRUE
#define	kNumIsocFramesToPrime		12

// LOGTIMESTAMPS prints the timestamp in nanoseconds whenever takeTimeStamp it is called
#define LOGTIMESTAMPS				FALSE

// RESETAFTERSLEEP causes a device reset to be issued after waking from sleep for all devices.
#define	RESETAFTERSLEEP				TRUE

// DEBUGANCHORS prints out the last kAnchorsToAccumulate anchors whenever the list fills; used to check anchor accuracy.
#define	DEBUGANCHORS				FALSE
#define	kAnchorsToAccumulate		10

// LOGWALLTIMEPERUSBCYCLE will display mWallTimePerUSBCycle * kExtraPrecision each time updateWallTimePerUSBCycle is executed.

#define LOGWALLTIMEPERUSBCYCLE		FALSE

// LOGDEVICEREQUESTS shows each device request and its result after it has been issued
#define	LOGDEVICEREQUESTS			FALSE

// DEBUGTIMER shows the entry and exit of all USB rate timer functions
#define DEBUGTIMER					FALSE

// DEBUGCONVERT shows the entry and exit of all calls to convertInputSamples
#define	DEBUGCONVERT				FALSE

// The following return codes are used by AppleUSBAudioDevice to detail the status of a format change.
enum 
{
	kAUAFormatChangeNormal				= kIOReturnSuccess,
	kAUAFormatChangeForced,
	kAUAFormatChangeForceFailure,
	kAUAFormatChangeError				= kIOReturnError
	
};


#endif /* _APPLEUSBAUDIOCOMMON_H */
