/*
 * Copyright (c) 1998-2000 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Copyright (c) 1999-2003 Apple Computer, Inc.  All Rights Reserved.
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef _IOKIT_IOAUDIOTYPES_H
#define _IOKIT_IOAUDIOTYPES_H

#include <libkern/OSTypes.h>
#include <mach/message.h>
#include <device/device_types.h>
/*!
 * @enum IOAudioEngineMemory
 * @abstract Used to identify the type of memory requested by a client process to be mapped into its process space
 * @discussion This is the parameter to the type field of IOMapMemory when called on an IOAudioEngine.  This is
 *  only intended for use by the Audio Device API library.
 * @constant kIOAudioSampleBuffer This requests the IOAudioEngine's sample buffer
 * @constant kIOAudioStatusBuffer This requests the IOAudioEngine's status buffer.  It's type is IOAudioEngineStatus.
 * @constant kIOAudioMixBuffer This requests the IOAudioEngine's mix buffer
*/
typedef enum _IOAudioEngineMemory {
    kIOAudioStatusBuffer 			= 0,
    kIOAudioSampleBuffer			= 1,
    kIOAudioMixBuffer				= 2,
	kIOAudioBytesInInputBuffer		= 3,
	kIOAudioBytesInOutputBuffer		= 4
} IOAudioEngineMemory;

/*!
 * @enum IOAudioEngineCalls
 * @abstract The set of constants passed to IOAudioEngineUserClient::getExternalMethodForIndex() when making calls
 *  from the IOAudioFamily user client code.
 */
typedef enum _IOAudioEngineCalls {
    kIOAudioEngineCallRegisterClientBuffer			= 0,
    kIOAudioEngineCallUnregisterClientBuffer		= 1,
    kIOAudioEngineCallGetConnectionID				= 2,
    kIOAudioEngineCallStart							= 3,
    kIOAudioEngineCallStop							= 4,
	kIOAudioEngineCallGetNearestStartTime			= 5
} IOAudioEngineCalls;

/*! @defined kIOAudioEngineNumCalls The number of elements in the IOAudioEngineCalls enum. */
#define kIOAudioEngineNumCalls		6

typedef enum _IOAudioEngineTraps {
    kIOAudioEngineTrapPerformClientIO				= 0
} IOAudioEngineTraps;

typedef enum _IOAudioEngineNotifications {
    kIOAudioEngineAllNotifications 					= 0,
    kIOAudioEngineStreamFormatChangeNotification	= 1,
    kIOAudioEngineChangeNotification 				= 2,
    kIOAudioEngineStartedNotification 				= 3,
    kIOAudioEngineStoppedNotification 				= 4,
    kIOAudioEnginePausedNotification				= 5,
    kIOAudioEngineResumedNotification				= 6
} IOAudioEngineNotifications;

/*!
 * @enum IOAudioEngineState
 * @abstract Represents the state of an IOAudioEngine
 * @constant kIOAudioEngineRunning The IOAudioEngine is currently running - it is transferring data to or 
 *           from the device.
 * @constant kIOAudioEngineStopped The IOAudioEngine is currently stopped - no activity is occurring.
 */

typedef enum _IOAudioEngineState {
    kIOAudioEngineStopped							= 0,
    kIOAudioEngineRunning							= 1,
    kIOAudioEnginePaused							= 2,
    kIOAudioEngineResumed							= 3
} IOAudioEngineState;


/*!
 * @typedef IOAudioEngineStatus
 * @abstract Shared-memory structure giving audio engine status
 * @discussion
 * @field fVersion Indicates version of this structure
 * @field fCurrentLoopCount Number of times around the ring buffer since the audio engine started
 * @field fLastLoopTime Timestamp of the last time the ring buffer wrapped
 * @field fEraseHeadSampleFrame Location of the erase head in sample frames - erased up to but not
 *        including the given sample frame
 */

typedef struct _IOAudioEngineStatus {
    UInt32					fVersion;
    volatile UInt32			fCurrentLoopCount;
    volatile AbsoluteTime	fLastLoopTime;
    volatile UInt32			fEraseHeadSampleFrame;
} IOAudioEngineStatus;

#define kIOAudioEngineCurrentStatusStructVersion		2

typedef struct _IOAudioStreamFormat {
    UInt32	fNumChannels;
    UInt32	fSampleFormat;
    UInt32	fNumericRepresentation;
    UInt8	fBitDepth;
    UInt8	fBitWidth;
    UInt8	fAlignment;
    UInt8	fByteOrder;
    UInt8	fIsMixable;
    UInt32	fDriverTag;
} IOAudioStreamFormat;

#define kFormatExtensionInvalidVersion					0
#define kFormatExtensionCurrentVersion					1

typedef struct _IOAudioStreamFormatExtension {
    UInt32	fVersion;
    UInt32	fFlags;
    UInt32	fFramesPerPacket;
    UInt32	fBytesPerPacket;
} IOAudioStreamFormatExtension;

#define kStreamDataDescriptorInvalidVersion				0
#define kStreamDataDescriptorCurrentVersion				1

typedef struct _IOAudioStreamDataDescriptor {
    UInt32	fVersion;
    UInt32	fNumberOfStreams;
    UInt32	fStreamLength[1];			// Array with fNumberOfStreams number of entries
} IOAudioStreamDataDescriptor;

typedef struct _IOAudioSampleIntervalDescriptor {
	UInt32	sampleIntervalHi;
	UInt32	sampleIntervalLo;
} IOAudioSampleIntervalDescriptor;

//	A struct for encapsulating a SMPTE time. The running rate should
//	be expressed in the AudioTimeStamp's mRateScalar field.
typedef struct _IOAudioSMPTETime
{
	UInt64	fCounter;			//	total number of messages received
	UInt32	fType;				//	the SMPTE type (see constants)
	UInt32	fFlags;				//	flags indicating state (see constants
	SInt16	fHours;				//	number of hours in the full message
	SInt16	fMinutes;			//	number of minutes in the full message
	SInt16	fSeconds;			//	number of seconds in the full message
	SInt16	fFrames;			//	number of frames in the full message
} IOAudioSMPTETime;

//	constants describing SMPTE types (taken from the MTC spec)
enum
{
	kIOAudioSMPTETimeType24			= 0,
	kIOAudioSMPTETimeType25			= 1,
	kIOAudioSMPTETimeType30Drop		= 2,
	kIOAudioSMPTETimeType30			= 3,
	kIOAudioSMPTETimeType2997		= 4,
	kIOAudioSMPTETimeType2997Drop	= 5
};

//	flags describing a SMPTE time stamp
enum
{
	kIOAudioSMPTETimeValid		= (1L << 0),	//	the full time is valid
	kIOAudioSMPTETimeRunning	= (1L << 1)		//	time is running
};

//	A struct for encapsulating the parts of a time stamp. The flags
//	say which fields are valid.
typedef struct _IOAudioTimeStamp
{
	UInt64				fSampleTime;	//	the absolute sample time, was a Float64
	UInt64				fHostTime;		//	the host's root timebase's time
	UInt64				fRateScalar;	//	the system rate scalar, was a Float64
	UInt64				fWordClockTime;	//	the word clock time
	IOAudioSMPTETime	fSMPTETime;		//	the SMPTE time
	UInt32				fFlags;			//	the flags indicate which fields are valid
	UInt32				fReserved;		//	reserved, pads the structure out to force 8 byte alignment
} IOAudioTimeStamp;

//	flags for the AudioTimeStamp sturcture
enum
{
	kIOAudioTimeStampSampleTimeValid	= (1L << 0),
	kIOAudioTimeStampHostTimeValid		= (1L << 1),
	kIOAudioTimeStampRateScalarValid	= (1L << 2),
	kIOAudioTimeStampWordClockTimeValid	= (1L << 3),
	kIOAudioTimeStampSMPTETimeValid		= (1L << 4)
};

//	Some commonly used combinations of timestamp flags
enum
{
	kIOAudioTimeStampSampleHostTimeValid	= (kIOAudioTimeStampSampleTimeValid | kIOAudioTimeStampHostTimeValid)
};

/*!
* @enum IOAudioStreamDirection
 * @abstract Represents the direction of an IOAudioStream
 * @constant kAudioOutput Output buffer
 * @constant kAudioInput Input buffer
 */

typedef enum _IOAudioStreamDirection {
    kIOAudioStreamDirectionOutput	= 0,
    kIOAudioStreamDirectionInput	= 1
} IOAudioStreamDirection;


/*!
 * @defined kIOAudioEngineDefaultMixBufferSampleSize
 */

#define kIOAudioEngineDefaultMixBufferSampleSize		sizeof(float)

/* The following are for use only by the IOKit.framework audio family code */

/*!
 * @enum IOAudioControlCalls
 * @abstract The set of constants passed to IOAudioControlUserClient::getExternalMethodForIndex() when making calls
 *  from the IOAudioFamily user client code.
 * @constant kIOAudioControlCallSetValue Used to set the value of an IOAudioControl.
 * @constant kIOAudioControlCallGetValue Used to get the value of an IOAudioControl.
 */
typedef enum _IOAudioControlCalls {
    kIOAudioControlCallSetValue = 0,
    kIOAudioControlCallGetValue = 1
} IOAudioControlCalls;

/*! @defined kIOAudioControlNumCalls The number of elements in the IOAudioControlCalls enum. */
#define kIOAudioControlNumCalls 	2

/*!
 * @enum IOAudioControlNotifications
 * @abstract The set of constants passed in the type field of IOAudioControlUserClient::registerNotificaitonPort().
 * @constant kIOAudioControlValueChangeNotification Used to request value change notifications.
 * @constant kIOAudioControlRangeChangeNotification Used to request range change notifications.
 */
typedef enum _IOAudioControlNotifications {
    kIOAudioControlValueChangeNotification = 0,
	kIOAudioControlRangeChangeNotification = 1
} IOAudioControlNotifications;

/*!
 * @struct IOAudioNotificationMessage
 * @abstract Used in the mach message for IOAudio notifications.
 * @field messageHeader Standard mach message header
 * @field ref The param passed to registerNotificationPort() in refCon.
 */
typedef struct _IOAudioNotificationMessage {
    mach_msg_header_t	messageHeader;
    UInt32		type;
    UInt32		ref;
    void *		sender;
} IOAudioNotificationMessage;

typedef struct _IOAudioSampleRate {
    UInt32	whole;
    UInt32	fraction;
} IOAudioSampleRate;

#define kNoIdleAudioPowerDown		0xffffffffffffffffULL

enum {
    kIOAudioPortTypeOutput		= 'outp',
    kIOAudioPortTypeInput		= 'inpt',
    kIOAudioPortTypeMixer		= 'mixr',
    kIOAudioPortTypePassThru	= 'pass',
    kIOAudioPortTypeProcessing	= 'proc'
};

enum {
    kIOAudioOutputPortSubTypeInternalSpeaker	= 'ispk',
    kIOAudioOutputPortSubTypeExternalSpeaker	= 'espk',
    kIOAudioOutputPortSubTypeHeadphones			= 'hdpn',
    kIOAudioOutputPortSubTypeLine				= 'line',
    kIOAudioOutputPortSubTypeSPDIF				= 'spdf',
    
    kIOAudioInputPortSubTypeInternalMicrophone	= 'imic',
    kIOAudioInputPortSubTypeExternalMicrophone	= 'emic',
    kIOAudioInputPortSubTypeCD					= 'cd  ',
    kIOAudioInputPortSubTypeLine				= 'line',
    kIOAudioInputPortSubTypeSPDIF				= 'spdf'
};

enum {
    kIOAudioControlTypeLevel			= 'levl',
    kIOAudioControlTypeToggle			= 'togl',
	kIOAudioControlTypeJack				= 'jack',
    kIOAudioControlTypeSelector			= 'slct'
};

enum {
    kIOAudioLevelControlSubTypeVolume						= 'vlme',
	kIOAudioLevelControlSubTypeLFEVolume					= 'subv',
	kIOAudioLevelControlSubTypePRAMVolume					= 'pram',
    kIOAudioToggleControlSubTypeMute						= 'mute',
	kIOAudioToggleControlSubTypeLFEMute						= 'subm',
	kIOAudioToggleControlSubTypeiSubAttach					= 'atch',
    kIOAudioSelectorControlSubTypeOutput					= 'outp',
    kIOAudioSelectorControlSubTypeInput						= 'inpt',
    kIOAudioSelectorControlSubTypeClockSource				= 'clck',
    kIOAudioSelectorControlSubTypeDestination				= 'dest',
	kIOAudioSelectorControlSubTypeChannelNominalLineLevel	= 'nlev'
};

enum {
    kIOAudioControlUsageOutput				= 'outp',
    kIOAudioControlUsageInput				= 'inpt',
    kIOAudioControlUsagePassThru			= 'pass',
    kIOAudioControlUsageCoreAudioProperty	= 'prop'
};

enum {
    kIOAudioControlChannelNumberInactive		= -1,
    kIOAudioControlChannelIDAll					= 0,
    kIOAudioControlChannelIDDefaultLeft			= 1,
    kIOAudioControlChannelIDDefaultRight		= 2,
    kIOAudioControlChannelIDDefaultCenter		= 3,
    kIOAudioControlChannelIDDefaultLeftRear		= 4,
    kIOAudioControlChannelIDDefaultRightRear	= 5,
    kIOAudioControlChannelIDDefaultSub			= 6
};

enum {
    kIOAudioSelectorControlSelectionValueNone				= 'none',
    
    // Output-specific selection IDs 
    kIOAudioSelectorControlSelectionValueInternalSpeaker	= 'ispk',
    kIOAudioSelectorControlSelectionValueExternalSpeaker	= 'espk',
    kIOAudioSelectorControlSelectionValueHeadphones			= 'hdpn',
    
    // Input-specific selection IDs
    kIOAudioSelectorControlSelectionValueInternalMicrophone	= 'imic',
    kIOAudioSelectorControlSelectionValueExternalMicrophone	= 'emic',
    kIOAudioSelectorControlSelectionValueCD					= 'cd  ',
    
    // Common selection IDs
    kIOAudioSelectorControlSelectionValueLine				= 'line',
    kIOAudioSelectorControlSelectionValueSPDIF				= 'spdf'
};

enum {
    kIOAudioStreamSampleFormatLinearPCM		= 'lpcm',
    kIOAudioStreamSampleFormatIEEEFloat		= 'ieee',
    kIOAudioStreamSampleFormatALaw			= 'alaw',
    kIOAudioStreamSampleFormatMuLaw			= 'ulaw',
    kIOAudioStreamSampleFormatMPEG			= 'mpeg',
    kIOAudioStreamSampleFormatAC3			= 'ac-3',
    kIOAudioStreamSampleFormat1937AC3		= 'cac3',
    kIOAudioStreamSampleFormat1937MPEG1		= 'mpg1',
    kIOAudioStreamSampleFormat1937MPEG2		= 'mpg2',
	kIOAudioStreamSampleFormatTimeCode		= 'time'
		//	a stream of IOAudioTimeStamp structures that capture any incoming time code information
};

enum {
    kIOAudioStreamNumericRepresentationSignedInt	= 'sint',
    kIOAudioStreamNumericRepresentationUnsignedInt	= 'uint',
	kIOAudioStreamNumericRepresentationIEEE754Float = 'flot'
};

enum {
    kIOAudioStreamAlignmentLowByte		= 0,
    kIOAudioStreamAlignmentHighByte		= 1
};

enum {
    kIOAudioStreamByteOrderBigEndian		= 0,
    kIOAudioStreamByteOrderLittleEndian		= 1
};

enum {
    kIOAudioLevelControlNegativeInfinity	= 0xffffffff
};

// Device connection types
enum {
	kIOAudioDeviceTransportTypeBuiltIn				= 'bltn',
	kIOAudioDeviceTransportTypePCI					= 'pci ',
	kIOAudioDeviceTransportTypeUSB					= 'usb ',
	kIOAudioDeviceTransportTypeFireWire				= '1394',
	kIOAudioDeviceTransportTypeNetwork				= 'ntwk',
	kIOAudioDeviceTransportTypeWireless				= 'wrls',
	kIOAudioDeviceTransportTypeOther				= 'othr'
};

// types that go nowhere
enum {
	OUTPUT_NULL										= 0x0100,
	INPUT_NULL										= 0x0101
};

// Input terminal types
enum {
	INPUT_UNDEFINED									= 0x0200,
	INPUT_MICROPHONE								= 0x0201,
	INPUT_DESKTOP_MICROPHONE						= 0x0202,
	INPUT_PERSONAL_MICROPHONE						= 0x0203,
	INPUT_OMNIDIRECTIONAL_MICROPHONE				= 0x0204,
	INPUT_MICROPHONE_ARRAY							= 0x0205,
	INPUT_PROCESSING_MICROPHONE_ARRAY				= 0x0206,
	INPUT_MODEM_AUDIO								= 0x207
};

// Output terminal types
enum {
	OUTPUT_UNDEFINED								= 0x0300,
	OUTPUT_SPEAKER									= 0x0301,
	OUTPUT_HEADPHONES								= 0x0302,
	OUTPUT_HEAD_MOUNTED_DISPLAY_AUDIO				= 0x0303,
	OUTPUT_DESKTOP_SPEAKER							= 0x0304,
	OUTPUT_ROOM_SPEAKER								= 0x0305,
	OUTPUT_COMMUNICATION_SPEAKER					= 0x0306,
	OUTPUT_LOW_FREQUENCY_EFFECTS_SPEAKER			= 0x0307
};

// Bi-directional terminal types
enum {
	BIDIRECTIONAL_UNDEFINED							= 0x0400,
	BIDIRECTIONAL_HANDSET							= 0x0401,
	BIDIRECTIONAL_HEADSET							= 0x0402,
	BIDIRECTIONAL_SPEAKERPHONE_NO_ECHO_REDX			= 0x0403,
	BIDIRECTIONAL_ECHO_SUPPRESSING_SPEAKERPHONE		= 0x0404,
	BIDIRECTIONAL_ECHO_CANCELING_SPEAKERPHONE		= 0x0405
};

// Telephony terminal types
enum {
	TELEPHONY_UNDEFINED								= 0x0500,
	TELEPHONY_PHONE_LINE							= 0x0501,
	TELEPHONY_TELEPHONE								= 0x0502,
	TELEPHONY_DOWN_LINE_PHONE						= 0x0503
};

// External terminal types
enum {
	EXTERNAL_UNDEFINED								= 0x0600,
	EXTERNAL_ANALOG_CONNECTOR						= 0x0601,
	EXTERNAL_DIGITAL_AUDIO_INTERFACE				= 0x0602,
	EXTERNAL_LINE_CONNECTOR							= 0x0603,
	EXTERNAL_LEGACY_AUDIO_CONNECTOR					= 0x0604,
	EXTERNAL_SPDIF_INTERFACE						= 0x0605,
	EXTERNAL_1394_DA_STREAM							= 0x0606,
	EXTERNAL_1394_DV_STREAM_SOUNDTRACK				= 0x0607,
	EXTERNAL_ADAT									= 0x0608,
	EXTERNAL_TDIF									= 0x0609,
	EXTERNAL_MADI									= 0x060A
};

// Embedded terminal types
enum {
	EMBEDDED_UNDEFINED								= 0x0700,
	EMBEDDED_LEVEL_CALIBRATION_NOISE_SOURCE			= 0x0701,
	EMBEDDED_EQUALIZATION_NOISE						= 0x0702,
	EMBEDDED_CD_PLAYER								= 0x0703,
	EMBEDDED_DAT									= 0x0704,
	EMBEDDED_DCC									= 0x0705,
	EMBEDDED_MINIDISK								= 0x0706,
	EMBEDDED_ANALOG_TAPE							= 0x0707,
	EMBEDDED_PHONOGRAPH								= 0x0708,
	EMBEDDED_VCR_AUDIO								= 0x0709,
	EMBEDDED_VIDEO_DISC_AUDIO						= 0x070A,
	EMBEDDED_DVD_AUDIO								= 0x070B,
	EMBEDDED_TV_TUNER_AUDIO							= 0x070C,
	EMBEDDED_SATELLITE_RECEIVER_AUDIO				= 0x070D,
	EMBEDDED_CABLE_TUNER_AUDIO						= 0x070E,
	EMBEDDED_DSS_AUDIO								= 0x070F,
	EMBEDDED_RADIO_RECEIVER							= 0x0710,
	EMBEDDED_RADIO_TRANSMITTER						= 0x0711,
	EMBEDDED_MULTITRACK_RECORDER					= 0x0712,
	EMBEDDED_SYNTHESIZER							= 0x0713
};

// Processing terminal types
enum {
	PROCESSOR_UNDEFINED								= 0x0800,
	PROCESSOR_GENERAL								= 0x0801
};

#endif /* _IOKIT_IOAUDIOTYPES_H */
