/*
 * Copyright (c) 1998-2002 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */


//
//	Includes
//

// SCSI Architecture Model Family includes
#include <IOKit/scsi/SCSICommandDefinitions.h>
#include "IOSCSIBlockCommandsDevice.h"
#include "SCSIBlockCommands.h"


//
//	Macros
//

#define DEBUG 									0
#define DEBUG_ASSERT_COMPONENT_NAME_STRING		"SBC"

#if DEBUG
#define SCSI_SBC_DEVICE_DEBUGGING_LEVEL			0
#endif


#include "IOSCSIArchitectureModelFamilyDebugging.h"


#if 0
#pragma mark -
#pragma mark  Block Commands Builders
#pragma mark -
#endif


//
//	 ERASE_10 - 	Builds a ERASE_10 command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::ERASE_10 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			ERA, 
						SCSICmdField1Bit 			RELADR, 
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS, 
						SCSICmdField2Byte 			TRANSFER_LENGTH, 
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->ERASE_10 (
					scsiRequest,
					ERA, 
					RELADR, 
					LOGICAL_BLOCK_ADDRESS, 
					TRANSFER_LENGTH, 
					CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 ERASE_12 - Builds a ERASE_12 command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::ERASE_12 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			ERA, 
						SCSICmdField1Bit 			RELADR, 
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS, 
						SCSICmdField4Byte 			TRANSFER_LENGTH, 
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->ERASE_12 (
				scsiRequest,
				ERA, 
				RELADR, 
				LOGICAL_BLOCK_ADDRESS, 
				TRANSFER_LENGTH, 
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 FORMAT_UNIT - Builds a FORMAT_UNIT command.					[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::FORMAT_UNIT(
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						IOByteCount					defectListSize,
						SCSICmdField1Bit 			FMTDATA, 
						SCSICmdField1Bit 			CMPLST, 
						SCSICmdField3Bit 			DEFECT_LIST_FORMAT, 
						SCSICmdField1Byte 			VENDOR_SPECIFIC, 
						SCSICmdField2Byte 			INTERLEAVE, 
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );

	status = GetSCSIBlockCommandObject ( )->FORMAT_UNIT (
				scsiRequest,
				dataBuffer,
				defectListSize,
				FMTDATA, 
				CMPLST, 
				DEFECT_LIST_FORMAT, 
				VENDOR_SPECIFIC, 
				INTERLEAVE, 
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 LOCK_UNLOCK_CACHE - Builds a LOCK_UNLOCK_CACHE command.		[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::LOCK_UNLOCK_CACHE (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			LOCK, 
						SCSICmdField1Bit 			RELADR, 
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS, 
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS, 
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );

	status = GetSCSIBlockCommandObject ( )->LOCK_UNLOCK_CACHE (
				scsiRequest,
				LOCK, 
				RELADR, 
				LOGICAL_BLOCK_ADDRESS, 
				NUMBER_OF_BLOCKS, 
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 MEDIUM_SCAN - Builds a MEDIUM_SCAN command.					[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::MEDIUM_SCAN (
						SCSITaskIdentifier			request,
				 		IOMemoryDescriptor *		dataBuffer,
			   			SCSICmdField1Bit 			WBS,
			   			SCSICmdField1Bit 			ASA,
			   			SCSICmdField1Bit 			RSD,
			   			SCSICmdField1Bit 			PRA,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			PARAMETER_LIST_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->MEDIUM_SCAN (
				scsiRequest,
				dataBuffer,
   				WBS,
   				ASA,
   				RSD,
   				PRA,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				PARAMETER_LIST_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 PREFETCH - Builds a PREFETCH command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::PREFETCH (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			IMMED,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->PREFETCH (
				scsiRequest,
				IMMED,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_6 - Builds a READ_6 command.								[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_6 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField21Bit 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt32 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	if ( TRANSFER_LENGTH == 0 )
	{
		
		// The TRANSFER_LENGTH is zero, this indicates that 256 blocks
		// should be transfer from the device
		requestedByteCount = 256 * blockSize;
		
	}
	
	else
	{
		requestedByteCount = TRANSFER_LENGTH * blockSize;
	}
	
	status = GetSCSIBlockCommandObject ( )->READ_6 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_10 - Builds a READ_10 command.							[PROTECTED]
//

bool 	
IOSCSIBlockCommandsDevice::READ_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit			FUA,
						SCSICmdField1Bit			RELADR,
						SCSICmdField4Byte			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte			TRANSFER_LENGTH,
						SCSICmdField1Byte			CONTROL )
{
	
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->READ_10 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				FUA,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_12 - Builds a READ_12 command.							[PROTECTED]
//

bool 	
IOSCSIBlockCommandsDevice::READ_12 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64 		requestedByteCount	= 0;

	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->READ_12 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				FUA,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_CAPACITY - Builds a READ_CAPACITY command.				[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_CAPACITY (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Bit 			PMI,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_CAPACITY (
				scsiRequest,
				dataBuffer,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				PMI,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_DEFECT_DATA_10 - Builds a READ_DEFECT_DATA_10 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_DEFECT_DATA_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			PLIST,
						SCSICmdField1Bit 			GLIST,
						SCSICmdField3Bit 			DEFECT_LIST_FORMAT,
						SCSICmdField2Byte 			ALLOCATION_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_DEFECT_DATA_10 (
				scsiRequest,
				dataBuffer,
				PLIST,
				GLIST,
				DEFECT_LIST_FORMAT,
				ALLOCATION_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_DEFECT_DATA_12 - Builds a READ_DEFECT_DATA_12 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_DEFECT_DATA_12 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			PLIST,
						SCSICmdField1Bit 			GLIST,
						SCSICmdField3Bit 			DEFECT_LIST_FORMAT,
						SCSICmdField4Byte 			ALLOCATION_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_DEFECT_DATA_12 (
				scsiRequest,
				dataBuffer,
				PLIST,
				GLIST,
				DEFECT_LIST_FORMAT,
				ALLOCATION_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_GENERATION - Builds a READ_GENERATION command.			[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_GENERATION (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			ALLOCATION_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_GENERATION (
				scsiRequest,
				dataBuffer,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				ALLOCATION_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_LONG - Builds a READ_LONG command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_LONG (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			CORRCT,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			BYTE_TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_LONG (
				scsiRequest,
				dataBuffer,
				CORRCT,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				BYTE_TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 READ_UPDATED_BLOCK_10 - Builds a READ_UPDATED_BLOCK_10 command.
//																	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::READ_UPDATED_BLOCK_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
					 	SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Bit 			LATEST,
					 	SCSICmdField15Bit 			GENERATION_ADDRESS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->READ_UPDATED_BLOCK_10 (
				scsiRequest,
				dataBuffer,
				fMediumBlockSize,
				DPO,
				FUA,
			 	RELADR,
				LOGICAL_BLOCK_ADDRESS,
				LATEST,
			 	GENERATION_ADDRESS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 REASSIGN_BLOCKS - Builds a REASSIGN_BLOCKS command.			[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::REASSIGN_BLOCKS (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->REASSIGN_BLOCKS (
				scsiRequest,
				dataBuffer,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 REBUILD - Builds a REBUILD command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::REBUILD (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
					 	SCSICmdField1Bit 			INTDATA,
						SCSICmdField2Bit 			PORT_CONTROL,
					 	SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			REBUILD_LENGTH,
						SCSICmdField4Byte 			PARAMETER_LIST_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->REBUILD (
				scsiRequest,
				dataBuffer,
				DPO,
				FUA,
			 	INTDATA,
				PORT_CONTROL,
			 	LOGICAL_BLOCK_ADDRESS,
				REBUILD_LENGTH,
				PARAMETER_LIST_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 REGENERATE - Builds a REGENERATE command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::REGENERATE (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit			DPO,
						SCSICmdField1Bit 			FUA,
					 	SCSICmdField1Bit 			INTDATA,
					 	SCSICmdField2Bit 			PORT_CONTROL,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			REBUILD_LENGTH,
						SCSICmdField4Byte 			PARAMETER_LIST_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->REGENERATE (
				scsiRequest,
				dataBuffer,
				DPO,
				FUA,
			 	INTDATA,
			 	PORT_CONTROL,
				LOGICAL_BLOCK_ADDRESS,
				REBUILD_LENGTH,
				PARAMETER_LIST_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 REZERO_UNIT - Builds a REZERO_UNIT command.					[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::REZERO_UNIT ( 
						SCSITaskIdentifier			request,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->REZERO_UNIT (
					scsiRequest,
					CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SEARCH_DATA_EQUAL_10 - Builds a SEARCH_DATA_EQUAL_10 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SEARCH_DATA_EQUAL_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			INVERT,
						SCSICmdField1Bit 			SPNDAT,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS_TO_SEARCH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SEARCH_DATA_EQUAL_10 (
				scsiRequest,
				dataBuffer,
				dataBuffer->getLength ( ),
				INVERT,
				SPNDAT,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS_TO_SEARCH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SEARCH_DATA_HIGH_10 - Builds a SEARCH_DATA_HIGH_10 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SEARCH_DATA_HIGH_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			INVERT,
						SCSICmdField1Bit 			SPNDAT,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS_TO_SEARCH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SEARCH_DATA_HIGH_10 (
				scsiRequest,
				dataBuffer,
				dataBuffer->getLength ( ),
				INVERT,
				SPNDAT,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS_TO_SEARCH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SEARCH_DATA_LOW_10 - Builds a SEARCH_DATA_LOW_10 command.		[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SEARCH_DATA_LOW_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			INVERT,
						SCSICmdField1Bit 			SPNDAT,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS_TO_SEARCH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SEARCH_DATA_LOW_10 (
				scsiRequest,
				dataBuffer,
				dataBuffer->getLength ( ),
				INVERT,
				SPNDAT,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS_TO_SEARCH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SEEK_6 - Builds a SEEK_6 command.								[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SEEK_6 (
						SCSITaskIdentifier			request,
						SCSICmdField21Bit 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SEEK_6 (
				scsiRequest,
				LOGICAL_BLOCK_ADDRESS, 
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SEEK_10 - Builds a SEEK_10 command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SEEK_10 (
						SCSITaskIdentifier			request,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SEEK_10 (
				scsiRequest,
				LOGICAL_BLOCK_ADDRESS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SET_LIMITS_10 - Builds a SET_LIMITS_10 command.				[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SET_LIMITS_10 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			RDINH,
						SCSICmdField1Bit 			WRINH,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SET_LIMITS_10 (
				scsiRequest,
				RDINH,
				WRINH,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SET_LIMITS_12 - Builds a SET_LIMITS_12 command.				[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SET_LIMITS_12 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			RDINH,
						SCSICmdField1Bit 			WRINH,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			NUMBER_OF_BLOCKS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SET_LIMITS_12 (
				scsiRequest,
				RDINH,
				WRINH,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 START_STOP_UNIT - Builds a START_STOP_UNIT command.			[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::START_STOP_UNIT (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			IMMED,
						SCSICmdField4Bit 			POWER_CONDITIONS,
						SCSICmdField1Bit 			LOEJ,
						SCSICmdField1Bit 			START,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->START_STOP_UNIT (
				scsiRequest,
				IMMED,
				POWER_CONDITIONS,
				LOEJ,
				START,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 SYNCHRONIZE_CACHE - Builds a SYNCHRONIZE_CACHE command.		[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::SYNCHRONIZE_CACHE (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			IMMED,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			NUMBER_OF_BLOCKS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->SYNCHRONIZE_CACHE (
				scsiRequest,
				IMMED,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				NUMBER_OF_BLOCKS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 UPDATE_BLOCK - Builds a UPDATE_BLOCK command.					[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::UPDATE_BLOCK (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->UPDATE_BLOCK (
				scsiRequest,
				dataBuffer,
				fMediumBlockSize,
				RELADR, 
				LOGICAL_BLOCK_ADDRESS, 
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 VERIFY_10 - Builds a VERIFY_10 command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::VERIFY_10 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			BLKVFY,
						SCSICmdField1Bit 			BYTCHK,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			VERIFICATION_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->VERIFY_10 (
				scsiRequest,
				DPO,
				BLKVFY,
				BYTCHK,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				VERIFICATION_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 VERIFY_12 - Builds a VERIFY_12 command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::VERIFY_12 (
						SCSITaskIdentifier			request,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			BLKVFY,
						SCSICmdField1Bit 			BYTCHK,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			VERIFICATION_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->VERIFY_12 (
				scsiRequest,
				DPO,
				BLKVFY,
				BYTCHK,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				VERIFICATION_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_6 - Builds a WRITE_6 command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_6 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField2Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField1Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
		
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt32 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	if ( TRANSFER_LENGTH == 0 )
	{
		
		// The TRANSFER_LENGTH is zero, this indicates that 256 blocks
		// should be transfer from the device
		requestedByteCount = 256 * blockSize;
		
	}
	
	else
	{
		requestedByteCount = TRANSFER_LENGTH * blockSize;
	}
	
	status = GetSCSIBlockCommandObject ( )->WRITE_6 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_10 - Builds a WRITE_10 command.							[PROTECTED]
//


bool
IOSCSIBlockCommandsDevice::WRITE_10 (
						SCSITaskIdentifier			request,
		   				IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField1Bit 			EBP,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
		
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->WRITE_10 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				FUA,
				EBP,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_12 - Builds a WRITE_12 command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_12 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField1Bit 			EBP,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
		
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->WRITE_12 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				FUA,
				EBP,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_AND_VERIFY_10 - Builds a WRITE_AND_VERIFY_10 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_AND_VERIFY_10 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			EBP,
						SCSICmdField1Bit 			BYTCHK,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
		
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->WRITE_AND_VERIFY_10 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				EBP,
				BYTCHK,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_AND_VERIFY_12 - Builds a WRITE_AND_VERIFY_12 command.	[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_AND_VERIFY_12 (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						UInt32						blockSize,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			EBP,
						SCSICmdField1Bit 			BYTCHK,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
		
	SCSITask *	scsiRequest			= NULL;
	bool		status 				= false;
	UInt64 		requestedByteCount	= 0;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	// Check the validity of the media
	require_nonzero ( blockSize, ErrorExit );
	
	requestedByteCount = TRANSFER_LENGTH * blockSize;
	
	status = GetSCSIBlockCommandObject ( )->WRITE_AND_VERIFY_12 (
				scsiRequest,
				dataBuffer,
				requestedByteCount,
				DPO,
				EBP,
				BYTCHK,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_LONG - Builds a WRITE_LONG command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_LONG (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->WRITE_LONG (
				scsiRequest,
				dataBuffer,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 WRITE_SAME - Builds a WRITE_SAME command.						[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::WRITE_SAME (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			PBDATA,
						SCSICmdField1Bit 			LBDATA,
						SCSICmdField1Bit 			RELADR,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->WRITE_SAME (
				scsiRequest,
				dataBuffer,
				PBDATA,
				LBDATA,
				RELADR,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 XDREAD - Builds a XDREAD command.								[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::XDREAD (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->XDREAD (
				scsiRequest,
				dataBuffer,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 XDWRITE - Builds a XDWRITE command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::XDWRITE (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField1Bit 			DISABLE_WRITE,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->XDWRITE (
				scsiRequest,
				dataBuffer,
				DPO,
				FUA,
				DISABLE_WRITE,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 XDWRITE_EXTENDED - Builds a XDWRITE_EXTENDED command.			[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::XDWRITE_EXTENDED (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			TABLE_ADDRESS,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField1Bit 			DISABLE_WRITE,
						SCSICmdField2Bit 			PORT_CONTROL,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField4Byte 			SECONDARY_BLOCK_ADDRESS,
						SCSICmdField4Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			SECONDARY_ADDRESS,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->XDWRITE_EXTENDED (
				scsiRequest,
				dataBuffer,
				TABLE_ADDRESS,
				DPO,
				FUA,
				DISABLE_WRITE,
				PORT_CONTROL,
				LOGICAL_BLOCK_ADDRESS,
				SECONDARY_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				SECONDARY_ADDRESS,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}


//
//	 XPWRITE - Builds a XPWRITE command.							[PROTECTED]
//

bool
IOSCSIBlockCommandsDevice::XPWRITE (
						SCSITaskIdentifier			request,
						IOMemoryDescriptor *		dataBuffer,
						SCSICmdField1Bit 			DPO,
						SCSICmdField1Bit 			FUA,
						SCSICmdField4Byte 			LOGICAL_BLOCK_ADDRESS,
						SCSICmdField2Byte 			TRANSFER_LENGTH,
						SCSICmdField1Byte 			CONTROL )
{
	
	SCSITask *	scsiRequest	= NULL;
	bool		status 		= false;
	
	scsiRequest = OSDynamicCast ( SCSITask, request );
	require_nonzero ( scsiRequest, ErrorExit );
	require ( scsiRequest->ResetForNewTask ( ), ErrorExit );
	
	status = GetSCSIBlockCommandObject ( )->XPWRITE (
				scsiRequest,
				dataBuffer,
				DPO,
				FUA,
				LOGICAL_BLOCK_ADDRESS,
				TRANSFER_LENGTH,
				CONTROL );
	
	
ErrorExit:
	
	
	return status;
	
}