/*
 * Copyright (c) 2000-2001 Apple Computer, Inc. All Rights Reserved.
 * 
 * The contents of this file constitute Original Code as defined in and are
 * subject to the Apple Public Source License Version 1.2 (the 'License').
 * You may not use this file except in compliance with the License. Please obtain
 * a copy of the License at http://www.apple.com/publicsource and read it before
 * using this file.
 * 
 * This Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
 * OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see the License for the
 * specific language governing rights and limitations under the License.
 */


//
// socks - socks version of IP sockets
//
// [Also see comments in header file.]
//
// This file contains the "generic" Socks functionality.
// Socks4 and Socks5 implementations are in their separate files.
//
#include "socks++.h"
#include "socks++4.h"
#include "socks++5.h"
#include "hosts.h"


namespace Security {
namespace IPPlusPlus {


//
// Static objects
//
ModuleNexus<SocksServer::Global> SocksServer::global;


//
// Create a SocksServer object
//
SocksServer *SocksServer::make(Version version, const IPSockAddress &addr)
{
    switch (version) {
    case 0:
        return NULL;		// no socks
    case 4:
        return new Socks4::Server(addr);
    case 5:
        return new Socks5::Server(addr);
    default:
        UnixError::throwMe(EINVAL);
    }
}


//
// TCPClientSockets (CONNECT access)
//    
void SocksClientSocket::open(const IPSockAddress &peer)
{
    if (mServer) {
        Support::connect(*this, peer);
        lastConnected(mPeerAddress.address());
    } else {
        TCPClientSocket::open(peer);
    }
}

void SocksClientSocket::open(const IPAddress &addr, IPPort port)
{
    open(IPSockAddress(addr, port));
}

void SocksClientSocket::open(const Host &host, IPPort port)
{
    if (mServer) {
        Support::connect(*this, host, port);
        lastConnected(mPeerAddress.address());
    } else {
        TCPClientSocket::open(host, port);
    }
}

void SocksClientSocket::setFd(int fd, const IPSockAddress &local, const IPSockAddress &peer)
{
    Socket::setFd(fd);
    mLocalAddress = local;
    mPeerAddress = peer;
}


//
// TCPServerSockets (BIND access)
//
void SocksServerSocket::open(const IPSockAddress &local, int)
{
    if (mServer) {
#if BUG_GCC
        if (mConnectionPeer)
            Support::bind(*this, mConnectionPeer, local.port());
        else
            Support::bind(*this, lastConnected(), local.port());
#else
        Support::bind(*this,
            mConnectionPeer ? mConnectionPeer : lastConnected(),
            local.port());
#endif
    } else {
        TCPServerSocket::open(local, 1);
    }
}

void SocksServerSocket::receive(SocksClientSocket &client)
{
    if (mServer) {
        Support::receive(*this, client);
    } else {
        TCPServerSocket::receive(client);
    }
}


//
// Address functions
//
IPSockAddress SocksServer::Support::localAddress(const Socket &me) const
{
    if (mServer)
        return mLocalAddress;
    else
        return me.localAddress();
}

IPSockAddress SocksServer::Support::peerAddress(const Socket &me) const
{
    if (mServer)
        return mPeerAddress;
    else
        return me.peerAddress();
}


}	// end namespace IPPlusPlus
}	// end namespace Security
