/*
 * Copyright (c) 2000-2001 Apple Computer, Inc. All Rights Reserved.
 * 
 * The contents of this file constitute Original Code as defined in and are
 * subject to the Apple Public Source License Version 1.2 (the 'License').
 * You may not use this file except in compliance with the License. Please obtain
 * a copy of the License at http://www.apple.com/publicsource and read it before
 * using this file.
 * 
 * This Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS
 * OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING WITHOUT
 * LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see the License for the
 * specific language governing rights and limitations under the License.
 */


/*
   File:      MDSAttrStrings.cpp

   Contains:  Static tables to map attribute names to numeric values.

   Copyright: (c) 2001 Apple Computer, Inc., all rights reserved.
*/

#include "MDSAttrStrings.h"
#include "MDSAttrUtils.h"
#include <Security/cssmapple.h>
#include <Security/mds_schema.h>
#include <Security/cssmkrapi.h> 	// CSSM_KR_POLICY_TYPE
#include <ctype.h>			// isDigit
#include <strings.h>		// strcmp 
#include <stdlib.h>			// atoi

namespace Security
{

/* declare one entry in a table of MDSNameValuePairs */
#define MNVP(attr)		{attr, #attr}

/* the NULL entry which terminates all MDSNameValuePair tables */
#define MNVP_END		{0, NULL}

/***
 *** common attributes
 ***/

/* CSSM_SERVICE_MASK */
const MDSNameValuePair MDSServiceNames[] =
{
	MNVP(CSSM_SERVICE_CSSM),
	MNVP(CSSM_SERVICE_CSP),
	MNVP(CSSM_SERVICE_DL),
	MNVP(CSSM_SERVICE_CL),
	MNVP(CSSM_SERVICE_TP),
	MNVP(CSSM_SERVICE_AC),
	MNVP(CSSM_SERVICE_KR),
	MNVP_END
};

/* CSSM_ACL_SUBJECT_TYPE */
const MDSNameValuePair MDSAclSubjectTypeNames[] =
{
	MNVP(CSSM_ACL_SUBJECT_TYPE_ANY),
	MNVP(CSSM_ACL_SUBJECT_TYPE_THRESHOLD),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PASSWORD),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PROTECTED_PASSWORD),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PROMPTED_PASSWORD),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PUBLIC_KEY),
	MNVP(CSSM_ACL_SUBJECT_TYPE_HASHED_SUBJECT),
	MNVP(CSSM_ACL_SUBJECT_TYPE_BIOMETRIC),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PROTECTED_BIOMETRIC),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PROMPTED_BIOMETRIC),
	MNVP(CSSM_ACL_SUBJECT_TYPE_LOGIN_NAME),
	MNVP(CSSM_ACL_SUBJECT_TYPE_EXT_PAM_NAME),
	/* from cssmapple.h */
	MNVP(CSSM_ACL_SUBJECT_TYPE_KEYCHAIN_PROMPT),
	MNVP(CSSM_ACL_SUBJECT_TYPE_PROCESS),
	MNVP(CSSM_ACL_SUBJECT_TYPE_CODE_SIGNATURE),
	MNVP(CSSM_ACL_SUBJECT_TYPE_COMMENT),
	MNVP_END
};

/* CSSM_ACL_AUTHORIZATION_TAG */
const MDSNameValuePair MDSAclAuthTagNames[] =
{
	MNVP(CSSM_ACL_AUTHORIZATION_ANY),
	MNVP(CSSM_ACL_AUTHORIZATION_LOGIN),
	MNVP(CSSM_ACL_AUTHORIZATION_GENKEY),
	MNVP(CSSM_ACL_AUTHORIZATION_DELETE),
	MNVP(CSSM_ACL_AUTHORIZATION_EXPORT_WRAPPED),
	MNVP(CSSM_ACL_AUTHORIZATION_EXPORT_CLEAR),
	MNVP(CSSM_ACL_AUTHORIZATION_IMPORT_WRAPPED),
	MNVP(CSSM_ACL_AUTHORIZATION_IMPORT_CLEAR),
	MNVP(CSSM_ACL_AUTHORIZATION_SIGN),
	MNVP(CSSM_ACL_AUTHORIZATION_ENCRYPT),
	MNVP(CSSM_ACL_AUTHORIZATION_DECRYPT),
	MNVP(CSSM_ACL_AUTHORIZATION_MAC),
	MNVP(CSSM_ACL_AUTHORIZATION_DERIVE),
	MNVP(CSSM_ACL_AUTHORIZATION_DBS_CREATE),
	MNVP(CSSM_ACL_AUTHORIZATION_DBS_DELETE),
	MNVP(CSSM_ACL_AUTHORIZATION_DB_READ),
	MNVP(CSSM_ACL_AUTHORIZATION_DB_INSERT),
	MNVP(CSSM_ACL_AUTHORIZATION_DB_MODIFY),
	MNVP(CSSM_ACL_AUTHORIZATION_DB_DELETE),
	/* from cssmapple.h */
	MNVP(CSSM_ACL_AUTHORIZATION_CHANGE_ACL),
	MNVP(CSSM_ACL_AUTHORIZATION_CHANGE_OWNER),
	MNVP_END
};

/***
 *** CSP attributes 
 ***/

/* CSSM_CONTEXT_TYPE */
const MDSNameValuePair MDSContextTypeNames[] =
{
	MNVP(CSSM_ALGCLASS_NONE),
	MNVP(CSSM_ALGCLASS_CUSTOM),
	MNVP(CSSM_ALGCLASS_SIGNATURE),
	MNVP(CSSM_ALGCLASS_SYMMETRIC),
	MNVP(CSSM_ALGCLASS_DIGEST),
	MNVP(CSSM_ALGCLASS_RANDOMGEN),
	MNVP(CSSM_ALGCLASS_UNIQUEGEN),
	MNVP(CSSM_ALGCLASS_MAC),
	MNVP(CSSM_ALGCLASS_ASYMMETRIC),
	MNVP(CSSM_ALGCLASS_KEYGEN),
	MNVP(CSSM_ALGCLASS_DERIVEKEY),
	MNVP(CSSM_ALGCLASS_NONE),
	MNVP_END
};

/* CSSM_ATTRIBUTE_TYPE */
const MDSNameValuePair MDSAttributeTypeNames[] = 
{
	MNVP(CSSM_ATTRIBUTE_NONE),
	MNVP(CSSM_ATTRIBUTE_CUSTOM),
	MNVP(CSSM_ATTRIBUTE_DESCRIPTION),
	MNVP(CSSM_ATTRIBUTE_KEY),
	MNVP(CSSM_ATTRIBUTE_INIT_VECTOR),
	MNVP(CSSM_ATTRIBUTE_SALT),
	MNVP(CSSM_ATTRIBUTE_PADDING),
	MNVP(CSSM_ATTRIBUTE_RANDOM),
	MNVP(CSSM_ATTRIBUTE_SEED),
	MNVP(CSSM_ATTRIBUTE_PASSPHRASE),
	MNVP(CSSM_ATTRIBUTE_KEY_LENGTH),
	MNVP(CSSM_ATTRIBUTE_KEY_LENGTH_RANGE),
	MNVP(CSSM_ATTRIBUTE_BLOCK_SIZE),
	MNVP(CSSM_ATTRIBUTE_OUTPUT_SIZE),
	MNVP(CSSM_ATTRIBUTE_ROUNDS),
	MNVP(CSSM_ATTRIBUTE_IV_SIZE),
	MNVP(CSSM_ATTRIBUTE_ALG_PARAMS),
	MNVP(CSSM_ATTRIBUTE_LABEL),
	MNVP(CSSM_ATTRIBUTE_KEY_TYPE),
	MNVP(CSSM_ATTRIBUTE_MODE),
	MNVP(CSSM_ATTRIBUTE_EFFECTIVE_BITS),
	MNVP(CSSM_ATTRIBUTE_START_DATE),
	MNVP(CSSM_ATTRIBUTE_END_DATE),
	MNVP(CSSM_ATTRIBUTE_KEYUSAGE),
	MNVP(CSSM_ATTRIBUTE_KEYATTR),
	MNVP(CSSM_ATTRIBUTE_VERSION),
	MNVP(CSSM_ATTRIBUTE_PRIME),
	MNVP(CSSM_ATTRIBUTE_SUBPRIME),
	MNVP(CSSM_ATTRIBUTE_ALG_ID),
	MNVP(CSSM_ATTRIBUTE_ITERATION_COUNT),
	MNVP(CSSM_ATTRIBUTE_ROUNDS_RANGE),
	MNVP(CSSM_ATTRIBUTE_KRPROFILE_LOCAL),
	MNVP(CSSM_ATTRIBUTE_KRPROFILE_REMOTE),
	MNVP(CSSM_ATTRIBUTE_CSP_HANDLE),
	MNVP(CSSM_ATTRIBUTE_DL_DB_HANDLE),
	MNVP(CSSM_ATTRIBUTE_ACCESS_CREDENTIALS),
	MNVP(CSSM_ATTRIBUTE_PUBLIC_KEY_FORMAT),
	MNVP(CSSM_ATTRIBUTE_PRIVATE_KEY_FORMAT),
	MNVP(CSSM_ATTRIBUTE_SYMMETRIC_KEY_FORMAT),
	MNVP(CSSM_ATTRIBUTE_WRAPPED_KEY_FORMAT),
	MNVP_END
};

/* CSSM_PADDING */
const MDSNameValuePair MDSPaddingNames[] = 
{
	MNVP(CSSM_PADDING_NONE),
	MNVP(CSSM_PADDING_CUSTOM),
	MNVP(CSSM_PADDING_ZERO),
	MNVP(CSSM_PADDING_ONE),
	MNVP(CSSM_PADDING_ALTERNATE),
	MNVP(CSSM_PADDING_FF),
	MNVP(CSSM_PADDING_PKCS5),
	MNVP(CSSM_PADDING_PKCS7),
	MNVP(CSSM_PADDING_CIPHERSTEALING),
	MNVP(CSSM_PADDING_RANDOM),
	MNVP(CSSM_PADDING_PKCS1),
	MNVP_END
};

/* CSSM_CSP_FLAGS */
const MDSNameValuePair MDSCspFlagsNames[] = 
{
	MNVP(CSSM_CSP_TOK_WRITE_PROTECTED),
	MNVP(CSSM_CSP_TOK_LOGIN_REQUIRED),
	MNVP(CSSM_CSP_TOK_USER_PIN_INITIALIZED),
	MNVP(CSSM_CSP_TOK_PROT_AUTHENTICATION),
	MNVP(CSSM_CSP_TOK_USER_PIN_EXPIRED),
	MNVP(CSSM_CSP_TOK_SESSION_KEY_PASSWORD),
	MNVP(CSSM_CSP_TOK_PRIVATE_KEY_PASSWORD),
	MNVP(CSSM_CSP_STORES_PRIVATE_KEYS),
	MNVP(CSSM_CSP_STORES_PUBLIC_KEYS),
	MNVP(CSSM_CSP_STORES_SESSION_KEYS),
	MNVP(CSSM_CSP_STORES_CERTIFICATES),
	MNVP(CSSM_CSP_STORES_GENERIC),
	MNVP_END
};

/* CSSM_ALGORITHMS */
const MDSNameValuePair MDSAlgorithmNames[] = 
{
	MNVP(CSSM_ALGID_NONE),
	MNVP(CSSM_ALGID_CUSTOM),
	MNVP(CSSM_ALGID_DH),
	MNVP(CSSM_ALGID_PH),
	MNVP(CSSM_ALGID_KEA),
	MNVP(CSSM_ALGID_MD2),
	MNVP(CSSM_ALGID_MD4),
	MNVP(CSSM_ALGID_MD5),
	MNVP(CSSM_ALGID_SHA1),
	MNVP(CSSM_ALGID_NHASH),
	MNVP(CSSM_ALGID_HAVAL),
	MNVP(CSSM_ALGID_RIPEMD),
	MNVP(CSSM_ALGID_IBCHASH),
	MNVP(CSSM_ALGID_RIPEMAC),
	MNVP(CSSM_ALGID_DES),
	MNVP(CSSM_ALGID_DESX),
	MNVP(CSSM_ALGID_RDES),
	MNVP(CSSM_ALGID_3DES_3KEY_EDE),
	MNVP(CSSM_ALGID_3DES_2KEY_EDE),
	MNVP(CSSM_ALGID_3DES_1KEY_EEE),
	MNVP(CSSM_ALGID_3DES_3KEY),
	MNVP(CSSM_ALGID_3DES_3KEY_EEE),
	MNVP(CSSM_ALGID_3DES_2KEY),
	MNVP(CSSM_ALGID_3DES_2KEY_EEE),
	MNVP(CSSM_ALGID_3DES_1KEY),
	MNVP(CSSM_ALGID_IDEA),
	MNVP(CSSM_ALGID_RC2),
	MNVP(CSSM_ALGID_RC5),
	MNVP(CSSM_ALGID_RC4),
	MNVP(CSSM_ALGID_SEAL),
	MNVP(CSSM_ALGID_CAST),
	MNVP(CSSM_ALGID_BLOWFISH),
	MNVP(CSSM_ALGID_SKIPJACK),
	MNVP(CSSM_ALGID_LUCIFER),
	MNVP(CSSM_ALGID_MADRYGA),
	MNVP(CSSM_ALGID_FEAL),
	MNVP(CSSM_ALGID_REDOC),
	MNVP(CSSM_ALGID_REDOC3),
	MNVP(CSSM_ALGID_LOKI),
	MNVP(CSSM_ALGID_KHUFU),
	MNVP(CSSM_ALGID_KHAFRE),
	MNVP(CSSM_ALGID_MMB),
	MNVP(CSSM_ALGID_GOST),
	MNVP(CSSM_ALGID_SAFER),
	MNVP(CSSM_ALGID_CRAB),
	MNVP(CSSM_ALGID_RSA),
	MNVP(CSSM_ALGID_DSA),
	MNVP(CSSM_ALGID_MD5WithRSA),
	MNVP(CSSM_ALGID_MD2WithRSA),
	MNVP(CSSM_ALGID_ElGamal),
	MNVP(CSSM_ALGID_MD2Random),
	MNVP(CSSM_ALGID_MD5Random),
	MNVP(CSSM_ALGID_SHARandom),
	MNVP(CSSM_ALGID_DESRandom),
	MNVP(CSSM_ALGID_SHA1WithRSA),
	MNVP(CSSM_ALGID_CDMF),
	MNVP(CSSM_ALGID_CAST3),
	MNVP(CSSM_ALGID_CAST5),
	MNVP(CSSM_ALGID_GenericSecret),
	MNVP(CSSM_ALGID_ConcatBaseAndKey),
	MNVP(CSSM_ALGID_ConcatKeyAndBase),
	MNVP(CSSM_ALGID_ConcatBaseAndData),
	MNVP(CSSM_ALGID_ConcatDataAndBase),
	MNVP(CSSM_ALGID_XORBaseAndData),
	MNVP(CSSM_ALGID_ExtractFromKey),
	MNVP(CSSM_ALGID_SSL3PreMasterGen),
	MNVP(CSSM_ALGID_SSL3MasterDerive),
	MNVP(CSSM_ALGID_SSL3KeyAndMacDerive),
	MNVP(CSSM_ALGID_SSL3MD5_MAC),
	MNVP(CSSM_ALGID_SSL3SHA1_MAC),
	MNVP(CSSM_ALGID_PKCS5_PBKDF1_MD5),
	MNVP(CSSM_ALGID_PKCS5_PBKDF1_MD2),
	MNVP(CSSM_ALGID_PKCS5_PBKDF1_SHA1),
	MNVP(CSSM_ALGID_WrapLynks),
	MNVP(CSSM_ALGID_WrapSET_OAEP),
	MNVP(CSSM_ALGID_BATON),
	MNVP(CSSM_ALGID_ECDSA),
	MNVP(CSSM_ALGID_MAYFLY),
	MNVP(CSSM_ALGID_JUNIPER),
	MNVP(CSSM_ALGID_FASTHASH),
	MNVP(CSSM_ALGID_3DES),
	MNVP(CSSM_ALGID_SSL3MD5),
	MNVP(CSSM_ALGID_SSL3SHA1),
	MNVP(CSSM_ALGID_FortezzaTimestamp),
	MNVP(CSSM_ALGID_SHA1WithDSA),
	MNVP(CSSM_ALGID_SHA1WithECDSA),
	MNVP(CSSM_ALGID_DSA_BSAFE),
	MNVP(CSSM_ALGID_ECDH),
	MNVP(CSSM_ALGID_ECMQV),
	MNVP(CSSM_ALGID_PKCS12_SHA1_PBE),
	MNVP(CSSM_ALGID_ECNRA),
	MNVP(CSSM_ALGID_SHA1WithECNRA),
	MNVP(CSSM_ALGID_ECES),
	MNVP(CSSM_ALGID_ECAES),
	MNVP(CSSM_ALGID_SHA1HMAC),
	MNVP(CSSM_ALGID_FIPS186Random),
	MNVP(CSSM_ALGID_ECC),
	MNVP(CSSM_ALGID_MQV),
	MNVP(CSSM_ALGID_NRA),
	MNVP(CSSM_ALGID_IntelPlatformRandom),
	MNVP(CSSM_ALGID_UTC),
	MNVP(CSSM_ALGID_HAVAL3),
	MNVP(CSSM_ALGID_HAVAL4),
	MNVP(CSSM_ALGID_HAVAL5),
	MNVP(CSSM_ALGID_TIGER),
	MNVP(CSSM_ALGID_MD5HMAC),
	MNVP(CSSM_ALGID_PKCS5_PBKDF2),
	MNVP(CSSM_ALGID_RUNNING_COUNTER),
	
	/* from cssmapple.h */
	MNVP(CSSM_ALGID_APPLE_YARROW),
	MNVP(CSSM_ALGID_AES),
	MNVP(CSSM_ALGID_FEE),
	MNVP(CSSM_ALGID_FEE_MD5),
	MNVP(CSSM_ALGID_FEE_SHA1),
	MNVP(CSSM_ALGID_FEED),
	MNVP(CSSM_ALGID_FEEDEXP),
	MNVP(CSSM_ALGID_ASC),
	MNVP(CSSM_ALGID_SHA1HMAC_LEGACY),
	MNVP_END
};

/* CSSM_ENCRYPT_MODE */
const MDSNameValuePair MDSEncryptModeNames[] = 
{
	MNVP(CSSM_ALGMODE_NONE),
	MNVP(CSSM_ALGMODE_CUSTOM),
	MNVP(CSSM_ALGMODE_ECB),
	MNVP(CSSM_ALGMODE_ECBPad),
	MNVP(CSSM_ALGMODE_CBC),
	MNVP(CSSM_ALGMODE_CBC_IV8),
	MNVP(CSSM_ALGMODE_CBCPadIV8),
	MNVP(CSSM_ALGMODE_CFB),
	MNVP(CSSM_ALGMODE_CFB_IV8),
	MNVP(CSSM_ALGMODE_CFBPadIV8),
	MNVP(CSSM_ALGMODE_OFB),
	MNVP(CSSM_ALGMODE_OFB_IV8),
	MNVP(CSSM_ALGMODE_OFBPadIV8),
	MNVP(CSSM_ALGMODE_COUNTER),
	MNVP(CSSM_ALGMODE_BC),
	MNVP(CSSM_ALGMODE_PCBC),
	MNVP(CSSM_ALGMODE_CBCC),
	MNVP(CSSM_ALGMODE_OFBNLF),
	MNVP(CSSM_ALGMODE_PBC),
	MNVP(CSSM_ALGMODE_PFB),
	MNVP(CSSM_ALGMODE_CBCPD),
	MNVP(CSSM_ALGMODE_PUBLIC_KEY),
	MNVP(CSSM_ALGMODE_PRIVATE_KEY),
	MNVP(CSSM_ALGMODE_SHUFFLE),
	MNVP(CSSM_ALGMODE_ECB64),
	MNVP(CSSM_ALGMODE_CBC64),
	MNVP(CSSM_ALGMODE_OFB64),
	MNVP(CSSM_ALGMODE_CFB32),
	MNVP(CSSM_ALGMODE_CFB16),
	MNVP(CSSM_ALGMODE_CFB8),
	MNVP(CSSM_ALGMODE_WRAP),
	MNVP(CSSM_ALGMODE_PRIVATE_WRAP),
	MNVP(CSSM_ALGMODE_RELAYX),
	MNVP(CSSM_ALGMODE_ECB128),
	MNVP(CSSM_ALGMODE_ECB96),
	MNVP(CSSM_ALGMODE_CBC128),
	MNVP(CSSM_ALGMODE_OAEP_HASH),
	MNVP(CSSM_ALGMODE_PKCS1_EME_V15),
	MNVP(CSSM_ALGMODE_PKCS1_EME_OAEP),
	MNVP(CSSM_ALGMODE_PKCS1_EMSA_V15),
	MNVP(CSSM_ALGMODE_ISO_9796),
	MNVP(CSSM_ALGMODE_X9_31),
	MNVP_END
};

/* CSSM_CSPTYPE */
const MDSNameValuePair MDSCspTypeNames[] = 
{
	MNVP(CSSM_CSP_SOFTWARE),
	MNVP(CSSM_CSP_HARDWARE),
	MNVP(CSSM_CSP_HYBRID),
	MNVP_END
};

/* CSSM_USEE_TAG */
const MDSNameValuePair MDSUseeTagsNames[] = 
{
	MNVP(CSSM_USEE_NONE),
	MNVP(CSSM_USEE_DOMESTIC),
	MNVP(CSSM_USEE_FINANCIAL),
	MNVP(CSSM_USEE_KRLE),
	MNVP(CSSM_USEE_KRENT),
	MNVP(CSSM_USEE_SSL),
	MNVP(CSSM_USEE_AUTHENTICATION),
	MNVP(CSSM_USEE_KEYEXCH),
	MNVP(CSSM_USEE_MEDICAL),
	MNVP(CSSM_USEE_INSURANCE),
	MNVP(CSSM_USEE_WEAK),
	MNVP_END
};

/* CSSM_CSP_READER_FLAGS */
const MDSNameValuePair MDSCspReaderFlagsNames[] = 
{
	MNVP(CSSM_CSP_RDR_TOKENPRESENT),
	MNVP(CSSM_CSP_RDR_EXISTS),
	MNVP(CSSM_CSP_RDR_HW),
	MNVP_END
};

/* CSSM_SC_FLAGS */
const MDSNameValuePair MDSCspScFlagsNames[] = 
{
	MNVP(CSSM_CSP_TOK_RNG),
	MNVP(CSSM_CSP_TOK_CLOCK_EXISTS),
	MNVP_END
};

/* CSSM_SAMPLE_TYPE */
const MDSNameValuePair MDSSampleTypeNames[] = 
{
	MNVP(CSSM_SAMPLE_TYPE_PASSWORD),
	MNVP(CSSM_SAMPLE_TYPE_HASHED_PASSWORD),
	MNVP(CSSM_SAMPLE_TYPE_PROTECTED_PASSWORD),
	MNVP(CSSM_SAMPLE_TYPE_PROMPTED_PASSWORD),
	MNVP(CSSM_SAMPLE_TYPE_SIGNED_NONCE),
	MNVP(CSSM_SAMPLE_TYPE_SIGNED_SECRET),
	MNVP(CSSM_SAMPLE_TYPE_BIOMETRIC),
	MNVP(CSSM_SAMPLE_TYPE_PROTECTED_BIOMETRIC),
	MNVP(CSSM_SAMPLE_TYPE_PROMPTED_BIOMETRIC),
	MNVP(CSSM_SAMPLE_TYPE_THRESHOLD),
	/* from cssmapple.h */
	MNVP(CSSM_SAMPLE_TYPE_KEYCHAIN_PROMPT),
	MNVP(CSSM_SAMPLE_TYPE_KEYCHAIN_LOCK),
	MNVP(CSSM_SAMPLE_TYPE_KEYCHAIN_CHANGE_LOCK),
	MNVP(CSSM_SAMPLE_TYPE_PROCESS),
	MNVP(CSSM_SAMPLE_TYPE_COMMENT),
	MNVP(CSSM_SAMPLE_TYPE_RETRY_ID),
	MNVP_END
};

/* CSSM_CERT_TYPE, CSSM_CERT_ENCODING (cert type in high 16 bits) */
const MDSNameValuePair MDSCertTypeNames[] = 
{
	/* CSSM_CERT_TYPE */
	MNVP(CSSM_CERT_UNKNOWN),
	MNVP(CSSM_CERT_X_509v1),
	MNVP(CSSM_CERT_X_509v2),
	MNVP(CSSM_CERT_X_509v3),
	MNVP(CSSM_CERT_PGP),
	MNVP(CSSM_CERT_PGP),
	MNVP(CSSM_CERT_SDSIv1),
	MNVP(CSSM_CERT_Intel),
	MNVP(CSSM_CERT_X_509_ATTRIBUTE),
	MNVP(CSSM_CERT_X9_ATTRIBUTE),
	MNVP(CSSM_CERT_TUPLE),
	MNVP(CSSM_CERT_ACL_ENTRY),
	MNVP(CSSM_CERT_MULTIPLE),
	/* CSSM_CERT_ENCODING */
	MNVP(CSSM_CERT_ENCODING_UNKNOWN),
	MNVP(CSSM_CERT_ENCODING_CUSTOM),
	MNVP(CSSM_CERT_ENCODING_BER),
	MNVP(CSSM_CERT_ENCODING_DER),
	MNVP(CSSM_CERT_ENCODING_NDR),
	MNVP(CSSM_CERT_ENCODING_SEXPR),
	MNVP(CSSM_CERT_ENCODING_PGP),
	MNVP(CSSM_CERT_ENCODING_MULTIPLE),
	MNVP_END
};

/* CSSM_CRL_TYPE, CSSM_CRL_ENCODING (CRL type in high 16 bits) */
const MDSNameValuePair MDSCrlTypeNames[] = 
{
	/* CSSM_CRL_TYPE */
	MNVP(CSSM_CRL_TYPE_UNKNOWN),
	MNVP(CSSM_CRL_TYPE_X_509v1),
	MNVP(CSSM_CRL_TYPE_X_509v2),
	MNVP(CSSM_CRL_TYPE_SPKI),
	MNVP(CSSM_CRL_TYPE_MULTIPLE),
	/* CSSM_CRL_ENCODING */
	MNVP(CSSM_CRL_ENCODING_UNKNOWN),
	MNVP(CSSM_CRL_ENCODING_CUSTOM),
	MNVP(CSSM_CRL_ENCODING_BER),
	MNVP(CSSM_CRL_ENCODING_DER),
	MNVP(CSSM_CRL_ENCODING_BLOOM),
	MNVP(CSSM_CRL_ENCODING_SEXPR),
	MNVP(CSSM_CRL_ENCODING_MULTIPLE),
	MNVP_END
};

/* CSSM_CERT_BUNDLE_TYPE, CSSM_CERT_BUNDLE_ENCODING (bundle type in high 16 bits) */
const MDSNameValuePair MDSCertBundleTypeNames[] = 
{
	/* CSSM_CERT_BUNDLE_TYPE */
	MNVP(CSSM_CERT_BUNDLE_UNKNOWN),
	MNVP(CSSM_CERT_BUNDLE_CUSTOM),
	MNVP(CSSM_CERT_BUNDLE_PKCS7_SIGNED_DATA),
	MNVP(CSSM_CERT_BUNDLE_PKCS7_SIGNED_ENVELOPED_DATA),
	MNVP(CSSM_CERT_BUNDLE_PKCS12),
	MNVP(CSSM_CERT_BUNDLE_PFX),
	MNVP(CSSM_CERT_BUNDLE_SPKI_SEQUENCE),
	MNVP(CSSM_CERT_BUNDLE_PGP_KEYRING),
	/* CSSM_CERT_BUNDLE_ENCODING */
	MNVP(CSSM_CERT_BUNDLE_ENCODING_UNKNOWN),
	MNVP(CSSM_CERT_BUNDLE_ENCODING_CUSTOM),
	MNVP(CSSM_CERT_BUNDLE_ENCODING_BER),
	MNVP(CSSM_CERT_BUNDLE_ENCODING_DER),
	MNVP(CSSM_CERT_BUNDLE_ENCODING_SEXPR),
	MNVP(CSSM_CERT_BUNDLE_PFX),
	MNVP(CSSM_CERT_BUNDLE_ENCODING_PGP),
	MNVP_END
};

/* CSSM_CL_TEMPLATE_TYPE */
const MDSNameValuePair MDSCertTemplateTypeNames[] = 
{
	MNVP(CSSM_CL_TEMPLATE_INTERMEDIATE_CERT),
	MNVP(CSSM_CL_TEMPLATE_PKIX_CERTTEMPLATE),
	MNVP_END
};

/* CSSM_TP_AUTHORITY_REQUEST_CERTISSUE */
const MDSNameValuePair MDSTpAuthRequestNames[] = 
{
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTISSUE),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTREVOKE),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTSUSPEND),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTRESUME),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTVERIFY),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTNOTARIZE),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CERTUSERECOVER),
	MNVP(CSSM_TP_AUTHORITY_REQUEST_CRLISSUE),
	MNVP_END
};

/* CSSM_DLTYPE */
const MDSNameValuePair MDSDlTypeNames[] = 
{
	MNVP(CSSM_DL_UNKNOWN),
	MNVP(CSSM_DL_CUSTOM),
	MNVP(CSSM_DL_LDAP),
	MNVP(CSSM_DL_ODBC),
	MNVP(CSSM_DL_PKCS11),
	MNVP(CSSM_DL_FFS),
	MNVP(CSSM_DL_MEMORY),
	MNVP(CSSM_DL_REMOTEDIR),
	MNVP_END
};

/* CSSM_DB_CONJUNCTIVE */
const MDSNameValuePair MDSDbConjunctiveNames[] = 
{
	MNVP(CSSM_DB_NONE),
	MNVP(CSSM_DB_AND),
	MNVP(CSSM_DB_OR),
	MNVP_END
};

/* CSSM_DB_OPERATOR */
const MDSNameValuePair MDSDbOperatorNames[] = 
{
	MNVP(CSSM_DB_EQUAL),
	MNVP(CSSM_DB_NOT_EQUAL),
	MNVP(CSSM_DB_LESS_THAN),
	MNVP(CSSM_DB_GREATER_THAN),
	MNVP(CSSM_DB_CONTAINS),
	MNVP(CSSM_DB_CONTAINS_INITIAL_SUBSTRING),
	MNVP(CSSM_DB_CONTAINS_FINAL_SUBSTRING),
	MNVP_END
};

/* CSSM_NET_PROTOCOL */
const MDSNameValuePair MDSNetProtocolNames[] = 
{
	MNVP(CSSM_NET_PROTO_NONE),
	MNVP(CSSM_NET_PROTO_CUSTOM),
	MNVP(CSSM_NET_PROTO_UNSPECIFIED),
	MNVP(CSSM_NET_PROTO_LDAP),
	MNVP(CSSM_NET_PROTO_LDAPS),
	MNVP(CSSM_NET_PROTO_LDAPNS),
	MNVP(CSSM_NET_PROTO_X500DAP),
	MNVP(CSSM_NET_PROTO_FTP),
	MNVP(CSSM_NET_PROTO_FTPS),
	MNVP(CSSM_NET_PROTO_OCSP),
	MNVP(CSSM_NET_PROTO_CMP),
	MNVP(CSSM_NET_PROTO_CMPS),
	MNVP_END
};

/* CSSM_DB_RETRIEVAL_MODES */
const MDSNameValuePair MDSDbRetrievalModeNames[] = 
{
	MNVP(CSSM_DB_TRANSACTIONAL_MODE),
	MNVP(CSSM_DB_FILESYSTEMSCAN_MODE),
	MNVP_END
};

/* CSSM_KR_POLICY_TYPE */
/* FIXME the spec is kind of ambiguous - do we want 
 * CSSM_KR_POLICY_TYPE or CSSM_KR_POLICY_FLAGS? */
const MDSNameValuePair MDSKrPolicyTypeNames[] = 
{
	MNVP(CSSM_KR_INDIV_POLICY),
	MNVP(CSSM_KR_ENT_POLICY),
	MNVP(CSSM_KR_LE_MAN_POLICY),
	MNVP(CSSM_KR_LE_USE_POLICY),
	MNVP_END
};


static bool isNumericStr(
	const char *str,
	bool hexOK)
{
	while(*str) {
		char c = *str++;
		if(isdigit(c)) {
			continue;
		}
		if(hexOK) {
			if((c >= 'a') && (c <= 'f')) {
				continue;
			}
			if((c >= 'A') && (c <= 'F')) {
				continue;
			}
		}
		return false;
	}
	return true;
}

/* convert ASCII hex digit - assumed validated already */
unsigned char hexDigit(
	unsigned char d)
{
	if((d >= '0') && (d <= '9')) {
		return d - '0';
	}
	else if((d >= 'a') && (d <= 'f')) {
		return d - 'a' + 10;
	}
	else {
		return d - 'A' + 10;
	}
}

static unsigned strToHex(
	const char *str)
{
	unsigned rtn = 0;
	while(*str) {
		rtn <<= 4;
		rtn |= hexDigit(*str++);
	}
	return rtn;
}

/*
 * Core routine to convert a single string token to a uint32. Incoming token can
 * be in the form of a string from the specified MDSNameValuePair table or a literal
 * number, either in hex (prefix "0x") or decimal. Tokens in any form may be 
 * prefixed by "<<" indicating the value is to be shifted left by 16 bits. 
 */
CSSM_RETURN MDSAttrNameToValue(
	const char *name,
	const MDSNameValuePair *table,	// optional, string must be decimal or hex if NULL
	uint32 &value)					// RETURNED
{
	if(name == NULL) {
		return CSSMERR_CSSM_MDS_ERROR;
	}
	if(*name == '\0') {
		/* empty string, legal */
		value = 0;
		return CSSM_OK;
	}
	
	/* prefixed by "<<"? */
	bool shiftBy16 = false;
	if((name != NULL) && (name[0] == '<') && (name[1] == '<')) {
		shiftBy16 = true;
		name += 2;
	}
	
	/* attempt to find the string in lookup table */
	if(table != NULL) {
		while(table->name != NULL) {
			if(!strcmp(table->name, name)) {
				value = table->value;
				if(shiftBy16) {
					value <<= 16;
				}
				return CSSM_OK;
			}
			table++;
		}
	}
	
	/* not found - is the string a number? */
	if(isdigit(name[0])) {
		bool isNum;
		bool isHex = false;
		if((name[0] == '0') && (name[1] == 'x')) {
			/* hex - skip first two chars */
			isHex = true;
			name += 2;
			isNum = isNumericStr(name, true);
		}
		else {
			isNum = isNumericStr(name, false);
		}
		if(!isNum) {
			return CSSMERR_CSSM_MDS_ERROR;
		}
		if(isHex) {
			value = strToHex(name);
		}
		else {
			value = atoi(name);
		}
		if(shiftBy16) {
			value <<= 16;
		}
		return CSSM_OK;
	}
	else {
		/* not a number */
		return CSSMERR_CSSM_MDS_ERROR;
	}
}


} // end namespace Security
