/*
 * Copyright (c) 1995
 *	A.R. Gordon (andrew.gordon@net-tel.co.uk).  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed for the FreeBSD project
 * 4. Neither the name of the author nor the names of any co-contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY ANDREW GORDON AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef lint
static const char rcsid[] =
  "$FreeBSD$";
#endif /* not lint */

/* main() function for status monitor daemon.  Some of the code in this	*/
/* file was generated by running rpcgen /usr/include/rpcsvc/sm_inter.x	*/
/* The actual program logic is in the file procs.c			*/

#include <err.h>
#include <stdio.h>
#include <stdlib.h>
#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "statd.h"

int debug = 0;		/* Controls syslog() calls for debug messages	*/
int notify_only = 0;	/* only send SM_NOTIFY messages */
const char *pid_file = NULL; /* name of any pid file that has been claimed */

extern void sm_prog_1(struct svc_req *rqstp, SVCXPRT *transp);
static void handle_sigchld(int sig);
static void handle_sig_cleanup(int sig);
static void cleanup_pid_file(void);
static void usage(void);

int
main(int argc, char **argv)
{
  SVCXPRT *transp;
  struct sigaction sa;
  int c;
  struct timespec ts;

  while ((c = getopt(argc, argv, "dn")) != EOF)
    switch (c) {
    case 'd':
      debug = 1;
      break;
    case 'n':
      notify_only = 1;
      break;
    default:
      usage();
    }

  /* Install signal handler to remove any pid file */
  signal(SIGINT, handle_sig_cleanup);
  signal(SIGTERM, handle_sig_cleanup);
  signal(SIGHUP, handle_sig_cleanup);
  signal(SIGQUIT, handle_sig_cleanup);

  init_file("/var/db/statd.status");

  openlog("rpc.statd", 0, LOG_DAEMON);

  if (notify_only) {
    notify_hosts();
    exit(0);
  }

  /* Note that it is NOT sensible to run this program from inetd - the 	*/
  /* protocol assumes that it will run immediately at boot time.	*/
  daemon(0, 0);

  if (claim_pid_file("/var/run/statd.pid", 0) < 0)
    errx(1, "rpc.statd already running");

  /* start portmapper, in case it hasn't been started yet */
  system("portmap");
  /* sleep a little to give portmap a chance to start */
  /* (better to sleep 50ms now than to timeout on portmap calls) */
  ts.tv_sec = 0;
  ts.tv_nsec = 50*1000*1000;
  nanosleep(&ts, NULL);

  (void)pmap_unset(SM_PROG, SM_VERS);

  transp = svcudp_create(RPC_ANYSOCK);
  if (transp == NULL)
    errx(1, "cannot create udp service");
  if (!svc_register(transp, SM_PROG, SM_VERS, sm_prog_1, IPPROTO_UDP))
    errx(1, "unable to register (SM_PROG, SM_VERS, udp)");

  transp = svctcp_create(RPC_ANYSOCK, 0, 0);
  if (transp == NULL)
    errx(1, "cannot create tcp service");
  if (!svc_register(transp, SM_PROG, SM_VERS, sm_prog_1, IPPROTO_TCP))
    errx(1, "unable to register (SM_PROG, SM_VERS, tcp)");

  if (debug) syslog(LOG_INFO, "Starting - debug enabled");
  else syslog(LOG_INFO, "Starting");

  /* Install signal handler to collect exit status of child processes	*/
  sa.sa_handler = handle_sigchld;
  sigemptyset(&sa.sa_mask);
  sigaddset(&sa.sa_mask, SIGCHLD);
  sa.sa_flags = SA_RESTART;
  sigaction(SIGCHLD, &sa, NULL);

  /* Initialisation now complete - start operating			*/
  notify_hosts();	/* Forks a process (if necessary) to do the	*/
			/* SM_NOTIFY calls, which may be slow.		*/

  svc_run();	/* Should never return					*/
  exit(1);
}

static void
usage()
{
      fprintf(stderr, "usage: rpc.statd [-dn]\n");
      exit(1);
}

/* handle_sigchld ---------------------------------------------------------- */
/*
   Purpose:	Catch SIGCHLD and collect process status
   Returns:	Nothing.
   Notes:	No special action required, other than to collect the
		process status and hence allow the child to die:
		we only use child processes for asynchronous transmission
		of SM_NOTIFY to other systems, so it is normal for the
		children to exit when they have done their work.
*/

static void handle_sigchld(int sig __unused)
{
  int pid, status;
  pid = wait4(-1, &status, WNOHANG, (struct rusage*)0);
  if (!pid) syslog(LOG_ERR, "Phantom SIGCHLD??");
  else if (status == 0)
  {
    if (debug) syslog(LOG_DEBUG, "Child %d exited OK", pid);
  }
  else syslog(LOG_ERR, "Child %d failed with status %d", pid,
    WEXITSTATUS(status));
}


/* claim_pid_file ---------------------------------------------------------- */
/*
   Purpose:	take ownership of and store pid in given pid_file
   Returns:	0 on success or -1 on failure
   Notes:	force parameter requests that current owner (if any) of
   		pid file be terminated.
*/

int
claim_pid_file(const char *name, int force)
{
  int pidfd, rv, retried = 0;
  FILE *pidfile;

try_again:

  /* attempt exclusive open of pid file */
  pidfd = open(name, O_EXCL|O_CREAT|O_WRONLY,
	       S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
  if (pidfd < 0) {
    char buf[16];
    pid_t pid;
    if (retried)
      return -1;
    bzero(buf, 16);
    retried = 1;
    /* pid file busy, check validity */
    pidfd = open(name, O_RDONLY);
    if (pidfd < 0)
      goto try_again;
    rv = read(pidfd, buf, 15);
    close(pidfd);
    if (rv <= 0)
      goto try_again;
    pid = atoi(buf);
    if (pid <= 0)
      goto try_again;
    rv = kill(pid, force ? SIGKILL : 0);
    /* if can't signal, assume stale pid file */
    if ((rv < 0) || force)
      unlink(name);
    goto try_again;
  }
  pid_file = name;
  atexit(cleanup_pid_file);

  pidfile = fdopen(pidfd, "w");
  if (pidfile) {
    fchmod(pidfd, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
    fprintf(pidfile, "%d\n", getpid());
    fclose(pidfile);
  } else
    perror("fdopen");
  close(pidfd);
  return 0;
}


/* cleanup_pid_file -------------------------------------------------------- */
/*
   Purpose:	delete any pid_file that has been claimed
   Returns:	Nothing
*/

void
cleanup_pid_file(void)
{
  if (pid_file) {
    unlink(pid_file);
    pid_file = NULL;
  }
}


/* handle_sig_cleanup ------------------------------------------------------ */
/*
   Purpose:	call pid file cleanup function on signal
   Returns:	Nothing
*/

static void
handle_sig_cleanup(int sig __unused)
{
  cleanup_pid_file();
  exit(1);
}

