/*
 * Copyright (c) 2000 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * The contents of this file constitute Original Code as defined in and
 * are subject to the Apple Public Source License Version 1.1 (the
 * "License").  You may not use this file except in compliance with the
 * License.  Please obtain a copy of the License at
 * http://www.apple.com/publicsource and read it before using this file.
 * 
 * This Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
 * Copyright (c) 1999 Apple Computer, Inc.  All rights reserved.
 *
 * HISTORY
 *
 */

#ifndef _OS_OSATOMIC_H
#define _OS_OSATOMIC_H

#include <libkern/OSBase.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*! @function OSCompareAndSwap
    @abstract Compare and swap operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSCompareAndSwap function compares the value at the specified address with oldVal. The value of newValue is written to the address only if oldValue and the value at the address are equal. OSCompareAndSwap returns true if newValue is written to the address; otherwise, it returns false.
    @param oldValue The value to compare at address.
    @param newValue The value to write to address if oldValue compares true.
    @param address The 4-byte aligned address of the data to update atomically.
    @result true if newValue was written to the address. */

extern Boolean OSCompareAndSwap( UInt32 oldValue, UInt32 newValue, UInt32 * address );

/*! @function OSAddAtomic
    @abstract 32-bit add operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSAddAtomic function adds the specified amount to the value at the specified address and returns the result.
    @param amount The amount to add.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The result of the addition. */

extern SInt32	OSAddAtomic(SInt32 amount, SInt32 * address);

/*! @function OSAddAtomic16
    @abstract 16-bit add operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSAddAtomic16 function adds the specified amount to the value at the specified address and returns the result.
    @param amount The amount to add.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The result of the addition. */

extern SInt16	OSAddAtomic16(SInt32 amount, SInt16 * address);

/*! @function OSAddAtomic8
    @abstract 8-bit add operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSAddAtomic8 function adds the specified amount to the value at the specified address and returns the result.
    @param amount The amount to add.
    @param address The address of the value to update atomically.
    @result The result of the addition. */

extern SInt8	OSAddAtomic8(SInt32 amount, SInt8 * address);

/*! @function OSIncrementAtomic
    @abstract 32-bit increment operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSIncrementAtomic function increments the value at the specified address by one and returns the value as it was before the change.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The value before the increment. */

extern SInt32	OSIncrementAtomic(SInt32 * address);

/*! @function OSIncrementAtomic16
    @abstract 16-bit increment operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSIncrementAtomic16 function increments the value at the specified address by one and returns the value as it was before the change.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The value before the increment. */

extern SInt16	OSIncrementAtomic16(SInt16 * address);

/*! @function OSIncrementAtomic8
    @abstract 8-bit increment operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSIncrementAtomic8 function increments the value at the specified address by one and returns the value as it was before the change.
    @param address The address of the value to update atomically.
    @result The value before the increment. */

extern SInt8	OSIncrementAtomic8(SInt8 * address);

/*! @function OSDecrementAtomic
    @abstract 32-bit decrement operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSDecrementAtomic function decrements the value at the specified address by one and returns the value as it was before the change.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The value before the decrement. */

extern SInt32	OSDecrementAtomic(SInt32 * address);

/*! @function OSDecrementAtomic16
    @abstract 16-bit decrement operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSDecrementAtomic16 function decrements the value at the specified address by one and returns the value as it was before the change.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The value before the decrement. */

extern SInt16	OSDecrementAtomic16(SInt16 * address);

/*! @function OSDecrementAtomic8
    @abstract 8-bit decrement operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSDecrementAtomic8 function decrements the value at the specified address by one and returns the value as it was before the change.
    @param address The address of the value to update atomically.
    @result The value before the decrement. */

extern SInt8	OSDecrementAtomic8(SInt8 * address);

/*! @function OSBitAndAtomic
    @abstract 32-bit logical and operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitAndAtomic function logically ands the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically and with the value.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The result of the logical and. */

extern UInt32	OSBitAndAtomic(UInt32 mask, UInt32 * address);

/*! @function OSBitAndAtomic16
    @abstract 16-bit logical and operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitAndAtomic16 function logically ands the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically and with the value.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The result of the logical and. */

extern UInt16	OSBitAndAtomic16(UInt32 mask, UInt16 * address);

/*! @function OSBitAndAtomic8
    @abstract 8-bit logical and operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitAndAtomic8 function logically ands the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically and with the value.
    @param address The address of the value to update atomically.
    @result The result of the logical and. */

extern UInt8	OSBitAndAtomic8(UInt32 mask, UInt8 * address);

/*! @function OSBitOrAtomic
    @abstract 32-bit logical or operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitOrAtomic function logically ors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The result of the logical or. */

extern UInt32	OSBitOrAtomic(UInt32 mask, UInt32 * address);

/*! @function OSBitOrAtomic16
    @abstract 16-bit logical or operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitOrAtomic16 function logically ors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The result of the logical or. */

extern UInt16	OSBitOrAtomic16(UInt32 mask, UInt16 * address);

/*! @function OSBitOrAtomic8
    @abstract 8-bit logical or operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitOrAtomic8 function logically ors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The address of the value to update atomically.
    @result The result of the logical or. */

extern UInt8	OSBitOrAtomic8(UInt32 mask, UInt8 * address);

/*! @function OSBitXorAtomic
    @abstract 32-bit logical xor operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitXorAtomic function logically xors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The 4-byte aligned address of the value to update atomically.
    @result The result of the logical xor. */

extern UInt32	OSBitXorAtomic(UInt32 mask, UInt32 * address);

/*! @function OSBitXorAtomic16
    @abstract 16-bit logical xor operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitXorAtomic16 function logically xors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The 2-byte aligned address of the value to update atomically.
    @result The result of the logical xor. */

extern UInt16	OSBitXorAtomic16(UInt32 mask, UInt16 * address);

/*! @function OSBitXorAtomic8
    @abstract 8-bit logical xor operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSBitXorAtomic8 function logically xors the bits of the specified mask into the value at the specified address and returns the result.
    @param mask The mask to logically or with the value.
    @param address The address of the value to update atomically.
    @result The result of the logical xor. */

extern UInt8	OSBitXorAtomic8(UInt32 mask, UInt8 * address);

/*! @function OSTestAndSet
    @abstract Bit test and set operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSTestAndSet function sets a single bit in a byte at a specified address. It returns true if the bit was already set, false otherwise.
    @param bit The bit number in the range 0 through 7.
    @param address The address of the byte to update atomically.
    @result true if the bit was already set, false otherwise. */

extern Boolean	OSTestAndSet(UInt32 bit, UInt8 * startAddress);

/*! @function OSTestAndClear
    @abstract Bit test and clear operation, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSTestAndClear function clears a single bit in a byte at a specified address. It returns true if the bit was already clear, false otherwise.
    @param bit The bit number in the range 0 through 7.
    @param address The address of the byte to update atomically.
    @result true if the bit was already clear, false otherwise. */

extern Boolean	OSTestAndClear(UInt32 bit, UInt8 * startAddress);

/*! @function OSEnqueueAtomic
    @abstract Singly linked list head insertion, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSEnqueueAtomic function places an element at the head of a single linked list, which is specified with the address of a head pointer, listHead. The element structure has a next field whose offset is specified.
    @param listHead The address of a head pointer for the list .
    @param element The list element to insert at the head of the list.
    @param elementNextFieldOffset The byte offset into the element where a pointer to the next element in the list is stored. */

extern void	OSEnqueueAtomic(void ** listHead, void * element,
				SInt32 elementNextFieldOffset);

/*! @function OSDequeueAtomic
    @abstract Singly linked list element head removal, performed atomically with respect to all devices that participate in the coherency architecture of the platform.
    @discussion The OSDequeueAtomic function removes an element from the head of a single linked list, which is specified with the address of a head pointer, listHead. The element structure has a next field whose offset is specified.
    @param listHead The address of a head pointer for the list .
    @param elementNextFieldOffset The byte offset into the element where a pointer to the next element in the list is stored.
    @result A removed element, or zero if the list is empty. */

extern void *	OSDequeueAtomic(void ** listHead,
				SInt32 elementNextFieldOffset);

/*! @function OSSynchronizeIO
    @abstract The OSSynchronizeIO routine ensures orderly load and store operations to noncached memory mapped I/O devices.
    @discussion The OSSynchronizeIO routine ensures orderly load and store operations to noncached memory mapped I/O devices. It executes the eieio instruction on PowerPC processors. */

static __inline__ void OSSynchronizeIO(void)
{
#if defined(__ppc__)
        __asm__ ("eieio");
#endif
}

#if defined(__cplusplus)
}
#endif

#endif /* ! _OS_OSATOMIC_H */
