/*
	$Id: CExpansion.cpp,v 1.1 2003/04/20 23:33:11 dasenbro Exp $

	File:		CExpansion.cpp

	Contains:	C++ implementation of generic 

	Version:	Apple Mail Server - Mac OS X :  $Revision: 1.1 $

	Written by:	Michael Dasenbrock

	Copyright:	 1996-2001 by Apple Computer, Inc., all rights reserved.

 	NOT_FOR_OPEN_SOURCE <to be reevaluated at a later time>

	Change History:

		$Log: CExpansion.cpp,v $
		Revision 1.1  2003/04/20 23:33:11  dasenbro
		Initial check-in.
		
		Revision 1.11  2002/07/27 02:15:33  dasenbro
		Add 127.0.0.1 to default SMTP relay list.
		
		Revision 1.10  2002/06/04 20:59:27  dasenbro
		Added UnLockTheCache to catch.
		
		Revision 1.9  2002/03/21 16:24:19  dasenbro
		Updated version information.
		
		Revision 1.8  2002/02/20 21:23:52  dasenbro
		Fixed a compiler warning.
		
		Revision 1.7  2001/06/21 20:50:50  dasenbro
		Updated file header info.
		
		Revision 1.6  2001/06/21 16:27:47  dasenbro
		Removed some dead code.
		

	Projector History:
		
	To Do:
*/


// PP
#include <string.h>
#include <time.h>

// App
#include "Database.h"
#include "CMailDatabase.h"
#include "CExpansion.h"
#include "CDBMailBaseObject.h"
#include "CString.h"
#include "CMailSpool.h"
#include "UException.h"
#include "CGlobals.h"


//--------------------------------------------------------------------------------------------------
//	* GetSetFields
//
//--------------------------------------------------------------------------------------------------

Boolean	CExpansion::GetSetFields (	const eFieldDataRequest	 inFieldRequest,
									const void				*inObjDataPtr,
									void					*outFieldData )
{
	Boolean			 result			= false;
	CExpandData	   *aclDataPtr		= (CExpandData *)inObjDataPtr;
	CExpansion	   *aTempObject		= NULL;
	ObjID		   *anObjIDPtr		= (ObjID *)outFieldData;

	if ( aclDataPtr != nil )
	{
		switch ( inFieldRequest )
		{
			case kSetObjID:
				aclDataPtr->fObjectID = *anObjIDPtr;
				break;

			case kObjectPINGRequest:
				// cause the objects constructor to get called, so we force a check of
				// the struct alignment...
				aTempObject = new CExpansion;
				delete aTempObject;
				break;

			case k2ndIDTargetChk:
				// fall through...
			
			default:
				result = false;
				break;
		}
	}

	return( result );
} // GetSetFields


//--------------------------------------------------------------------------------------------------
//	* GetObjectID
//
//--------------------------------------------------------------------------------------------------

ObjID CExpansion::GetObjectID ( void )
{
	return ( this->GetMyObjectID() );
}

//--------------------------------------------------------------------------------------------------
//	 Done
//
//--------------------------------------------------------------------------------------------------

void CExpansion::Done ( CExpansion* &inPtr )
{
	CDBBaseObject *aDBObject = (CDBBaseObject *) inPtr;
	aDBObject->Done(aDBObject);
	inPtr = (CExpansion *)aDBObject;

}


//--------------------------------------------------------------------------------------------------
//	* ObjectAllocNew
//
//--------------------------------------------------------------------------------------------------

CDBBaseObject* CExpansion::ObjectAllocNew ( void )
{
	return( new CExpansion );
} // ObjectAllocNew


//--------------------------------------------------------------------------------------------------
//	* FindByID
//
//--------------------------------------------------------------------------------------------------

CExpansion* CExpansion::FindByID ( const ObjID inObjectID )
{
	CExpansion	*outObj	= nil;

	outObj = (CExpansion *) CDBMailBaseObject::MailUniversalFindByID ( inObjectID,
															CExpansion::GetObjTypeConstant(),
															CExpansion::GetObjVersConstant(),
															CExpansion::GetObjSizeConstant(),
															CExpansion::ObjectAllocNew );
	if ( outObj != nil )
	{
		outObj->SetMaxItems();
	}

	return( outObj );

} // FindByID



//--------------------------------------------------------------------------------------------------
//	* Create
//
//--------------------------------------------------------------------------------------------------

CExpansion* CExpansion::Create ( ObjID inOwner, uInt32 inItemSize )
{
	CExpansion	*objPtr	= nil;		// creates a reference
	OSErr		result	= kNoErr;
	ObjID		aNewID	= 0;

	try
	{
		// If the itemsize is greater than we have room for, bail
		ThrowIf_( (sInt16)inItemSize > kMaxDataBlockSize );

		objPtr = new CExpansion;
		ThrowIfMemFail_( objPtr );

		(gDB->GetBaseObjectCachePtr())->LockTheCache(CExpansion::GetObjTypeConstant());

		result = gDB->CreateObj( kExpansionSignature, aNewID, objPtr->GetMyObjectData() );
		if ( result == CMailDatabase::kDBNoErr )
		{
			objPtr->Use();
			objPtr->SetOwnerID( inOwner );
			objPtr->SetItemSize( inItemSize );
			objPtr->SetDirty();
		}
		else
		{
			delete objPtr;
			objPtr = nil;
		}
		
		(gDB->GetBaseObjectCachePtr())->UnLockTheCache( CExpansion::GetObjTypeConstant() );
	}

	catch ( long err )
	{
		(gDB->GetBaseObjectCachePtr())->UnLockTheCache( CExpansion::GetObjTypeConstant() );
	}
	
	return( objPtr );

} // Create



//--------------------------------------------------------------------------------------------------
//	* Delete
//
//--------------------------------------------------------------------------------------------------

Boolean CExpansion::Delete ( CExpansion *inObjPtr )
{
	Boolean deleted = false;
	OSErr	result;

	Try_
	{
		if ( inObjPtr != nil )
		{
			result = gDB->RemoveObj( kExpansionSignature, inObjPtr->GetMyObjectID());
			deleted = ( result == kNoErr );
			if ( deleted )
			{
				inObjPtr->SetMyObjectID( 0 );
			}
		}
	}
	Catch_( err )
	{
	}

	return( deleted );

} // Delete



//--------------------------------------------------------------------------------------------------
//	* Count
//
//--------------------------------------------------------------------------------------------------

uInt32 CExpansion::Count ( void )
{
	OSErr	result;
	uInt32	aCount = 0;

	if ( gDB != nil )
	{
		result = gDB->GetObjectCount( kExpansionSignature, aCount );
	}

	return( aCount );

} // Count



//--------------------------------------------------------------------------------------------------
//	* 
//
//--------------------------------------------------------------------------------------------------

void CExpansion::VerifyAll( void )
{
} // VerifyAll


#pragma mark -
#pragma mark * Construct

//--------------------------------------------------------------------------------------------------
//	* CExpansion
//
//--------------------------------------------------------------------------------------------------

CExpansion::CExpansion ( void ) :
	CDBMailBaseObject (	&fObjectData.fDBHeader,
				&fObjectData.fDBFooter,
				&fObjectData,
				CExpansion::GetObjTypeConstant(),
				CExpansion::GetObjVersConstant(),
				CExpansion::GetObjSizeConstant() )

{
	fObjectData.fObjectID		= 0;
	fObjectData.fOwnerID		= 0;

	fObjectData.fPrevObjectID	= 0;
	fObjectData.fNextObjectID	= 0;

	fObjectData.fItemSize		= 0;
	fObjectData.fItemCount		= 0;

	// Reserved
	fObjectData.fReserved1		= kReservedConst;
	fObjectData.fReserved2		= kReservedConst;

	::memset( fObjectData.fDataBlock, 0, sizeof( fObjectData.fDataBlock ) );

	fTotalItems = 0;

	this->ChkCompilerStructAlignment();
} // CExpansion



//--------------------------------------------------------------------------------------------------
//	* ~CExpansion
//
//--------------------------------------------------------------------------------------------------

CExpansion::~CExpansion ( void )
{
} // ~CExpansion



void	CExpansion::ChkCompilerStructAlignment	( void )
{
	/*
		this dump was aquired by using the MPW command DumpSYM - TTE from the .xsym file produced
		from a Code Warrior Pro 1 Release Build of the Apple Mail Server for MacOS version 6.2
		
		this version of the structure is consider the "reference" standard for all future ASIP Mail Servers.
		If the structure does NOT match these offsets the version of the Mail Server that 
		you are compiling will _NOT_ be compatible with AppleShare IP 6.0, 6.1, or 6.2 (this also effects
		the AppleShare IP Mail Tool, and to a more limited extent the AppleShare IP Mail Admin since some
		of these structures are passed across the wire..)
		
		If _ANY_ of these Asserts "fire" you've got a backwards compatibility problem that _MUST_ be fixed!!
		
		this code wasn't put here for my personal amusement, this code was put here from a hard earned lesson
		that simple changes to headers/compilers _WILL_ break this stuff, better to know now than after we
		ship.  Look upon this code as a debugging AIDE, not an impediment.
		
		Type Name:  CExpandData		TINFO offset: 50884		(TINFO SYM file offset (hex): $1246C4)
			RecordOf 
			offset 0 NamedTypeOf fDBHeader TypeDef of "Type ID 248"
			offset 12 NamedTypeOf fObjectID 
			offset 16 NamedTypeOf fOwnerID unsigned long 
			offset 20 NamedTypeOf fPrevObjectID 
			offset 24 NamedTypeOf fNextObjectID unsigned long 
			offset 28 NamedTypeOf fItemCount 
			offset 30 NamedTypeOf fItemSize signed short 
			offset 32 NamedTypeOf fReserved1 
			offset 36 NamedTypeOf fReserved2 unsigned long 
			offset 40 NamedTypeOf fDataBlock 
			offset 168 NamedTypeOf fDBFooter TypeDef of "Type ID 250"
	*/

	static Boolean	aRunOnceFlag = false;	
	if (aRunOnceFlag == false)
	{
		CExpandData	*tempData = &(this->fObjectData);
		register unsigned long baseAddr = (unsigned long) tempData;

		//FileFormatSaftyChk(CExpandData, fDBHeader,		0);
		//FileFormatSaftyChk(CExpandData, fObjectID,		12);				// this is us (denormalized)
		//FileFormatSaftyChk(CExpandData, fOwnerID,		16);				// who ownes us
		//FileFormatSaftyChk(CExpandData, fPrevObjectID,	20);			// Link to previous object
		//FileFormatSaftyChk(CExpandData, fNextObjectID,	24);			// Link to next object
		//FileFormatSaftyChk(CExpandData, fItemCount,		28);
		//FileFormatSaftyChk(CExpandData, fItemSize,		30);
		//FileFormatSaftyChk(CExpandData, fReserved1,		32);
		//FileFormatSaftyChk(CExpandData, fReserved2,		36);
		//FileFormatSaftyChk(CExpandData, fDataBlock[0],	40);
		//FileFormatSaftyChk(CExpandData, fDBFooter,		168);
//		this->ReportBackwardCompatibility(sizeof(CExpandData),	172,	"CExpandData", "SIZEOF()");
//		this->ReportBackwardCompatibility(kExpansionDataSize,	172,	"CExpandData", "kExpansionDataSize");
	}
	
	aRunOnceFlag = true;
}

//--------------------------------------------------------------------------------------------------
//	* GetMyObjectData
//
//--------------------------------------------------------------------------------------------------

CExpandData* CExpansion::GetMyObjectData ( void )
{
	CExpandData* result = nil;
	if ( this != nil )
	{
		result = &( this->fObjectData );
	}

	return( result );

} // GetMyObjectData



//--------------------------------------------------------------------------------------------------
//	* SetDirty
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetDirty	( void )
{
	if ( (this != nil) && (this->GetMyObjectID() != 0) )
	{
		setDirty();
	}
} // SetDirty


#pragma mark -
#pragma mark * Getters/Setters

//--------------------------------------------------------------------------------------------------
//	** GetMyObjectID
//
//--------------------------------------------------------------------------------------------------

ObjID CExpansion::GetMyObjectID ( void )
{
	return( fObjectData.fObjectID );
} // GetMyObjectID



//--------------------------------------------------------------------------------------------------
//	* SetMyObjectID
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetMyObjectID ( ObjID inObjID )
{
	if ( fObjectData.fObjectID != inObjID )
	{
		fObjectData.fObjectID = inObjID;
		this->SetDirty();
	}
} // SetMyObjectID



//--------------------------------------------------------------------------------------------------
//	** GetOwnerID
//
//--------------------------------------------------------------------------------------------------

ObjID CExpansion::GetOwnerID ( void )
{
	return( fObjectData.fOwnerID );
} // GetOwnerID



//--------------------------------------------------------------------------------------------------
//	* SetOwnerID
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetOwnerID ( ObjID inObjID )
{
	if ( fObjectData.fOwnerID != inObjID )
	{
		fObjectData.fOwnerID = inObjID;
		this->SetDirty();
	}
} // SetOwnerID



//--------------------------------------------------------------------------------------------------
//	** GetPrevObjectID
//
//--------------------------------------------------------------------------------------------------

ObjID CExpansion::GetPrevObjectID ( void )
{
	return( fObjectData.fPrevObjectID );
} // GetPrevObjectID



//--------------------------------------------------------------------------------------------------
//	* SetPrevObjectID
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetPrevObjectID ( ObjID inObjID )
{
	if ( fObjectData.fPrevObjectID != inObjID )
	{
		fObjectData.fPrevObjectID = inObjID;
		this->SetDirty();
	}
} // SetPrevObjectID



//--------------------------------------------------------------------------------------------------
//	** GetNextObjectID
//
//--------------------------------------------------------------------------------------------------

ObjID CExpansion::GetNextObjectID ( void )
{
	return( fObjectData.fNextObjectID );
} // GetNextObjectID



//--------------------------------------------------------------------------------------------------
//	* SetNextObjectID
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetNextObjectID ( ObjID inObjID )
{
	if ( fObjectData.fNextObjectID != inObjID )
	{
		fObjectData.fNextObjectID = inObjID;
		this->SetDirty();
	}
} // SetNextObjectID



//--------------------------------------------------------------------------------------------------
//	** GetItemCount
//
//--------------------------------------------------------------------------------------------------

sInt16 CExpansion::GetItemCount ( void )
{
	return( fObjectData.fItemCount );
} // GetItemCount



//--------------------------------------------------------------------------------------------------
//	* SetItemCount
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetItemCount ( sInt16 inCount )
{
	if ( fObjectData.fItemCount != inCount )
	{
		fObjectData.fItemCount = inCount;
		this->SetDirty();
	}
} // SetItemCount



//--------------------------------------------------------------------------------------------------
//	* IncItemCount
//
//--------------------------------------------------------------------------------------------------

void CExpansion::IncItemCount ( void )
{
	if ( fObjectData.fItemCount < GetMaxItems() )
	{
		fObjectData.fItemCount++;
		this->SetDirty();
	}
} // IncItemCount



//--------------------------------------------------------------------------------------------------
//	* DecItemCount
//
//--------------------------------------------------------------------------------------------------

void CExpansion::DecItemCount ( void )
{
	if ( fObjectData.fItemCount > 0 )
	{
		fObjectData.fItemCount--;
		this->SetDirty();
	}
} // DecItemCount



//--------------------------------------------------------------------------------------------------
//	* GetItemSize
//
//--------------------------------------------------------------------------------------------------

sInt16 CExpansion::GetItemSize ( void )
{
	return( fObjectData.fItemSize );
} // GetItemSize



//--------------------------------------------------------------------------------------------------
//	* SetItemSize
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetItemSize ( sInt16 inSize )
{
	if ( (fObjectData.fItemSize != inSize) && ( inSize <= kMaxDataBlockSize) )
	{
		fObjectData.fItemSize = inSize;
		this->SetMaxItems();
		this->SetDirty();
	}
} // SetItemSize


//--------------------------------------------------------------------------------------------------
//	* GetMaxItems
//
//--------------------------------------------------------------------------------------------------

sInt16 CExpansion::GetMaxItems ( void )
{
	return( fTotalItems );
} // GetMaxItems



//--------------------------------------------------------------------------------------------------
//	* SetMaxItems
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetMaxItems ( void )
{
	sInt16	total	= 0;
	sInt16	size	= this->GetItemSize();

	if ( size > 0 )
	{
		total = kMaxDataBlockSize / size;
	}

	if ( fTotalItems != total )
	{
		fTotalItems = total;
		this->SetDirty();
	}
} // SetMaxItems




//--------------------------------------------------------------------------------------------------
//	* GetDataBlock
//
//--------------------------------------------------------------------------------------------------

void CExpansion::GetDataBlock ( char *outData )
{
	::memcpy( outData, &fObjectData.fDataBlock, kMaxDataBlockSize );
} // GetDataBlock



//--------------------------------------------------------------------------------------------------
//	* SetDataBlock
//
//--------------------------------------------------------------------------------------------------

void CExpansion::SetDataBlock ( char *inData )
{
	::memcpy( &fObjectData.fDataBlock, inData, kMaxDataBlockSize );
	this->SetDirty();
} // SetDataBlock



