/*
	$Id: CDBBaseObject.10.1.cpp,v 1.1 2003/04/20 23:32:27 dasenbro Exp $

	File:		CDBObject.cpp

	Contains:	C++ implementation of basic persistent database object

	Written by:	Nick Brosnahan

	Copyright:	 1996-2001 by Apple Computer, Inc., all rights reserved.

 	NOT_FOR_OPEN_SOURCE <to be reevaluated at a later time>

	Change History:

		$Log: CDBBaseObject.10.1.cpp,v $
		Revision 1.1  2003/04/20 23:32:27  dasenbro
		Initial check-in.
		
		Revision 1.16  2001/06/21 20:50:53  dasenbro
		Updated file header info.
		
		Revision 1.15  2001/06/21 17:01:16  dasenbro
		Added Change History.
		

	Projector History:

	  <ASX6>	  6/7/99	DOR		Change some Assert/If pairs to BaseDBAssert_IF...
	  <ASX5>	  6/7/99	DOR		Use "this" rather than globals, t'is better programming.
	  <ASX4>	  6/4/99	DOR		Add check compiler struct alignment routine.
	 <ASM10>	 1/29/99	DOR		We look the cache to prevent race conditions in writing objects
									to disk in CDBObject::Done.  When trying to grab a semaphore the
									retry loop should be a while rather than an if...{sigh}
		 <9>	  6/4/98	DOR		Add an Assert.
		 <8>	 5/26/98	DOR		Tighten up some code and do some better boundary checking.
		 <7>	03/23/98	DOR		Add support for Get/Release ObjectLock, and remove support for
									Get/Set/Clear OwnerThreadID...
		 <6>	 3/15/98	MED		Allocate the cache object if it doesn't exist.
		 <5>	 3/14/98	DOR		Add New Classes for list mgmt. there are now multiple cache
									lists to relieve search times...and close up some race
									conditions.
		 <4>	 3/12/98	DOR		Minor changes.
		 <2>	 3/11/98	MED		Changed setDirty to take a force write flag and also changed the
									was Done deals with a 1 ref. count.
		 <1>	  3/9/98	DOR		first checked in
		 <8>	  3/9/98	DOR		Add support for OwnerThreadID.
		 <7>	  3/9/98	DOR		Add support for Get/Set/Clear OwnerThreadID as a feature of
									CDBBaseObject_10_1.
		 <6>	 2/27/98	DOR		Lock the list while we are writting things to the database.
		 <5>	 2/26/98	DOR		Make sure to release semaphores that we "lock".
		 <4>	 2/26/98	DOR		Clean up some race conditions in the CDBObjectDone area...

	To Do:
*/

#include <time.h>

#include "CBaseDatabase.10.1.h"
#include "CDBBaseObject.10.1.h"

// ----------------------------------------------------------------------------------------
#pragma mark * Static Methods *
// ----------------------------------------------------------------------------------------

CSmartTimer_10_1::CSmartTimer_10_1 (uInt32 inTime2Wait)
{
	this->ResetTimer(inTime2Wait);
}

CSmartTimer_10_1::~CSmartTimer_10_1 ()
{
	fStartTime = 0xffffffff;
	fTime2Wait = fStartTime;
	fCount = 0;
}

void CSmartTimer_10_1::ResetTimer(uInt32 inTime2Wait)
{
//	fStartTime = ::TickCount();
	fStartTime = ::time( NULL );
	fTime2Wait = inTime2Wait;
	fCount = 0;
}

Boolean CSmartTimer_10_1::KeepWaiting (void)
{
	if (fCount == 0)
	{
		this->ResetTimer(this->fTime2Wait);
		fCount++;
		return (true);
	}

	if ((fTime2Wait == 0) && (fCount == 1))
	{
		return false;
	}

	if (fCount == 1)
	{
		fCount++;
		return true;
	}

	return ( (fStartTime + fTime2Wait) > (uTime_t)::time( NULL ) );
//	return ( (fStartTime + fTime2Wait) > ::TickCount() );
}

#pragma mark -

CDBBaseObjectCacheList_10_1::CDBBaseObjectCacheList_10_1()
{
	fListHead		= NULL;
	fListCount		= 0;
	fListMaxCount	= 0;
}

CDBBaseObjectCacheList_10_1::~CDBBaseObjectCacheList_10_1()
{
	BaseDBAssert( fListHead == NULL );
	BaseDBAssert( fListCount == 0 );
}

CDBBaseObject_10_1* CDBBaseObjectCacheList_10_1::AddObject		( CDBBaseObject_10_1 *inObject )
{
	CDBBaseObject_10_1	*tempObj		= NULL;
	CDBBaseObject_10_1	*aReallyTempObj	= NULL;

	BaseDBAssert_if (this != NULL)
	{
		BaseDBAssert(inObject != NULL);
		BaseDBAssert(inObject->GetObjectRefCount() == 1);	// we shouldn't be adding objects with more than
														// one reference...

		this->DoListLock();

		tempObj = this->FindObject(inObject->GetObjectID());	// make sure some hasn't already added the "same" object..
																// while we were "waiting"...

		if (tempObj == NULL)
		{	// no duplicate found....
			// let's add this object to the list....

			tempObj = this->GetListHead();
			if (tempObj != NULL)
			{
				tempObj->SetPrevPtr(inObject);
			}

			inObject->SetNextPtr(tempObj);
			inObject->SetPrevPtr(NULL);
			this->SetListHead(inObject);
			this->SetListCount(this->GetListCount() + 1);
			if (this->GetListCount() > this->GetListMax())
			{
				this->SetListMax(this->GetListCount());
			}

			tempObj = inObject;
		}
		else if (tempObj != inObject)
		{
			BaseDBAssert_if (tempObj->GetObjectRefCount() != 0)
			{	// ok this is tricky...but with all the waits & signals in this code, there is a high
				// probability that we've yielded at some point during this code path execution...
				// therefore while we've been working out way upto adding "inObject" another thread
				// managed to slip in and add the same object to the DBCacheList, and we just tried to
				// "add" it again...{sigh} this is bad...but we've detected the case.

				// We'll go ahead and "Use" the object already on the list, this will prevent it from being
				// deleted in case the "Signal/Release" below yields, and the thread that owns it calls "Done"...
				// we are relying on the fact that "Use" doesn't yield in the case where refcount > zero...

				aReallyTempObj = tempObj->Use(tempObj->GetOwnerDBPtr());
				BaseDBAssert(aReallyTempObj == tempObj);	// this had better work!!! or else we're screwed!!
			}
		}
		else
		{
			BaseDBAssert(tempObj != inObject);	// force a user break...this is bad, because we just found
												// our selves on the list, but we haven't added ourselves
												// yet...{sigh}...this is a _VERY_ bad situation, and will
												// need to be carefully debugged...we have a _SERIOUS_
												// race condition problem....
		}

		this->DoListRelease();
	}

	return (tempObj);
}

Boolean CDBBaseObjectCacheList_10_1::RemoveObject	( CDBBaseObject_10_1 *inObject )
{
	Boolean		result = false;
	CDBBaseObject_10_1	*tempObj = NULL;

	BaseDBAssert_if (this != NULL)
	{
		BaseDBAssert(inObject != NULL);
		this->DoListLock();

		if (inObject->GetObjectRefCount() == 1)
		{
			if (this->GetListHead() == inObject)
			{
				this->SetListHead(inObject->GetNextPtr());
			}

			tempObj = inObject->GetPrevPtr();
			if (tempObj != NULL)
			{
				tempObj->SetNextPtr(inObject->GetNextPtr());
				tempObj = NULL;
			}

			tempObj = inObject->GetNextPtr();
			if (tempObj != NULL)
			{
				tempObj->SetPrevPtr(inObject->GetPrevPtr());
				tempObj = NULL;
			}

			inObject->SetNextPtr(NULL);
			inObject->SetPrevPtr(NULL);
			this->SetListCount(this->GetListCount() - 1);

			result = true;
		}

		this->DoListRelease();
	}

	return (result);
}

CDBBaseObject_10_1* CDBBaseObjectCacheList_10_1::GetListHead ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return ((CDBBaseObject_10_1 *)this->fListHead);
	}

	return (NULL);
}

void	CDBBaseObjectCacheList_10_1::SetListHead ( CDBBaseObject_10_1* inListHead )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fListHead = inListHead;
	}
}

uInt32	CDBBaseObjectCacheList_10_1::GetListCount ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->fListCount);
	}

	return (0);
}

void	CDBBaseObjectCacheList_10_1::SetListCount ( const uInt32 inListCount )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fListCount = inListCount;
	}
}

uInt32	CDBBaseObjectCacheList_10_1::GetListMax ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->fListMaxCount);
	}

	return (0);
}

void	CDBBaseObjectCacheList_10_1::SetListMax ( const uInt32 inListMax )
{
	BaseDBAssert_if (this != NULL)
	{
		BaseDBAssert(inListMax > this->GetListMax());
		this->fListMaxCount = inListMax;
	}
}

CDBBaseObject_10_1* CDBBaseObjectCacheList_10_1::FindObject ( const ObjID inObjID )
{
	CDBBaseObject_10_1 *aCDBObject = NULL;
	CDBBaseObject_10_1 *tempObject = NULL;

	BaseDBAssert_if ((this != NULL) && (inObjID != 0))
	{
		this->DoListLock();

		aCDBObject = this->GetListHead();
		while (aCDBObject != NULL)
		{
			if (aCDBObject->GetObjectID() == inObjID)
			{
				BaseDBAssert(aCDBObject->GetObjectRefCount() != 0);
				tempObject = aCDBObject->Use( aCDBObject->GetOwnerDBPtr() );
				BaseDBAssert(tempObject == aCDBObject);	// this had better work!!!
				break;
			}

			aCDBObject = aCDBObject->GetNextPtr();
		}


		this->DoListRelease();
	}

	return aCDBObject;
}

void CDBBaseObjectCacheList_10_1::FlushList ( const Boolean inWakeupThreads )
{
	CDBBaseObject_10_1  *aCDBObject	= NULL;
	OSErr			result		= kNoErr;

	BaseDBAssert_if (this != NULL)
	{
		try
		{
			this->DoListLock();

			aCDBObject = this->GetListHead();
			while ( aCDBObject != NULL )
			{
				if ( aCDBObject->GetDirtyFlag() == true )
				{
					aCDBObject->SetDirtyFlag( false );
					result = aCDBObject->WriteObject();
				}

				if ( inWakeupThreads ==  true )
				{
					aCDBObject->WakeUpObjectThreads();
				}
				aCDBObject = aCDBObject->GetNextPtr();
			}
			this->DoListRelease();
		}

		catch ( ExceptionCode err )
		{
			this->DoListRelease();
		}
	}
}


void CDBBaseObjectCacheList_10_1::DoListLock ( void )
{
}


void CDBBaseObjectCacheList_10_1::DoListRelease ( void )
{
}


#pragma mark -

CDBBaseObjectCache_10_1::CDBBaseObjectCache_10_1( CBaseDatabase_10_1 *inBaseDBPtr )
{
	BaseDBAssert_if (this != NULL)
	{
		fCacheList = NULL;
		fCacheListCount = 0;
		for (uInt32 i = 0; i < kCacheHashTableSize_10_1; ++i)
		{
			fHashTable[i] = NULL;
		}
		SetBaseDBPtr(inBaseDBPtr);
	}
}

CDBBaseObjectCache_10_1::~CDBBaseObjectCache_10_1()
{
	BaseDBAssert_if (this != NULL)
	{
		TypeToIndexMapping_10_1* mappingPtr = NULL;
		TypeToIndexMapping_10_1* nextPtr = NULL;

		if (fCacheList != NULL)
		{
			delete [] fCacheList;
		}
		for (uInt32 i = 0; i < kCacheHashTableSize_10_1; ++i)
		{
			mappingPtr = fHashTable[i];
			while (mappingPtr != NULL)
			{
				nextPtr = mappingPtr->fNextPtr;
				delete mappingPtr;
				mappingPtr = nextPtr;
			}
		}
	}
}

void CDBBaseObjectCache_10_1::SetBaseDBPtr( CBaseDatabase_10_1 *inBaseDBPtr )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fBaseDBPtr = inBaseDBPtr;
		this->SetUpHashTable();
	}
}

void CDBBaseObjectCache_10_1::SetUpHashTable	( )
{
	BaseDBAssert_if (this != NULL)
	{
		// tear down existing hash table

		TypeToIndexMapping_10_1* mappingPtr = NULL;
		TypeToIndexMapping_10_1* nextPtr = NULL;

		if (fCacheList != NULL)
		{
			delete [] fCacheList;
		}
		for (uInt32 i = 0; i < kCacheHashTableSize_10_1; ++i)
		{
			mappingPtr = fHashTable[i];
			while (mappingPtr != NULL)
			{
				nextPtr = mappingPtr->fNextPtr;
				delete mappingPtr;
				mappingPtr = nextPtr;
			}
			fHashTable[i] = NULL;
		}

		// set up new hash table
		if ( fBaseDBPtr != NULL )
		{
			// We need to interrogate the database in order to set up a hash table to
			// map object types to cache lists. There should be one cache list per
			// object type so that we don't get weird deadlock situations from collisions.
			SObjTypeInfo_10_1 objTypeInfo;
			uInt32 objTypeCount = 0;
			uInt32 hashIndex = 0;
			uInt32 i = 0;

			objTypeCount = fBaseDBPtr->GetObjTypeCount();
			fCacheList = new CDBBaseObjectCacheList_10_1[objTypeCount];
			fCacheListCount = objTypeCount;
			for (i = 0; i < objTypeCount; ++i)
			{
				TypeToIndexMapping_10_1* typeMapping = new TypeToIndexMapping_10_1;
				BaseDBAssert_if (typeMapping != NULL )
				{
					fBaseDBPtr->GetObjTypeInfo(i, objTypeInfo);
					hashIndex = objTypeInfo.fObjType % kCacheHashTableSize_10_1;

					typeMapping->fObjType = objTypeInfo.fObjType;
					typeMapping->fCacheListIndex = i;
					typeMapping->fNextPtr = fHashTable[hashIndex];

					fHashTable[hashIndex] = typeMapping;
				}
			}
		}
	}
}

CBaseDatabase_10_1	*CDBBaseObjectCache_10_1::GetBaseDBPtr()
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->fBaseDBPtr);
	}

	return (NULL);
}

CDBBaseObject_10_1* CDBBaseObjectCache_10_1::AddObject		( CDBBaseObject_10_1 *inObject )
{
	CDBBaseObject_10_1* 	aTempObj = NULL;
	uInt32		aListIndex = 0;

	BaseDBAssert_if (this != NULL)
	{
		aListIndex = this->DoListHash(inObject->GetObjectType(), inObject->GetObjectID());

		this->Wait(aListIndex);

		aTempObj = fCacheList[aListIndex].AddObject(inObject);

		this->Signal(aListIndex);
	}

	return (aTempObj);
}

Boolean CDBBaseObjectCache_10_1::RemoveObject ( CDBBaseObject_10_1 *inObject )
{
	uInt32	aListIndex = 0;
	Boolean	result = false;
	BaseDBAssert_if (this != NULL)
	{
		aListIndex = this->DoListHash( inObject->GetObjectType(), inObject->GetObjectID() );
		this->Wait( aListIndex );
		result = fCacheList[aListIndex].RemoveObject(inObject);
		this->Signal(aListIndex);
	}

	return (result);
}

CDBBaseObject_10_1*	CDBBaseObjectCache_10_1::FindObject ( const ObjID inObjectID )
{
	CDBBaseObject_10_1	*anObject	= NULL;
	uInt32		aListIndex	= 0;

	BaseDBAssert_if (this != NULL)
	{
		for (aListIndex = 0; aListIndex < this->GetListCount(); aListIndex++)
		{
			this->Wait(aListIndex);

			anObject = fCacheList[aListIndex].FindObject(inObjectID);
			if (anObject != NULL)
			{
				break;
			}

			this->Signal(aListIndex);
		}
	}

	return (anObject);
}


CDBBaseObject_10_1*	CDBBaseObjectCache_10_1::FindObject ( const OSType inObjectType, const ObjID inObjectID )
{
	CDBBaseObject_10_1	*anObject	= NULL;
	uInt32		aListIndex	= 0;

	BaseDBAssert_if (this != NULL)
	{
		aListIndex = this->DoListHash( inObjectType, inObjectID );
		this->Wait( aListIndex );
		anObject = fCacheList[ aListIndex ].FindObject( inObjectID );
		this->Signal( aListIndex );
	}

	return (anObject);
}

uInt32	CDBBaseObjectCache_10_1::GetListCount		( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return fCacheListCount;
	}

	return (0);
}

void CDBBaseObjectCache_10_1::FlushAllList ( const Boolean inWakeupThreads )
{
	BaseDBAssert_if (this != NULL)
	{
		this->Flush( kAllListFlushValue_10_1, inWakeupThreads );
	}
}


void CDBBaseObjectCache_10_1::FlushList ( const OSType inObjectType, const Boolean inWakeupThreads )
{
	BaseDBAssert_if ( this != NULL )
	{
		this->Flush( inObjectType, inWakeupThreads );
	}
}


void CDBBaseObjectCache_10_1::Flush ( const OSType inObjectType, const Boolean inWakeupThreads )
{
	uInt32		aListIndex	= 0;

	BaseDBAssert_if( this != NULL )
	{
		for ( aListIndex = 0; aListIndex < this->GetListCount(); aListIndex++ )
		{
			this->Wait( aListIndex );

			if ( (inObjectType == kAllListFlushValue_10_1) || (aListIndex == (this->DoListHash( inObjectType, 0 ))) )
			{
				fCacheList[ aListIndex ].FlushList( inWakeupThreads );
			}

			this->Signal( aListIndex );
		}
	}
}

CDBBaseObject_10_1* CDBBaseObjectCache_10_1::AddObject2List ( CDBBaseObject_10_1 *inObject )
{
	CDBBaseObject_10_1* aTempObj = NULL;

	if (this != NULL)
	{
		 aTempObj = this->AddObject(inObject);
	}

	return aTempObj;
}

Boolean	CDBBaseObjectCache_10_1::RemoveObjFromList	( CDBBaseObject_10_1 *inObject )
{
	if ( this != NULL )
	{
		return (this->RemoveObject(inObject));
	}

	return (false);
}

CDBBaseObject_10_1*	CDBBaseObjectCache_10_1::Search4Object	( const ObjID inObjectID )
{
	CDBBaseObject_10_1*	aCDBObject = NULL;

	if (this != NULL)
	{
		aCDBObject = this->FindObject(inObjectID);
	}

	return (aCDBObject);
}


CDBBaseObject_10_1*	CDBBaseObjectCache_10_1::Search4Object ( const OSType inObjectType, const ObjID inObjectID )
{
	CDBBaseObject_10_1*	aCDBObject = NULL;

	if (this != NULL)
	{
		aCDBObject = this->FindObject( inObjectType, inObjectID );
	}

	return (aCDBObject);
}


void CDBBaseObjectCache_10_1::LockTheCache ( const OSType inObjectType )
{
	uInt32	aListIndex = 0;

	if ( this != NULL )
	{
		aListIndex = this->DoListHash( inObjectType, 0 );
		this->Wait(aListIndex);
	}
}

void CDBBaseObjectCache_10_1::UnLockTheCache ( const OSType inObjectType )
{
	uInt32	aListIndex = 0;

	if (this != NULL)
	{
		aListIndex = this->DoListHash(inObjectType, 0);
		this->Signal(aListIndex);
	}
}

void CDBBaseObjectCache_10_1::Wait ( const uInt32 inIndex )
{
#ifndef DEBUG
	#pragma unused (inIndex)
#endif

	BaseDBAssert(this != NULL);
	BaseDBAssert(inIndex < this->GetListCount());
	BaseDBAssert(false);
}

void CDBBaseObjectCache_10_1::Signal ( const uInt32 inIndex )
{
#ifndef DEBUG
	#pragma unused (inIndex)
#endif

	BaseDBAssert(this != NULL);
	BaseDBAssert(inIndex < this->GetListCount());
	BaseDBAssert(false);
}

uInt32 CDBBaseObjectCache_10_1::DoListHash ( const OSType inObjectType, const ObjID )
{
	uInt32	anIndex = 0;

	BaseDBAssert_if ( this != NULL )
	{
		TypeToIndexMapping_10_1* mappingPtr = fHashTable[ inObjectType % kCacheHashTableSize_10_1 ];
		while ( mappingPtr != NULL )
		{
			if ( mappingPtr->fObjType == inObjectType )
			{
				// found it
				anIndex = mappingPtr->fCacheListIndex;
				break;
			}
			else
			{
				mappingPtr = mappingPtr->fNextPtr;
			}
		}
	}

	BaseDBAssert( anIndex < this->GetListCount() );

	return( anIndex );
}

uInt32 CDBBaseObjectCache_10_1::CalcObjectCount ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->CalcCacheObjCount());
	}

	return (0);
}

uInt32 CDBBaseObjectCache_10_1::CalcCacheObjCount ( void )
{
	uInt32	aListIndex	= 0;
	uInt32	aCount		= 0;

	if (this != NULL)
	{
		for (aListIndex = 0; aListIndex < this->GetListCount(); aListIndex++)
		{
			aCount += fCacheList[aListIndex].GetListCount();
		}
	}

	return (aCount);
}

#pragma mark -

CDBBaseObject_10_1::CDBBaseObject_10_1 (	StdDBObjHeader_10_1	*inObjHeader,
				 	  			StdDBObjFooter_10_1	*inObjFooter,
				 	   			void			*inObjData,
				 	   			const OSType	inObjSignature,
				 	   			const uInt32	inObjVersion,
				 	   			const uInt32	inObjSize 		)
{
	fRefCount		= 0;
	fNextPtr		= NULL;
	fPrevPtr		= NULL;
	fDirty			= false;

	BaseDBAssert(inObjHeader != NULL);
	BaseDBAssert(inObjFooter != NULL);
	BaseDBAssert(inObjData != NULL);

        BaseDBAssert(inObjSize <= kMaxIDTableSize);
        
	fObjHeader = inObjHeader;
	fObjFooter = inObjFooter;
	fObjData   = inObjData;

	if (fObjHeader != NULL)
	{
		fObjHeader->fSignature	= inObjSignature;
		fObjHeader->fVersion	= inObjVersion;
		fObjHeader->fSize		= inObjSize;
	}

	if (fObjFooter != NULL)
	{
		fObjFooter->fSignature	= inObjSignature;
	}
}

CDBBaseObject_10_1::~CDBBaseObject_10_1 ( void )
{
	BaseDBAssert(fRefCount 		== 0);
	BaseDBAssert(fNextPtr 		== NULL);
	BaseDBAssert(fPrevPtr 		== NULL);
	BaseDBAssert(fDirty 		== false);
	BaseDBAssert(fObjHeader		!= NULL);
	BaseDBAssert(fObjFooter		!= NULL);
	BaseDBAssert(fObjData		!= NULL);
}

void CDBBaseObject_10_1::ChkCompilerStructAlignment ( void )
{
}

CDBBaseObject_10_1* CDBBaseObject_10_1::UniversalFindByID	(	CBaseDatabase_10_1		*inBaseDB,
													const ObjID			inObjID,
													const OSType		inObjType,
													const uInt32		inObjVersion,
													const uInt32		inObjSize,
													ObjAllocFuncPtr_10_1		inObjAllocFuncPtr	)
{
	OSStatus	result;
	CDBBaseObject_10_1	*theObject	= NULL;
	CDBBaseObject_10_1	*tempObject	= NULL;

	if (inObjID == 0)
	{
		return (NULL);
	}

	try
	{
		(inBaseDB->GetBaseObjectCachePtr())->LockTheCache(inObjType);

		theObject = (inBaseDB->GetBaseObjectCachePtr())->Search4Object(inObjType, inObjID); 	// danger, danger, potential yield here!!
		if (theObject != NULL)
		{
			// We may find an object in cache that matches the inAccountID. if this is the
			//	case, we just return null.  Therefor, there is no need for this assert...
			// BaseDBAssert(theAccount->fAccountData.fDBHeader.fSignature == kAccountSignature);

			if ((theObject->GetObjectType() != inObjType) ||
				(theObject->GetObjectSize() != inObjSize) ||
				(theObject->GetObjectVersion() != inObjVersion) )
			{
				theObject->Done( theObject );
				theObject = NULL;
			}
		}
		else if (theObject == NULL)
		{
			theObject = inObjAllocFuncPtr();
			BaseDBAssert(theObject != NULL);
			ThrowIfNULL_(theObject);

			result = inBaseDB->GetObjData( inObjType, inObjID, theObject->GetObjectData() );	// danger, danger, potential yield here!!
			if (result != kNoErr)
			{
				delete theObject;
				theObject = NULL;
			}
			else
			{
				BaseDBAssert(theObject->GetObjectID()		== inObjID);
				BaseDBAssert(theObject->GetObjectVersion()	== inObjVersion);
				BaseDBAssert(theObject->GetObjectType()		== inObjType);
				BaseDBAssert(theObject->GetObjectSize()		== inObjSize);

				if ( (theObject->GetObjectType() 	!= inObjType) ||
					 (theObject->GetObjectVersion() != inObjVersion) ||
					 (theObject->GetObjectSize()	!= inObjSize))
				{
					delete theObject;
					theObject = NULL;
				}

				if ( theObject != NULL )
				{
					tempObject = theObject->Use(inBaseDB);	// danger, danger, potential yield here!!
					if (tempObject != theObject)
					{	// hmm the object got added to the list while we were futzing around in the
						// Database...one of those "danger" "danger" points caused us grief...
						// oh well we detected the case and now can fix it...
						// Use has incremented the use count of the object it "returned"
						// therefore all we need to do now is Dispose of the new copy we created..
						// and return the object that "use" found...

						delete theObject;
						theObject = NULL;

						// hold this "newly" found object to the "same" standards as the other object...
						BaseDBAssert(tempObject->GetObjectID()		== inObjID);
						BaseDBAssert(tempObject->GetObjectVersion()	== inObjVersion);
						BaseDBAssert(tempObject->GetObjectType()		== inObjType);
						BaseDBAssert(tempObject->GetObjectSize()		== inObjSize);
						if ( (tempObject->GetObjectType() 	!= inObjType) ||
							 (tempObject->GetObjectVersion() != inObjVersion) ||
					 		 (tempObject->GetObjectSize()	!= inObjSize))
					 	{
					 		// we're _VERY_ confused now, safest thing is to just return nothing....
					 		tempObject->Done(tempObject);
					 	}
					 	else
					 	{
							theObject = tempObject;
							tempObject = NULL;
					 	}
					}
				}
			}
		}

		(inBaseDB->GetBaseObjectCachePtr())->UnLockTheCache(inObjType);
	}
	catch ( ExceptionCode err )
	{
		(inBaseDB->GetBaseObjectCachePtr())->UnLockTheCache(inObjType);

		if ( theObject != NULL )
		{
			delete theObject;
			theObject = NULL;
		}
	}

	return( theObject );
}

CBaseDatabase_10_1 *CDBBaseObject_10_1::GetOwnerDBPtr ( void )
{
	BaseDBAssert(this != NULL);
	BaseDBAssert(fObjHeader != NULL);
	BaseDBAssert(fObjFooter != NULL);
	BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);
	BaseDBAssert(this->fOwnerDBPtr != NULL);

	return ((CBaseDatabase_10_1 *)this->fOwnerDBPtr);
}

OSType	CDBBaseObject_10_1::GetObjectType ( void )
{
	BaseDBAssert(fObjHeader != NULL);
	BaseDBAssert(fObjFooter != NULL);
	BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);

	return (fObjHeader->fSignature);
}

uInt32	CDBBaseObject_10_1::GetObjectVersion ( void )
{
	BaseDBAssert(fObjHeader != NULL);
	BaseDBAssert(fObjFooter != NULL);
	BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);

	return (fObjHeader->fVersion);
}

uInt32	CDBBaseObject_10_1::GetObjectSize ( void )
{
	BaseDBAssert(fObjHeader != NULL);
	BaseDBAssert(fObjFooter != NULL);
	BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);

	return (fObjHeader->fSize);
}

void *	CDBBaseObject_10_1::GetObjectData ( void )
{
	BaseDBAssert(fObjHeader != NULL);
	BaseDBAssert(fObjFooter != NULL);
	BaseDBAssert(fObjData != NULL);
	BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);

	return (fObjData);
}

void	CDBBaseObject_10_1::SetNextPtr	( CDBBaseObject_10_1* inNextPtr )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fNextPtr = inNextPtr;
	}
}

CDBBaseObject_10_1 *	CDBBaseObject_10_1::GetNextPtr	(void)
{
	BaseDBAssert_if (this != NULL)
	{
		return ((CDBBaseObject_10_1 *)this->fNextPtr);
	}

	return NULL;
}

void	CDBBaseObject_10_1::SetPrevPtr	( CDBBaseObject_10_1* inPrevPtr )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fPrevPtr = inPrevPtr;
	}
}

CDBBaseObject_10_1 *	CDBBaseObject_10_1::GetPrevPtr	(void)
{
	BaseDBAssert_if (this != NULL)
	{
		return ((CDBBaseObject_10_1 *)this->fPrevPtr);
	}

	return NULL;
}

uInt32	CDBBaseObject_10_1::GetObjectRefCount	( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->fRefCount);
	}

	return 0;
}

void	CDBBaseObject_10_1::SetObjectRefCount	( const uInt32 inObjectRefCount )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fRefCount = inObjectRefCount;
	}
}

void CDBBaseObject_10_1::AddObjectReference ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		this->SetObjectRefCount(this->GetObjectRefCount() + 1);
	}
}

void CDBBaseObject_10_1::RemoveObjectReference ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		BaseDBAssert_if (this->GetObjectRefCount() != 0)
		{
			this->SetObjectRefCount(this->GetObjectRefCount() - 1);
		}
	}
}

CDBBaseObject_10_1* CDBBaseObject_10_1::Use ( CBaseDatabase_10_1	*inBaseDB )
{
	CDBBaseObject_10_1*	tempObject = NULL;

	BaseDBAssert_if ( this != NULL )
	{
		if (this->GetObjectRefCount() == 0)
		{
			this->AddObjectReference();
			this->fOwnerDBPtr = inBaseDB;
			tempObject = (inBaseDB->GetBaseObjectCachePtr())->AddObject2List(this);	// danger, danger, yield potential...{sigh}

			if (tempObject != this)
			{	// hmmm, we found an object with the same ID already on the list...
				// it must've been add while we were blocked on one of the list semaphores
				// if tempObject != this...that means we didn't successfully add "this" to the
				// CDBBaseObjectCache_10_1....and as a courtesy AddObject2List called use on the duplicate
				// already in the list... so we're done...we'll let who ever called "use" sort it out..

				// since we didn't add the object to the list...we should remove the reference that we've
				// added...
				this->RemoveObjectReference();
			}
			else if (tempObject == this)
			{
				tempObject = this;
			}
		}
		else
		{
			this->AddObjectReference();
			BaseDBAssert(this->fOwnerDBPtr == inBaseDB);
			tempObject = this;
		}
	}

	return (tempObject);
} // setDirty

void CDBBaseObject_10_1::Done ( CDBBaseObject_10_1* &inPtr )
{
	OSType	anObjType = NULL;
	CBaseDatabase_10_1 *anOwnerDBPtr = NULL;
	CDBBaseObjectCache_10_1 *aCachePtr = NULL;

	// CDBBaseObject_10_1 is _VERY_ tricky code, do _NOT_ modify it unless you fully understand
	// all possible race conditions....code is commented with "danger, danger" around areas.
	// that cause potential yields/blockage......after passing one of these code sections...
	// you _MUST_ make sure that your changes don't blow away an object that may have been
	// referenced by another thread while you were blocked....

	BaseDBAssert( this == inPtr );
	if (inPtr != NULL)
	{
		anOwnerDBPtr = this->GetOwnerDBPtr();
		aCachePtr = anOwnerDBPtr->GetBaseObjectCachePtr();
		anObjType = this->GetObjectType();

		aCachePtr->LockTheCache(anObjType);

		BaseDBAssert( this->GetObjectRefCount() != 0 );
		if ( (this->GetObjectRefCount() - 1) == 0 ) 
		{
			if (this->GetDirtyFlag() == true)
			{
				this->SetDirtyFlag(false);
				this->setDirty(true);	// danger, danger, potential Yield HERE!!!
			}

			if (aCachePtr->RemoveObjFromList(this))
			{
				this->RemoveObjectReference();
				BaseDBAssert(this->GetObjectRefCount() == 0);

				delete inPtr;
				inPtr = NULL;
			}
			else
			{	// we yielded at some point, and someone has "re-referenced the object
				// while we were ditzing around...oh well, we'll still decrement the
				// refcount, we just won't remove it from the list...
				BaseDBAssert((this->GetObjectRefCount() - 1) != 0);
				this->RemoveObjectReference();
			}
		}
		else
		{
			this->RemoveObjectReference();
		}

		aCachePtr->UnLockTheCache(anObjType);
	}

	// always NULL out the inPtr...
	inPtr = NULL;
} // Done

Boolean CDBBaseObject_10_1::GetDirtyFlag ( void )
{
	BaseDBAssert_if (this != NULL)
	{
		return (this->fDirty);
	}

	return false;
}

void CDBBaseObject_10_1::SetDirtyFlag ( const Boolean inDirtyFlag )
{
	BaseDBAssert_if (this != NULL)
	{
		this->fDirty = inDirtyFlag;
	}
}


OSErr CDBBaseObject_10_1::WriteObject ( void )
{
	OSErr	result = -1;

	BaseDBAssert_if( (this != NULL) && (this->fOwnerDBPtr != NULL) )
	{
		BaseDBAssert( this->GetObjectID() != 0 );
		BaseDBAssert( this->GetObjectData() != NULL );
		BaseDBAssert( this->fObjHeader	!= NULL );
		BaseDBAssert( this->fObjFooter	!= NULL );
		BaseDBAssert( this->fObjData	!= NULL );
		BaseDBAssert( this->fObjHeader->fSignature == this->fObjFooter->fSignature );

		result = ( (this->GetOwnerDBPtr())->SetObjData( this->GetObjectType(),
														this->GetObjectID(),
														this->GetObjectData()));	// this can yield....
	}

	return (result);
}

//--------------------------------------------------------------------------------------------------
//	* GetObjectLock
//
//--------------------------------------------------------------------------------------------------

ExceptionCode CDBBaseObject_10_1::GetObjectLock ( const Boolean inWaitForObjectLock,
										 const uInt32  inMaxWaitTime )
{
	#pragma unused (inWaitForObjectLock)
	#pragma unused (inMaxWaitTime)

	return (kNoErr);
} // GetObjectLock



//--------------------------------------------------------------------------------------------------
//	* ReleaseObjectLock
//
//--------------------------------------------------------------------------------------------------

void CDBBaseObject_10_1::ReleaseObjectLock ( const Boolean inNotifyOtherThreadsFlag )
{
	#pragma unused (inNotifyOtherThreadsFlag)
} // ReleaseObjectLock

void CDBBaseObject_10_1::WakeUpObjectThreads ( void )
{
}


void CDBBaseObject_10_1::setDirty ( const Boolean inForceWriteFlag )
{
	OSErr	result = kNoErr;

	if ( (this->GetObjectRefCount() == 0) || (inForceWriteFlag == true) )
	{
		if (this->GetObjectID() != 0)
		{
			this->SetDirtyFlag(false);
			this->WriteObject();
		}
	}
	else
	{
		BaseDBAssert(this->GetObjectRefCount() != 0);
		BaseDBAssert(fObjHeader != NULL);
		BaseDBAssert(fObjFooter != NULL);
		BaseDBAssert(fObjData != NULL);
		BaseDBAssert(fObjHeader->fSignature == fObjFooter->fSignature);

		this->SetDirtyFlag(true);
	}
	BaseDBAssert(result == kNoErr);
} // setDirty


