/*
	File:		CStr2IDTable.10.1.cpp

	Copyright:	 2003 by Apple Computer, Inc., all rights reserved.

 	NOT_FOR_OPEN_SOURCE <to be reevaluated at a later time>

	Change History:
*/


#include <ctype.h>
#include <string.h>

#include "CStr2IDTable.10.1.h"

void CStr2IDTable_10_1::strncpy ( char *inTargetStr, char *inSourceStr, const uInt32 inMaxLength )
{
	BaseDBAssert(inTargetStr != NULL);
	BaseDBAssert(inSourceStr != NULL);
	BaseDBAssert(inMaxLength > 0);
	
	if ((inTargetStr != NULL) && (inSourceStr != NULL) && (inMaxLength > 0))
	{
		uInt32 sourceLength = ::strlen(inSourceStr);
		BaseDBAssert(sourceLength <= inMaxLength);
		if (sourceLength > inMaxLength)
		{
			sourceLength = inMaxLength;
			::strncpy(inTargetStr, inSourceStr, sourceLength);
			inTargetStr[inMaxLength] = 0x00;
		}
		else
		{
			::strcpy(inTargetStr, inSourceStr);
		}
	}
}

CStr2IDTable_10_1::CStr2IDTable_10_1 ( Boolean inCaseSensitiveFlag )
{
	fPageID = 0;
	fTempHeader.fDataTableVersion = kStr2IDTableCurVersion_10_1;
	
	fTempHeader.fTableType			= kStr2IDTableType_10_1;
	fTempHeader.fCaseSensitiveFlag	= inCaseSensitiveFlag;			// current number of entries in this table
	fTempHeader.fCount = 0;			// current number of entries in this table

	fTempHeader.fNextID = 0;
	fTempHeader.fPrevID = 0;
	fTempHeader.fParentID = 0;
	
	fTempHeader.fStringHeap = 0;

	::memset(fTempHeader.fCurrentMinString, 0, kStr2IDMaxStringSize_10_1 );
	fTempHeader.f1stStringEntry[0].fStrOffset = 0;
	fTempHeader.f1stStringEntry[0].fObjID = 0;
	
	fStr2IDTablePtr = NULL;
	
	CStr2IDTable_10_1::CheckBackwardCompatibility();
}

void CStr2IDTable_10_1::CheckBackwardCompatibility ( void )
{
}

void CStr2IDTable_10_1::ReportBackwardCompatibility ( const uInt32 curSize, const uInt32 correctSize )
{
	if (curSize != correctSize)
	{
//		DebugStr("\pWARNING: CStr2IDTable_10_1 Base Class data element size problem.  Potential File format compatibility problem detected!");
	}
}

CStr2IDTable_10_1::~CStr2IDTable_10_1 ( void )
{
	fPageID = 0;
	fTempHeader.fDataTableVersion = kStr2IDTableCurVersion_10_1;
	
	fTempHeader.fTableType = kStr2IDTableType_10_1;
	fTempHeader.fCaseSensitiveFlag = false;			// current number of entries in this table
	fTempHeader.fCount = 0;			// current number of entries in this table

	fTempHeader.fNextID = 0;
	fTempHeader.fPrevID = 0;
	fTempHeader.fParentID = 0;
	
	fTempHeader.fStringHeap = 0;

	fTempHeader.f1stStringEntry[0].fStrOffset = 0;
	fTempHeader.f1stStringEntry[0].fObjID = 0;
	::memset(fTempHeader.fCurrentMinString, 0, kStr2IDMaxStringSize_10_1);
	
	fStr2IDTablePtr = NULL;
}

OSStatus CStr2IDTable_10_1::SetTable ( void *inTablePtr, const PageID inPageID)
{
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}
		
	fPageID = inPageID;
	fStr2IDTablePtr = (SStr2IDTable_10_1 *) inTablePtr;
	
	fTempHeader.fStringHeap = kMaxStr2IDTableSize_10_1 - 1;
	return kStr2IDTableNoErr;
}

OSStatus CStr2IDTable_10_1::ClearStr2IDTable	( Boolean inCaseSensitiveFlag )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	fTempHeader.fNextID = 0;
	fTempHeader.fPrevID = 0;
	fTempHeader.fParentID = 0;

	fTempHeader.fCaseSensitiveFlag = inCaseSensitiveFlag;			// current number of entries in this table
	fTempHeader.fCount = 0;			// current number of entries in this table

	::memset(fTempHeader.fCurrentMinString, 0, kStr2IDMaxStringSize_10_1);

	fTempHeader.f1stStringEntry[0].fStrOffset = 0;
	fTempHeader.f1stStringEntry[0].fObjID = 0;

	fTempHeader.fStringHeap = kMaxStr2IDTableSize_10_1 - 1;
	
	if (fStr2IDTablePtr != NULL)
	{
		fStr2IDTablePtr->fStr2IDTableHeader = fTempHeader;
		memset(fStr2IDTablePtr->fStr2IDDTableData, 0, kMaxStr2IDTableSize_10_1 - 1);
	}
	
	return kStr2IDTableNoErr;
}

OSStatus CStr2IDTable_10_1::AddStr2List ( const char *inEntryString, const ObjID inEntryID, const PageID inObjPageID, Boolean allowNewMin)
{
	OSStatus		result = kStr2IDTableNoErr;
	const uInt32	inStrLength = strlen(inEntryString);
	uInt32			anIndex = 0;
	uInt32			aStrIndex = 0;
	char			*aStrEntry = NULL;
	char			*aKeyString = NULL;
	ObjID			anObjID = 0;
	SStr2IDEntry_10_1	anEntry;
	SStr2IDEntry_10_1	aTempEntry;
	long			aCmpResult = 0;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	BaseDBAssert(inEntryString != NULL);
	if (inEntryString == NULL)
	{
		return kStr2IDBadString;
	}

	BaseDBAssert(inStrLength < kStr2IDMaxStringSize_10_1);
	if (inStrLength >= kStr2IDMaxStringSize_10_1)
	{
		return kStr2IDString2Big;
	}
	BaseDBAssert(inEntryID != 0);

	if (this->CalcFreeSpace() > ((inStrLength) + sizeof(SStr2IDEntry_10_1)))
	{
		result = this->FindEntry(inEntryString, anIndex);
		if (result != kStr2IDTableNoErr)
		{
			// lets check to see if we have a new min. string value...
			if (strlen(this->GetMinCharStar()) > 0)
			{
				aCmpResult = ::strcmp(inEntryString, this->GetMinCharStar());
				if ((aCmpResult < 0) && (allowNewMin == false))
				{	// we have a new min. value, but we're not allowed to add it to the list...
					return kStr2IDNewMinNotAllowed;
				}
			}
			else
			{
				aCmpResult = -1;	// fake a new min. value...
			}
			
			if ((aCmpResult < 0) && (allowNewMin == true))
			{
				memset(this->GetMinCharStar(), 0, kStr2IDMaxStringSize_10_1);
				CStr2IDTable_10_1::strncpy(this->GetMinCharStar(), (char *) inEntryString, kStr2IDMaxStringSize_10_1);
			}

			// lets copy the new string into the string-heap...
			this->SetStringHeap(this->GetStringHeap() - (inStrLength+1));
			anEntry.fStrOffset = this->GetStringHeap();
			anEntry.fObjID = inEntryID;
			anEntry.fObjPageID = inObjPageID;
			
			aKeyString = this->GetCharStar(anEntry.fStrOffset);
			CStr2IDTable_10_1::strncpy(aKeyString, (char *) inEntryString, inStrLength);
			aKeyString[inStrLength] = 0x00;
			
			if (this->GetCaseSensitiveFlag() == false)
			{
				for (aStrIndex = 0; aStrIndex < inStrLength; aStrIndex++)
				{
					aKeyString[aStrIndex] = ::tolower(aKeyString[aStrIndex]);
				}
			}

			if (this->GetEntryCount() > 0)
			{
				result = this->GetEntry(anIndex, aTempEntry);
				aCmpResult = ::strcmp(this->GetCharStar(aTempEntry.fStrOffset), aKeyString);
				
				if (aCmpResult < 0)
				{
					anIndex = anIndex + 1;
				}
				else if (aCmpResult == 0)
				{
					BaseDBAssert(aCmpResult != 0);
				}
			}
			else
			{
				anIndex = 0;
			}		
			
			result = InsertEntry(anIndex, anEntry);
		}
		else
		{
			result = kStr2IDTableDuplicateObject;
		}
	}
	else
	{
		result = kStr2IDTableFull;
	}
	
	return result;
}

char * CStr2IDTable_10_1::GetCharStar ( const unsigned short inStringOffset )
{
	char *result = NULL;
	
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = &(fStr2IDTablePtr->fStr2IDDTableData[inStringOffset]);
	
	return result;
}

char * CStr2IDTable_10_1::GetMinCharStar ( void )
{
	char *result = NULL;
	
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fCurrentMinString;
	
	return result;
}

uInt32 CStr2IDTable_10_1::CalcFreeSpace ( void )
{
	uInt32	dataSize = 0;
	uInt32	freeSpace = 0;
	
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return freeSpace;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return freeSpace;
	}
	
	dataSize = this->GetEntryCount() * sizeof(SStr2IDEntry_10_1);
	freeSpace = this->GetStringHeap() - dataSize;
	
	return freeSpace;
}

OSStatus CStr2IDTable_10_1::RemoveStrFromList ( const char *inEntryString )
{
	OSStatus	result = kStr2IDTableNoErr;
	uInt32		anIndex = 0;
	
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	result = this->FindEntry(inEntryString, anIndex);
	if (result == kStr2IDTableNoErr)
	{
		result = this->DeleteEntry(anIndex);
	}
	
	return result;
}

char * CStr2IDTable_10_1::GetNewString ( const char *inSourceString, Boolean inMakeLowerCaseFlag )
{
	uInt32	index = 0;
	uInt32	strSize = 0;
	char	*aNewString = NULL;
	
	strSize = strlen(inSourceString);
	aNewString = new char[strSize + 1];
	BaseDBAssert(aNewString != NULL);
	
	CStr2IDTable_10_1::strncpy(aNewString, (char *) inSourceString, (strSize + 1));
	if (inMakeLowerCaseFlag == true)
	{
		for (index = 0; index < strSize; index++)
		{
			aNewString[index] = ::tolower(aNewString[index]);
		}
	}
	
	return aNewString;
}

OSStatus CStr2IDTable_10_1::FindEntry ( const char *inEntryString, uInt32 &outIndex )
{
	OSStatus		result = kStr2IDTableNoSuchObject;
	uInt32			index = 0;
	const uInt32	strSize = strlen(inEntryString);
	long			high = 0;
	long			low = 0;
	long			cmpResult = 0;
	char			*aSearchStr = NULL;
	SStr2IDEntry_10_1	anEntry;
	Boolean			doSearch = true;
	
	outIndex = kStr2IDNullIndex_10_1;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	BaseDBAssert(strSize < kStr2IDMaxStringSize_10_1);
	if (strSize >= kStr2IDMaxStringSize_10_1)
	{
		return kStr2IDString2Big;
	}

	if (this->GetEntryCount() > 0)
	{
		// now that we've made it past the error checking, lets start the "meat" of the
		// function

		// setup the boundaries for the binary search...
		low = 0;
		high = this->GetEntryCount() - 1;

		aSearchStr = this->GetNewString(inEntryString, (this->GetCaseSensitiveFlag() == false));
		BaseDBAssert(aSearchStr != NULL);

		if (this->GetEntryCount() > 2)
		{	// we'll do some summary checks to see if we should even bother looking...
			// we only do the summary boundary checks if there are more than 2 entires, since we'll do
			// at least two string compares in this function, if there are only 2 entries in the array there's
			// no point duplicating the effort with the normal algorythm.
			
			result = GetEntry(0, anEntry);
			BaseDBAssert(result == kStr2IDTableNoErr);
			
			cmpResult = ::strcmp(aSearchStr, this->GetCharStar(anEntry.fStrOffset));
			if (cmpResult == 0)
			{	// WOW, we found that the search string matches the 1st entry in the array
				// lucky US...this is like an O(1) function performance....
				doSearch = false;
				outIndex = 0;
				result = kStr2IDTableNoErr;
			}
			else if (cmpResult < 0)
			{	// hmmm, the string we're searching for is less than the min. entry in the array, no point searching
				// the array for this string, it's not there....
				doSearch = false;
				outIndex = 0;
				result = kStr2IDTableNoSuchObject;
			}
			else
			{
				low = 1;		// we've already checked the zero entry, no point in checking it again...
				doSearch = true;
				result = kStr2IDTableNoSuchObject;
			}

			if (doSearch)
			{
				result = GetEntry((this->GetEntryCount() - 1), anEntry);
				BaseDBAssert(result == kStr2IDTableNoErr);
				cmpResult = ::strcmp(aSearchStr, this->GetCharStar(anEntry.fStrOffset));
				if (cmpResult == 0)
				{	// WOW, we found that the search string matches the last entry in the array
					// lucky US...this is like an O(2) function performance....
					doSearch = false;
					outIndex = (this->GetEntryCount() - 1);
					result = kStr2IDTableNoErr;
				}
				else if (cmpResult > 0)
				{	// ok the string we're searching for is greater than the last entry in the array..
					// no point in going any further with this charade...
					doSearch = false;
					outIndex = (this->GetEntryCount() - 1);
					result = kStr2IDTableNoSuchObject;
				}
				else
				{
					high =  this->GetEntryCount() - 2;	// we've already checked the last entry, no point in checking it again...
					doSearch = true;
					result = kStr2IDTableNoSuchObject;
				}
			}			
		}

		if (doSearch)
		{
			while (low <= high)
			{
				index = low + ((high - low) / 2);
				outIndex = index;

				result = GetEntry(index, anEntry);
				BaseDBAssert(result == kStr2IDTableNoErr);
				
				cmpResult = ::strcmp(this->GetCharStar(anEntry.fStrOffset), aSearchStr);
				if (cmpResult == 0)
				{
					result = kStr2IDTableNoErr;
					break;
				}
				else
				{
					result = kStr2IDTableNoSuchObject;
					if (cmpResult < 0)
						low = index + 1;
					else
						high = index - 1;
				}
			}
		}		

		if (aSearchStr != NULL)
		{
			delete aSearchStr;
			aSearchStr = NULL;
		}
	}
	
	return result;
}

OSStatus CStr2IDTable_10_1::GetEntry ( const uInt32 inIndex, SStr2IDEntry_10_1 &outEntry )
{
	OSStatus	result = kStr2IDTableNoErr;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	BaseDBAssert((inIndex >= 0) && (inIndex < this->GetEntryCount()));
	
	if ((inIndex >= 0) && (inIndex < this->GetEntryCount()))
	{
		outEntry = this->fStr2IDTablePtr->fStr2IDTableHeader.f1stStringEntry[inIndex];
	}
	else
	{
		result = kStr2IDTableIndexOutOfRange;
	}
	
	return result;
}

OSType CStr2IDTable_10_1::GetTableType ( void )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return NULL;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return NULL;
	}
	
	return this->fStr2IDTablePtr->fStr2IDTableHeader.fTableType;
}
		
OSStatus CStr2IDTable_10_1::SetStringHeap ( const unsigned short inStringHeap )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	if (inStringHeap != (this->GetStringHeap()))
	{
		this->fStr2IDTablePtr->fStr2IDTableHeader.fStringHeap = inStringHeap;
	}
	
	return kStr2IDTableNoErr;
}

Boolean CStr2IDTable_10_1::GetCaseSensitiveFlag ( void )
{
	Boolean	result = false;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fCaseSensitiveFlag;
	return result;
}

unsigned short CStr2IDTable_10_1::GetStringHeap ( void )
{
	unsigned short	result = 0;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fStringHeap;
	return result;
}

OSStatus CStr2IDTable_10_1::SetEntryCount ( const uInt32 inEntryCount )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	if (inEntryCount != (this->GetEntryCount()))
	{
		this->fStr2IDTablePtr->fStr2IDTableHeader.fCount = inEntryCount;
	}
	
	return kStr2IDTableNoErr;
}

uInt32 CStr2IDTable_10_1::GetEntryCount ( void )
{
	uInt32	result = 0;
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fCount;
	
	return result;
}

PageID CStr2IDTable_10_1::GetNextID ( void )
{
	PageID	result = 0;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fNextID;
	
	return result;
}

OSStatus CStr2IDTable_10_1::SetNextID ( const PageID inNextIDTable )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	this->fStr2IDTablePtr->fStr2IDTableHeader.fNextID = inNextIDTable;
	
	return kStr2IDTableNoErr;
}


PageID CStr2IDTable_10_1::GetPrevID ( void )
{
	PageID	result = 0;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fPrevID;
	
	return result;
}

OSStatus CStr2IDTable_10_1::SetPrevID ( const PageID inPrevIDTable )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	this->fStr2IDTablePtr->fStr2IDTableHeader.fPrevID = inPrevIDTable;
	
	return kStr2IDTableNoErr;
}

PageID CStr2IDTable_10_1::GetParentID ( void )
{
	PageID	result = 0;

	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return result;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return result;
	}
	
	result = this->fStr2IDTablePtr->fStr2IDTableHeader.fParentID;
	
	return result;
}

OSStatus CStr2IDTable_10_1::SetParentID ( const PageID inParentIDTable )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}

	BaseDBAssert(fStr2IDTablePtr != NULL);
	if (fStr2IDTablePtr == NULL)
	{
		return kStr2IDBadTablePtr;
	}
	
	this->fStr2IDTablePtr->fStr2IDTableHeader.fParentID = inParentIDTable;
	
	return kStr2IDTableNoErr;
}

OSStatus CStr2IDTable_10_1::SetTablePageID ( const PageID inPageID )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return kStr2IDBadThisObject;
	}
	
	fPageID = inPageID;

	return kStr2IDTableNoErr;
}

PageID CStr2IDTable_10_1::GetTablePageID ( void )
{
	// Basic Error checking before doing the "meat" of the function...	
	BaseDBAssert(this != NULL);
	if (this == NULL)
	{
		return 0;
	}
	
	return fPageID;
}

OSStatus CStr2IDTable_10_1::DeleteEntry ( const uInt32 inIndex )
{
	SStr2IDEntry_10_1	anEntry;
	OSStatus		result			= kStr2IDTableNoErr;
	uInt32			anIndex			= 0;
	uInt32			bytes2Move		= 0;
	uInt32			distance		= 0;
	char			*anEntryString	= NULL;
	char			*the1stString	= NULL;
	
	BaseDBAssert(this != NULL);
	
	// 1st we've got to compress the string heap....	
	result = this->GetEntry( inIndex, anEntry );
	BaseDBAssert( result == kStr2IDTableNoErr );

	anEntryString = this->GetCharStar( anEntry.fStrOffset );
	the1stString = this->GetCharStar( this->GetStringHeap() );
	
	bytes2Move = anEntryString - the1stString;
	distance = strlen(anEntryString) + 1;
	
	for (anEntryString = anEntryString - 1; anEntryString >= the1stString; anEntryString--)
	{
		anEntryString[distance] = *anEntryString;
	}

	this->SetStringHeap( this->GetStringHeap() + distance );
	::memset( the1stString, 0, distance );

	anEntryString = this->GetCharStar(anEntry.fStrOffset);

	// buzz through the array and adjust any offsets that were affected by our
	// recent string heap compression...
	for (anIndex = 0; anIndex < this->GetEntryCount(); anIndex++)
	{
		result = this->GetEntry(anIndex, anEntry);
		BaseDBAssert(result == kStr2IDTableNoErr);
		
		if (this->GetCharStar(anEntry.fStrOffset) < anEntryString)
		{
			anEntry.fStrOffset += distance;
			result = this->SetEntry(anIndex, anEntry);
			BaseDBAssert(result == kStr2IDTableNoErr);
		}
	}
	
	// now that we've compressed the string heap, we can compress the entry array..	
	for (anIndex = inIndex + 1; anIndex < this->GetEntryCount(); anIndex++)
	{
		result = this->GetEntry(anIndex, anEntry);
		BaseDBAssert(result == kStr2IDTableNoErr);
		
		result = this->SetEntry(anIndex-1, anEntry);
		BaseDBAssert(result == kStr2IDTableNoErr);
	}
	
	this->SetEntryCount(this->GetEntryCount() - 1);
	return result;
}

OSStatus CStr2IDTable_10_1::SetEntry (const uInt32 inIndex, const SStr2IDEntry_10_1 inEntry)
{
	OSStatus	result = kStr2IDTableNoErr;

	BaseDBAssert(this != NULL);
	BaseDBAssert((inIndex >= 0) && (inIndex <= this->GetEntryCount()));
	
	if ((inIndex >= 0) && (inIndex <= this->GetEntryCount()))
	{
		this->fStr2IDTablePtr->fStr2IDTableHeader.f1stStringEntry[inIndex] = inEntry;
	}
	else
	{
		result = kStr2IDTableIndexOutOfRange;
	}
	
	return result;
}

OSStatus CStr2IDTable_10_1::InsertEntry (const uInt32 inIndex, const SStr2IDEntry_10_1 inEntry)
{
	OSStatus		result = kStr2IDTableNoErr;
	uInt32			anIndex = 0;
	SStr2IDEntry_10_1	anEntry;
	
	BaseDBAssert(this != NULL);
	
	if (this->GetEntryCount() > 0)
	{
		for (anIndex = this->GetEntryCount(); anIndex > inIndex; anIndex--)
		{
			result = this->GetEntry(anIndex-1, anEntry);
			BaseDBAssert(result == kStr2IDTableNoErr);
			
			result = this->SetEntry(anIndex, anEntry);
			BaseDBAssert(result == kStr2IDTableNoErr);
		}
	}
	else
	{
		BaseDBAssert(inIndex == 0);
	}
	
	this->SetEntryCount(this->GetEntryCount() + 1);
	this->SetEntry(inIndex, inEntry);
	
	return result;
}

OSStatus CStr2IDTable_10_1::SanityCheck ( void )
{
	return kStr2IDTableNoErr;
}
