/*
	$Id: CHostEntry.cpp,v 1.1 2003/08/10 18:09:25 dasenbro Exp $

	File:		CHostEntry.cpp

	Contains:	C++ implementation of persistent Host Entry objects

	Version:	Apple Mail Server - Mac OS X :  $Revision: 1.1 $

	Written by:	Nick Brosnahan

	Copyright:	 1996-2001 by Apple Computer, Inc., all rights reserved.

 	NOT_FOR_OPEN_SOURCE <to be reevaluated at a later time>

	Change History:

		$Log: CHostEntry.cpp,v $
		Revision 1.1  2003/08/10 18:09:25  dasenbro
		added to export local host names.
		
		Revision 1.26  2002/06/07 23:03:41  dasenbro
		Removed commented code in CalcNextAttemptTime().
		
		Revision 1.25  2002/06/04 17:50:21  dasenbro
		Added UnLockTheCache() to the catch().
		
		Revision 1.24  2002/05/09 16:59:02  dasenbro
		Changed all str... calls to CUtils::Str... to be NULL safe.
		
		Revision 1.23  2002/04/18 18:09:14  dasenbro
		Changed bool to Bool for word alignment.
		
		Revision 1.22  2002/04/16 05:35:13  dasenbro
		Redirected log output to SMTP log file.
		
		Revision 1.21  2002/03/21 16:41:37  dasenbro
		Updated file version information.
		
		Revision 1.20  2002/03/11 23:15:02  dasenbro
		Read the host prefs from the directory and removed calls that
		had no HI to support changing values.
		
		Revision 1.19  2002/01/14 17:13:56  dasenbro
		Initial S4 updates.
		
		Revision 1.18  2001/07/09 16:03:45  dasenbro
		Syntax changes for better readability.
		
		Revision 1.17  2001/06/21 20:51:06  dasenbro
		Updated file header info.
		
		Revision 1.16  2001/06/21 19:49:42  dasenbro
		Added Change History.
		

	Projector History:

	  <ASM9>	 3/15/99	MED		Removed CSmartYield.
	  <ASM8>	  3/1/99	DOR		Change SetNextTime to do the "right" thing.
	  <ASM7>	 1/24/99	MED		Added a boolean to Create to prevent searching for host prefs.
	  <ASM6>	12/10/98	DOR		We should do actual assignements rather than compares when doing
									our "Set" routines.
		 <5>	10/30/98	DOR		Added support for 2nd ID's to host entries.  We now move host's
									onto and off of a 2nd ID list based on their "nextAttemptTime"
									value.  The outgoing threads will only visit hosts on the
									"active" host list, therefore limiting the number of hosts that
									must be visited to process outgoing mail.
		 <4>	10/29/98	DOR		Merge the expire host changes onto the main code branch.
		 <3>	10/18/98	MED		Set redirect host flag to false on error.
		 <2>	10/13/98	DOR		Lock the cache when deleting an object.
		<69>	10/13/98	DOR		Demo for Mike
		<68>	 9/23/98	DOR		Optimize FindHostByTypeName to check the last host we "saw" to
									see if it's what we're looking for.
		<67>	 9/21/98	MED		Create() first looks for the host and only creates it if it
									isn't found.
		<66>	 8/24/98	MED		Added set/get members to ignore schedule.
		<65>	 6/17/98	DOR		Initialize the  NextAttempt time to be a large value so that
									newly created hosts don't occupy very much of the Mail Server's
									time..
		<64>	  6/4/98	DOR		#2241916 - properly calculate the elasped time for AppleTalk
									connections.
		<63>	  6/1/98	DOR		#2240866 - changed the ETRN code to use the Next deferal
									interval..
		<62>	  6/1/98	DOR		Remove the "kAdd" argument from Create since we no longer use
									it.
		<61>	  6/1/98	MED		Changed ::Create to be a more kinder and gentler ::Create.
		<60>	  6/1/98	DOR		Change finish connection to properly calculate the connection
									time.
		<59>	 5/22/98	DOR		Add code to support ignoring DNS cache settings while doing
									local-host startup checks..
		<58>	 5/19/98	DOR		Fix for bug #2237747
		<57>	 5/19/98	MED		Added future expansion data members.
		<56>	 5/11/98	DOR		Properly use the Server cache setting values as min. rather than
									seconds.  Fill in a "Fake" 4 hour value for RespectTTL..
		<55>	  5/5/98	DOR		Fix another CalcNextETRN bugs.
		<54>	  5/5/98	DOR		Fix CalcNextETRNTime, and add a ETRNConnectionFinish routine.
		<53>	  4/8/98	DOR		Fix warnings.
		<52>	  4/8/98	DOR		Change from using ::strncpy to CUtils::Strncpy
		<51>	  4/8/98	DOR		Fix some ::strcpy's to be ::strncpy's
		<50>	  4/2/98	DOR		Change call to gDB->removeObject
		<49>	 3/31/98	DOR		Change some calls to "SetConnectionInfo" to set "more"
									information.
		<48>	 3/31/98	DOR		Fix it so that we duplicate suppress when posting to a host
									spool.
		<47>	 3/27/98	DOR		Clear the remote forwarding flag if the domain name is "bad".
		<46>	03/26/98	DOR		Fix CHostEntry so that it can't create a host with an empty
									HostName... Bug #2223086
		<45>	03/23/98	DOR		Grab the ObjectLock in ::Create.
		<44>	03/23/98	DOR		Get the object lock in some key places to prevent race
									conditions.
		<43>	03/18/98	DOR		Change an If condition to do an explicite compare with a value,
									rather than relying on C's implicite zero is false policy.
		<42>	03/17/98	DOR		Use elapsed time utility function to format elapsed time strings
		<41>	 3/14/98	DOR		Close up a race condition in ::Create..
		<40>	 3/12/98	DOR		Print out preference error's when the forwarding DNS info is
									bad.
		<39>	 3/11/98	DOR		Merge changes off of a branch.
		<38>	 3/11/98	MED		Changed FindByID() to do a Use() only if it created the object,
									FindMe() now does an explicit Use().
		<37>	 3/10/98	DOR		Fix a bug where we were resetting the resolved host name _EVERY_
									time we posted a message.
		<36>	  3/9/98	DOR		Fix CalcNextAttemptTime to move the NextAttemptTime forward if a
									custome schedule is currently in a "active" time-window...
		<35>	  3/9/98	DOR		Move Get/Set/Clear OwnerThreadID to CDBObject.
		<34>	  3/7/98	DOR		Minor changes.
		<33>	  3/5/98	DOR		Remove code that was setting a host to "kBounceDestination & add
									some debugging code to prevent it..
		<31>	 2/16/98	DOR		Update some log messages.
		<30>	 2/11/98	DOR		Changes in usage of IsLocalHost.
		<29>	  2/3/98	MED		Cleaned OSUtil.cp into CUtils.cp, COSUtils.cp and
									CSmartYield.cp.
		<28>	 1/12/98	MED		Fixed a conditional check in IsSendETRNOn().
		<27>	 1/12/98	MED		Added support for ETRN.
		<26>	12/16/97	MED		Added a ; after a mail assert, Dave.
		<25>	12/15/97	DOR		Update the object version to "include" the size of the
									dataportion of the object, and add an assert to "FindByID" that
									checks the version of the object.
		<24>	12/12/97	DOR		Update CalcNextAttempt time to use schedule information if
									available.
		<23>	12/11/97	DOR		Deal with calculating the next connection retry frequency based
									on host preferences.
		<22>	12/10/97	DOR		Minor changes.
		<21>	12/10/97	DOR		Minor changes.
		<20>	12/10/97	DOR		Add code to age the DNS entry when the MX/A-Record flags change
									from their previous settings.
		<19>	12/10/97	DOR		Pay attention to "use server general" & "resolve MX & A-Record"
		<18>	12/10/97	DOR		Now pay attention to global mail redirection & global SMTP
									outgoing port #.
		<17>	 12/9/97	DOR		Psuedo code to pay attention to Global server settings.
		<16>	 12/9/97	DOR		New log messages.
		<15>	 12/5/97	MED		fixed SetDeleteThisHost().
		<14>	 12/5/97	MED		Added stop now and delete host flags with getters and setters.
		<13>	 12/4/97	DOR		No longer cache preferences or DNS info, let the Database do
									it!!!  Don't be _CLEVER_!!!!
		<12>	 12/4/97	DOR		Create the prefs object when creating a HostEntry.
		<11>	 12/3/97	DOR		More tweaks.
		<10>	 12/3/97	DOR		Minor syntax errors, file now compiles.
		 <9>	 12/3/97	DOR		Add all the CMXRecord code.  This file may not compile as is.
		 <8>	 12/3/97	MED		Changed CInHostEntryPref to CHostEntryPref.
		 <7>	 12/3/97	MED		Combined In and Out host entry prefs.
		 <6>	 12/2/97	DOR		Make set next attempt time smart enough to "kick" the outgoing
									threads.
		 <5>	 12/2/97	DOR		Add function to find and cache HostEntry preferences...
		 <4>	11/21/97	DOR		Add code to generate non-delivery warnings.
		 <3>	11/19/97	MED		Integrated 5.0.3 changes.
		 <2>	 10/9/97	DOR		Add two fields to CHostEntry, two ObjID fields to "point" to the
									incoming & outgoing routing preferences.

	To Do:
*/

#include <time.h>

#include "CHostEntry.h"
#include "CMailSpool.h"
#include "DSMgr.h"
#include "DSHostPrefs.h"
#include "DSServerPrefs.h"
#include "AppResources.h"
#include "MailGlobals.h"
#include "CEnvelopeInfo.h"
#include "CEnvelope.h"
#include "Database.h"
#include "CUtils.h"
#include "CRootObject.h"
#include "CGlobals.h"


Boolean	CHostEntry::GetSetFields (	const eFieldDataRequest inFieldRequest,
									const void				*inObjDataPtr,
									void					*outFieldData )
{
	Boolean			result			= false;
	CHostEntryData	*theData		= (CHostEntryData *)inObjDataPtr;
	CHostEntry		*aTempObject	= NULL;
	char			*theDestString	= (char *)outFieldData;
	ObjID			*anObjIDPtr		= (ObjID *)outFieldData;

	switch ( inFieldRequest )
	{
		case kSetObjID:
			theData->fHostEntryID = *anObjIDPtr;
			break;

		case kStringNameRequest:
			if (CUtils::Strlen(theData->fHostName) != 0)
			{
				CUtils::Strncpy(theDestString, theData->fHostName, kStr2IDMaxStringSize);
				result = true;
			}
			break;
		
		case k2ndIDRequest:
			result = false;
			if ( (theData->fHostStatistics.fShortFlags & k2ndIDHasBeenSet) != 0 )
			{
				if ( ((theData->fHostStatistics.nextConnectionAttempt != 0) && (theData->fHostStatistics.nextConnectionAttempt < kNeverAttemptToConnectValue)) ||
					((theData->fHostStatistics.nextETRNTime != 0) && (theData->fHostStatistics.nextETRNTime < kNeverAttemptToConnectValue)) )
				{
					if ( gRootObj != nil )
					{
						*anObjIDPtr = gRootObj->GetActiveHost2ndID( false );
						result = true;
					}
					else
					{
					}
				}
			}
			break;

		case kObjectPINGRequest:
			// cause the objects constructor to get called, so we force a check of
			// the struct alignment...
			aTempObject = new CHostEntry;
			delete( aTempObject );
			break;

		case k2ndIDTargetChk:
			// fall through...
			
		default:
			result = false;
			break;
	}

	return result;
}

ObjID CHostEntry::GetObjectID	( void )
{
	return fHostEntryData.fHostEntryID;
}


//--------------------------------------------------------------------------------------------------
//	 Done
//
//--------------------------------------------------------------------------------------------------

void CHostEntry::Done ( CHostEntry* &inPtr )
{
	CDBBaseObject *aDBObject = (CDBBaseObject *) inPtr;
	aDBObject->Done(aDBObject);
	inPtr = (CHostEntry *)aDBObject;

}


CDBBaseObject* CHostEntry::ObjectAllocNew	( void )
{
	return new CHostEntry;
}

CHostEntry* CHostEntry::FindByID ( const ObjID inObjectID )
{
	return (CHostEntry *) CDBMailBaseObject::MailUniversalFindByID(	inObjectID,
														CHostEntry::GetObjTypeConstant(),
														CHostEntry::GetObjVersConstant(),
														CHostEntry::GetObjSizeConstant(),
														CHostEntry::ObjectAllocNew		);
} // FindByID



CHostEntry* CHostEntry::FindByTypeName ( OSType inAddressType, const char* inHostName )
{
	static ObjID	lastHostIDFound = 0;

	Boolean		foundOne = false;
	ObjID		theHostID;
	CHostEntry	*theHostEntry = NULL;
	OSErr 		result;

	try
	{
		(gDB->GetBaseObjectCachePtr())->LockTheCache(CHostEntry::GetObjTypeConstant());

		if (lastHostIDFound != 0)
		{
			theHostEntry = CHostEntry::FindByID(lastHostIDFound);
			if (theHostEntry != NULL)
			{
				if ((CUtils::Stricmp(inHostName, theHostEntry->GetHostName()) == 0) &&
					(theHostEntry->GetAddressType() == inAddressType))
				{
					foundOne = true;
				}
				else
				{
					theHostEntry->Done(theHostEntry);
				}
			}
		}

		if (foundOne == false)
		{
			result = gDB->GetObjDataByString(kHostEntrySignature, inHostName, theHostID);
			if (result == CMailDatabase::kDBNoErr)
			{
				theHostEntry = CHostEntry::FindByID(theHostID);
				if (theHostEntry != NULL)
				{
					if (theHostEntry->GetAddressType() == inAddressType)
					{
						lastHostIDFound = theHostID;
						foundOne = true;
					}
					else
					{
						theHostEntry->Done(theHostEntry);
					}
				}
			}
		}		
		(gDB->GetBaseObjectCachePtr())->UnLockTheCache(CHostEntry::GetObjTypeConstant());
	}

	catch ( long err )
	{
		(gDB->GetBaseObjectCachePtr())->UnLockTheCache(CHostEntry::GetObjTypeConstant());

		Throw_( err );
	}

	return theHostEntry;
}

CHostEntry* CHostEntry::Create ( OSType			inAddressType,
								 const char	   *inHostName,
								 Boolean	   &outHostCreated,
								 const Boolean	inSetConnectionTime )
{
	CHostEntry		*theHostEntry	= nil;
//	CHostEntryPref	*pHostPrefsObj	= NULL;
	OSErr			result = kNoErr;
	ObjID			aNewID = 0;
	Boolean			junk = false;
	uInt32			curTime	= 0;

	try
	{
		outHostCreated = false;

		if ( CUtils::Strlen( inHostName ) == 0 )
		{
			return ( NULL );
		}

		(gDB->GetBaseObjectCachePtr())->LockTheCache(CHostEntry::GetObjTypeConstant());

		// MUST do a find before a create to see if it already exists...
		theHostEntry = CHostEntry::FindByTypeName( inAddressType, inHostName );
		if ( theHostEntry == nil )
		{
			theHostEntry = new CHostEntry;

			ThrowIfMemFail_( theHostEntry );

			// NOTE: this needs to be set _BEFORE_ we create the object in the DB
			// so that at the first entry of the object into the DB we can "find" this
			// host entry from other threads...
			theHostEntry->SetHostName( inHostName );

			result = gDB->CreateObj( kHostEntrySignature, aNewID, theHostEntry->GetHostEntryData() );
			if ( result == CMailDatabase::kDBNoErr )
			{
				if ( theHostEntry->GetObjectLock(false) == kNoErr )
				{
					theHostEntry->Use();
					theHostEntry->SetAddressType (inAddressType);	// kRFC822 or kAppleTalk
					theHostEntry->SetResolveHostName (inHostName, junk);

					if (inSetConnectionTime == true)
					{
						curTime = ::time(NULL);
						curTime = curTime - (curTime % 60);	// round off to last min...this will improve the odds
															// that last connect time & last failed time will _NEVER_ match
															// and for this operation losing upto 59 seconds is not an issue...
						theHostEntry->SetLastSuccessConnectTime(curTime);
						theHostEntry->SetLastFailedConnectTime(curTime);
					}
					
					theHostEntry->SetShortFlags(theHostEntry->GetShortFlags() | k2ndIDHasBeenSet);
					theHostEntry->SetDirty();

					outHostCreated = true;
				}
				else
				{
					delete( theHostEntry );
					theHostEntry = NULL;
				}
			}
			else
			{
				delete( theHostEntry );
				theHostEntry = NULL;
			}

			if (theHostEntry != NULL)
			{
				theHostEntry->ReleaseObjectLock(true);
			}
		}

		(gDB->GetBaseObjectCachePtr())->UnLockTheCache(CHostEntry::GetObjTypeConstant());
	}

	catch( long err )
	{
		(gDB->GetBaseObjectCachePtr())->UnLockTheCache( CHostEntry::GetObjTypeConstant() );
	}

	return (theHostEntry);
}


CHostEntryData* CHostEntry::GetHostEntryData	( void )
{
	if (this != NULL)
	{
		return &(this->fHostEntryData);
	}
	else
	{
		return NULL;
	}
}

void CHostEntry::SetDirty	( const Boolean inForceWriteFlag )
{
	if ((this != NULL) && (this->GetHostEntryID() != 0))
	{
		setDirty(inForceWriteFlag);
	}
}



Boolean CHostEntry::Delete ( CHostEntry *inHostEntry )
{
	Boolean deleted = false;
	CMailSpool	*theMailSpool	= NULL;
	OSErr		result			= kNoErr;

	try
	{
		if ( inHostEntry != NULL )
		{
			( gDB->GetBaseObjectCachePtr())->LockTheCache(CHostEntry::GetObjTypeConstant() );

			result = gDB->RemoveObj(kHostEntrySignature, inHostEntry->GetHostEntryID());
			deleted = (result == kNoErr);
			if ( deleted )
			{
				inHostEntry->fHostEntryData.fHostEntryID = 0;
			}

			// let's delete the mail spool that is associated with this Host Entry
			theMailSpool = CMailSpool::FindByID( inHostEntry->GetMailSpoolID() );
			ThrowIfNULL_( theMailSpool );

			CMailSpool::Delete( theMailSpool );
			theMailSpool->Done( theMailSpool );

			(gDB->GetBaseObjectCachePtr())->UnLockTheCache(CHostEntry::GetObjTypeConstant());
		}
	}

	catch ( long err )
	{
		theMailSpool->Done (theMailSpool);

		(gDB->GetBaseObjectCachePtr())->UnLockTheCache(CHostEntry::GetObjTypeConstant());
	}

	return( deleted );

} // Delete

uInt32 CHostEntry::Count()
{
	uInt32	aCount = 0;
	OSErr	result;

	result = gDB->GetObjectCount( kHostEntrySignature, aCount );

	if (result != kNoErr)
	{
		aCount = 0;
	}

	return aCount;
}

#pragma mark -

CHostEntry::CHostEntry(void) :
	CDBMailBaseObject(	&fHostEntryData.fDBHeader,
				&fHostEntryData.fDBFooter,
				&fHostEntryData,
				CHostEntry::GetObjTypeConstant(),
				CHostEntry::GetObjVersConstant(),
				CHostEntry::GetObjSizeConstant() )
{
	// NeoAccess IDs
	fHostEntryData.fHostEntryID		= 0;
	fHostEntryData.fMailSpoolID		= 0;

	fGoPostal						= false;
	fDeleteThisHost					= false;

	// HostEntry Information
	fHostEntryData.fAddressType		 = 0;
	fHostEntryData.fHostName[0]		 = '\0';
	fHostEntryData.fDestination		 = 0;
	fHostEntryData.fAccessTime		 = 0;
	fHostEntryData.fDestinationClass = kDestClassUnknown;
	fHostEntryData.fIgnoreSchedule	 = true;		// Ignore schedule flag
	

	// Resolution
	fHostEntryData.fResolveHostName[0]	 = '\0';
	fHostEntryData.fHostPrefsID			 = 0;
	fHostEntryData.fRealHostDNSCacheID	 = 0;
	fHostEntryData.fResolveHostDNSCacheID = 0;

#if USE_APPLETALK
	// AppleTalk
	::memset (&fDDPAddress, 0, sizeof (DDPAddress));
#endif
	::memset (&fHostEntryData.fHostStatistics, 0, sizeof (HostStatistics));
	fHostEntryData.fHostStatistics.nextConnectionAttempt = kNeverAttemptToConnectValue;
	fHostEntryData.fHostStatistics.nextETRNTime = kNeverAttemptToConnectValue;

	// Reserved
	fHostEntryData.fReserved1	= kReservedConst;
	fHostEntryData.fReserved2	= kReservedConst;
	
	this->ChkCompilerStructAlignment();
}



CHostEntry::~CHostEntry (void)
{
}

void	CHostEntry::ChkCompilerStructAlignment	( void )
{
	/*
		this dump was aquired by using the MPW command DumpSYM - TTE from the .xsym file produced
		from a Code Warrior Pro 1 Release Build of the AppleShare IP Mail Server for MacOS version 6.2
		
		this version of the structure is consider the "reference" standard for all future ASIP Mail Servers.
		If the structure does NOT match these offsets the version of the Mail Server that 
		you are compiling will _NOT_ be compatible with AppleShare IP 6.0, 6.1, or 6.2 (this also effects
		the AppleShare IP Mail Tool, and to a more limited extent the AppleShare IP Mail Admin since some
		of these structures are passed across the wire..)
		
		If _ANY_ of these Asserts "fire" you've got a backwards compatibility problem that _MUST_ be fixed!!
		
		this code wasn't put here for my personal amusement, this code was put here from a hard earned lesson
		that simple changes to headers/compilers _WILL_ break this stuff, better to know now than after we
		ship.  Look upon this code as a debugging AIDE, not an impediment.
		
		Type Name:  CHostEntryData		TINFO offset: 55282		(TINFO SYM file offset (hex): $1257F2)
			RecordOf 
			offset 0 NamedTypeOf fDBHeader TypeDef of "Type ID 248"
			offset 12 NamedTypeOf fHostEntryID 
			offset 16 NamedTypeOf fMailSpoolID unsigned long 
			offset 20 NamedTypeOf fReserved1 
			offset 24 NamedTypeOf fReserved2 unsigned long 
			offset 28 NamedTypeOf fAddressType 
			offset 32 NamedTypeOf fHostName CString 
			offset 288 NamedTypeOf fDestination unsigned long 
			offset 292 
			offset 296 NamedTypeOf fDestinationClass unsigned long 
			offset 300 
			offset 556 NamedTypeOf fHostStatistics TypeDef of "Type ID 255"
			offset 624 NamedTypeOf fHostPrefsID 
			offset 628 NamedTypeOf fRealHostDNSCacheID unsigned long 
			offset 632 NamedTypeOf fResolveHostDNSCacheID 
			offset 636 NamedTypeOf fDBFooter TypeDef of "Type ID 250"
	*/

	static Boolean	aRunOnceFlag = false;	
	if (aRunOnceFlag == false)
	{
		//FileFormatSaftyChk(CHostEntryData, fDBHeader,				0);
		//FileFormatSaftyChk(CHostEntryData, fHostEntryID,			12);				// this is us (denormalized)
		//FileFormatSaftyChk(CHostEntryData, fMailSpoolID,			16);				// this is our mail spool
		//FileFormatSaftyChk(CHostEntryData, fReserved1,			20);
		//FileFormatSaftyChk(CHostEntryData, fReserved2,			24);
		//FileFormatSaftyChk(CHostEntryData, fAddressType,			28);				// kRFC822 or kAppleTalk
		//FileFormatSaftyChk(CHostEntryData, fHostName,				32);					// Original hostname or NBPAddressString
		//FileFormatSaftyChk(CHostEntryData, fDestination,			288);				// TCP/IP or ADSP or UUCP or Bounce
		//FileFormatSaftyChk(CHostEntryData, fAccessTime,				292);				// Last time accessed
		//FileFormatSaftyChk(CHostEntryData, fDestinationClass,		296);			// keep track of local vs. Remote...
		//FileFormatSaftyChk(CHostEntryData, fResolveHostName,		300);			// Hostname or NBPAddressString to send to
		//FileFormatSaftyChk(CHostEntryData, fHostStatistics,			556);
		//FileFormatSaftyChk(CHostEntryData, fHostPrefsID,			624);				// ObjID of a CHostEntryPref record...
		//FileFormatSaftyChk(CHostEntryData, fRealHostDNSCacheID,		628);
		//FileFormatSaftyChk(CHostEntryData, fResolveHostDNSCacheID,	632);
		//FileFormatSaftyChk(CHostEntryData, fDBFooter,				636);
//		this->ReportBackwardCompatibility(sizeof(CHostEntryData),	640,	"CHostEntryData", "SIZEOF()");
//		this->ReportBackwardCompatibility(kHostEntryDataSize, 640, "CHostEntryData", "kHostEntryDataSize");

		/*
			//-----------------------------------------------------------------------------------------
			// now lets check the embedded struct, to make sure it hasn't realigned itself, but kept the
			// same size...
			Type Name:  HostStatistics		TINFO offset: 55426		(TINFO SYM file offset (hex): $125882)
				RecordOf
				offset 0 NamedTypeOf numMessages unsigned long 
				offset 4 NamedTypeOf numMessagesThisHour 
				offset 8 NamedTypeOf numRecipients unsigned long 
				offset 12 NamedTypeOf numBytesTransferred 
				offset 16 NamedTypeOf numSuccessfulConnections unsigned long 
				offset 20 NamedTypeOf numFailedConnections 
				offset 24 NamedTypeOf lastConnectionSuccess unsigned long 
				offset 28 NamedTypeOf lastConnectionFailure 
				offset 32 NamedTypeOf nextConnectionAttempt unsigned long 
				offset 36 NamedTypeOf nextDeferralInterval 
				offset 40 NamedTypeOf nextETRNTime unsigned long 
				offset 44 NamedTypeOf totalConnectionTime 
				offset 48 NamedTypeOf numBytesThisConnection unsigned long 
				offset 52 NamedTypeOf fShortFlags 
				offset 56 NamedTypeOf fLongFlags unsigned long 
				offset 60 NamedTypeOf maxMessageSize 
				offset 64 NamedTypeOf numBadRecipients unsigned long 
		*/

		//FileFormatSaftyChk(HostStatistics, numMessages,					0);	
		//FileFormatSaftyChk(HostStatistics, numMessagesThisHour,			4);
		//FileFormatSaftyChk(HostStatistics, numRecipients,				8);
		//FileFormatSaftyChk(HostStatistics, numBytesTransferred,			12);
		//FileFormatSaftyChk(HostStatistics, numSuccessfulConnections,	16);
		//FileFormatSaftyChk(HostStatistics, numFailedConnections,		20);
		//FileFormatSaftyChk(HostStatistics, lastConnectionSuccess,		24);
		//FileFormatSaftyChk(HostStatistics, lastConnectionFailure,		28);
		//FileFormatSaftyChk(HostStatistics, nextConnectionAttempt,		32);
		//FileFormatSaftyChk(HostStatistics, nextDeferralInterval,		36);
		//FileFormatSaftyChk(HostStatistics, nextETRNTime,				40);
		//FileFormatSaftyChk(HostStatistics, totalConnectionTime,			44);
		//FileFormatSaftyChk(HostStatistics, numBytesThisConnection,		48);
		//FileFormatSaftyChk(HostStatistics, fShortFlags,					52);
		//FileFormatSaftyChk(HostStatistics, fLongFlags,					56);
		//FileFormatSaftyChk(HostStatistics, maxMessageSize,				60);
		//FileFormatSaftyChk(HostStatistics, numBadRecipients,			64);
//		this->ReportBackwardCompatibility(sizeof(HostStatistics),	68,	"HostStatistics", "SIZEOF()");
	}
	
	aRunOnceFlag = true;
}
#pragma mark -

ObjID CHostEntry::GetHostEntryID (void)
{
	return (fHostEntryData.fHostEntryID);
}

ObjID CHostEntry::GetMailSpoolID (void)
{
	while ( fHostEntryData.fMailSpoolID == 0 )	// Mailspool does not exist yet
	{
		if (this->GetObjectLock(true) == kNoErr)
		{
			// Create a MailSpool for this HostEntry
			CMailSpool* theMailSpool =  CMailSpool::Create(CMailSpool::kHostSpool, this->GetHostName());

			if (theMailSpool != NULL)
			{
				// Save the MailSpoolID
				this->SetMailSpoolID (theMailSpool->GetMailSpoolID());

				theMailSpool->Done (theMailSpool);
			}

			this->ReleaseObjectLock(true);
		}
	}

	return (fHostEntryData.fMailSpoolID);
}

void CHostEntry::SetMailSpoolID (ObjID inMailSpoolID)
{
	if (fHostEntryData.fMailSpoolID != inMailSpoolID)
	{
		fHostEntryData.fMailSpoolID = inMailSpoolID;
		this->SetDirty();
	}
}

ObjID CHostEntry::GetRealHostDNSCacheID (void)
{
	return (fHostEntryData.fRealHostDNSCacheID);
}

void CHostEntry::SetRealHostDNSCacheID (ObjID inHostDNSCacheID)
{
	if (fHostEntryData.fRealHostDNSCacheID != inHostDNSCacheID)
	{
		fHostEntryData.fRealHostDNSCacheID = inHostDNSCacheID;
		this->SetDirty();
	}
}

ObjID CHostEntry::GetResolveHostDNSCacheID (void)
{
	return (fHostEntryData.fResolveHostDNSCacheID);
}

void CHostEntry::SetResolveHostDNSCacheID (ObjID inHostDNSCacheID)
{
	if (fHostEntryData.fResolveHostDNSCacheID != inHostDNSCacheID)
	{
		fHostEntryData.fResolveHostDNSCacheID = inHostDNSCacheID;
		this->SetDirty();
	}
}

#pragma mark -

OSType CHostEntry::GetAddressType(void)
{
	return (fHostEntryData.fAddressType);
}

void CHostEntry::SetAddressType(OSType inAddressType)
{
	if (fHostEntryData.fAddressType != inAddressType)
	{
		fHostEntryData.fAddressType = inAddressType;
		this->SetDirty();
	}
}

uInt32 CHostEntry::GetDestination (void)
{
	return (fHostEntryData.fDestination);
}

void CHostEntry::SetDestination (uInt32 inDestination)
{
	if (fHostEntryData.fDestination != inDestination)
	{
		fHostEntryData.fDestination = inDestination;
		this->SetDirty();
	}
}

uInt32 CHostEntry::GetDestClass (void)
{
	return (fHostEntryData.fDestinationClass);
}

void CHostEntry::SetDestClass (uInt32 inDestClass)
{
	if ((inDestClass != fHostEntryData.fDestinationClass) && (inDestClass != kBounceDestination))
	{
		fHostEntryData.fDestinationClass = inDestClass;
		this->SetDirty();
	}
}

ObjID CHostEntry::GetHostPrefID (void)
{
	return (fHostEntryData.fHostPrefsID);
}

void CHostEntry::SetHostPrefID (const ObjID inHostPrefID)
{
	if (inHostPrefID != fHostEntryData.fHostPrefsID)
	{
		fHostEntryData.fHostPrefsID = inHostPrefID;
		this->SetDirty();
	}
}

const char* CHostEntry::GetHostName(void)
{
	return (fHostEntryData.fHostName);
}

void CHostEntry::SetHostName(const char* inHostName)
{
	CUtils::Strncpy(fHostEntryData.fHostName, (char *) inHostName, sizeof(InetDomainName));
	this->SetDirty();
}

uTime_t CHostEntry::GetAccessTime (void)
{
	return (fHostEntryData.fAccessTime);
}

void CHostEntry::UpdateAccessTime (void)
{
	fHostEntryData.fAccessTime = ::time(NULL);
	this->SetDirty();
}

const char* CHostEntry::GetResolveHostName(void)
{
	return (fHostEntryData.fResolveHostName);
}

Boolean CHostEntry::SetResolveHostName( const char* inHostName, Boolean &outBadName )
{
	Boolean changed		= (CUtils::Stricmp(fHostEntryData.fResolveHostName, inHostName) != 0);
	Boolean goodName	= true;
	uInt32	nameLength	= 0;
	uInt32	index		= 0;

	outBadName = false;
	if (changed == true)
	{
		nameLength = CUtils::Strlen(inHostName);
		goodName = (goodName == true) && (CUtils::Strlen(inHostName) != 0);
		for (index = 0; (goodName == true) && (index < nameLength); index++)
		{
			goodName = (goodName == true) && ((inHostName[index] > ' ') && (inHostName[index] <= '~'));
		}

		if (goodName == true)
		{
			CUtils::Strncpy (fHostEntryData.fResolveHostName, (char *) inHostName, sizeof(InetDomainName));
			this->SetDirty();
			outBadName = false;
		}
		else if (CUtils::Stricmp(fHostEntryData.fResolveHostName, fHostEntryData.fHostName) != 0)
		{
			CUtils::Strncpy(fHostEntryData.fResolveHostName, fHostEntryData.fHostName, sizeof(InetDomainName));
			this->SetDirty();
			outBadName = true;
		}
		else
		{
			outBadName = true;
		}
	}

	return changed;
}

void CHostEntry::SetNextAttemptTime ( const uTime_t inNextAttemptTime )
{
	if ( fHostEntryData.fHostStatistics.nextConnectionAttempt != inNextAttemptTime )
	{
		fHostEntryData.fHostStatistics.nextConnectionAttempt = inNextAttemptTime;
		this->SetDirty( true );
	}
}

#pragma mark -

Boolean	CHostEntry::ResolveTCPHost (	CSMTPClient *inSMTPClientPtr,
										const eDNSCacheNameType	inDNSCacheType,
										const Boolean			inResolveMXIPFlag,
										const Boolean			inPrintCacheStatus )
{
}


#if USE_APPLETALK
#pragma mark -

Boolean	CHostEntry::ResolveADSPHost (void)
{
	DDPAddress theDDPAddress;

	Try_
	{
		ThrowIfOSErr_( CNetworkUtilities::ResolveToDDPAddress( (const uInt8 *)fHostEntryData.fResolveHostName, &theDDPAddress ) );

		this->SetDDPAddress( &theDDPAddress );

		return( true );
	}

	Catch_( err )
	{
		this->FailedConnect(::time(NULL), (char*)this->GetResolveHostName(), kSLSmtpADSPConnectFail, 0, true);
		return false;
	}
}

void CHostEntry::GetDDPAddress (DDPAddress* inDDPAddress)
{
	memcpy (inDDPAddress, &fDDPAddress, sizeof (DDPAddress));
}

void CHostEntry::SetDDPAddress (DDPAddress* inDDPAddress)
{
	memcpy (&fDDPAddress, inDDPAddress, sizeof (DDPAddress));
	this->SetDirty();
}
#endif // USE_APPLETALK
#pragma mark -

void CHostEntry::Post (ObjID inEnvelopeID)
{
}

void CHostEntry::DoExpire ( void )
{
}



//--------------------------------------------------------------------------------------------------
//	* GetGoPostal
//
//--------------------------------------------------------------------------------------------------

Boolean CHostEntry::GetGoPostal ( void )
{
	return( fGoPostal );
} // GetGoPostal



//--------------------------------------------------------------------------------------------------
//	* SetGoPostal
//
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetGoPostal ( Boolean inGoPostal )
{
	fGoPostal = inGoPostal;
} // SetGoPostal



//--------------------------------------------------------------------------------------------------
//	* GetDeleteThisHost
//
//--------------------------------------------------------------------------------------------------

Boolean CHostEntry::GetDeleteThisHost ( void )
{
	return( fDeleteThisHost );
} // GetDeleteThisHost



//--------------------------------------------------------------------------------------------------
//	* SetDeleteThisHost
//
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetDeleteThisHost ( Boolean inDeleteThisHost )
{
	// we want to stop processing this host
	this->SetGoPostal( true );

	// set the delete flag
	fDeleteThisHost = inDeleteThisHost;

} // SetDeleteThisHost



#pragma mark -

void CHostEntry::ResetStatistics (void)
{
	memset(&fHostEntryData.fHostStatistics, 0, sizeof (HostStatistics));
	this->SetDirty();
}

HostStatistics* CHostEntry::GetStatistics (void)
{
	return (&fHostEntryData.fHostStatistics);
}

void CHostEntry::SuccessfulConnect( uTime_t inTime, char* inResolveHostName, uInt32 inTransport, const uInt32 inPort2Use )
{
}

void CHostEntry::CalcNextAttemptTime ( const uTime_t inTime )
{
	DSHostPrefs	   *pHostPrefs	= NULL;
	uInt32			aSleepDelta = 0;

	pHostPrefs = DSMgr::GetHostPrefsObj();
	if ( pHostPrefs != NULL )
	{
		pHostPrefs->GetRetryFailedMinutes( &aSleepDelta );
		aSleepDelta = aSleepDelta * 60;

		this->SetNextAttemptTime( inTime + aSleepDelta );
	}
	else
	{
		// Start at 5 minutes and back off until we reach the maximum interval
		fHostEntryData.fHostStatistics.nextDeferralInterval += kDeferralInterval;
		if ( fHostEntryData.fHostStatistics.nextDeferralInterval > kMaxDeferralInterval )
		{
			fHostEntryData.fHostStatistics.nextDeferralInterval = kMaxDeferralInterval;
		}

		this->SetNextAttemptTime( (uTime_t)(inTime + ((fHostEntryData.fHostStatistics.nextDeferralInterval) * 60)) );
	}
} // CalcNextAttemptTime



//--------------------------------------------------------------------------------------------------
//	* GetNextETRNTime
//
//--------------------------------------------------------------------------------------------------

uTime_t CHostEntry::GetNextETRNTime ( void )
{
	return( fHostEntryData.fHostStatistics.nextETRNTime );
} // GetNextETRNTime



//--------------------------------------------------------------------------------------------------
//	* SetNextETRNTime
//
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetNextETRNTime ( uTime_t inNextTime )
{
} // SetNextETRNTime



//--------------------------------------------------------------------------------------------------
//	* CalcNextETRNTime
//
//--------------------------------------------------------------------------------------------------

void CHostEntry::CalcNextETRNTime ( const uTime_t inTime )
{
} // CalcNextETRNTime



//--------------------------------------------------------------------------------------------------
//	* IsETRNOn
//
//--------------------------------------------------------------------------------------------------

Boolean CHostEntry::IsETRNOn ( void )
{
	Boolean			result			= false;
	return( result );

} // IsETRNOn



void CHostEntry::FailedConnect( uTime_t			inTime,
								char		   *inResolveHostName,
								uInt32			inTransport,
								const uInt32	inPortNumber,
								const Boolean	inCalcNextAttempt )
{
}

void CHostEntry::ConnectionFinished ( char* inResolveHostName, uInt32 inMsgs, uInt32 inKBytes,
									uInt32 inRecipientCount, uTime_t inTime, uInt32 inTransport)
{
}

void CHostEntry::ETRNConnectionFinished ( char* inResolveHostName, const uTime_t inTime )
{
	uInt32		deltaTime		= 0;
	CString		aTimeString( 64 );

	if ( inTime < this->GetLastSuccessConnectTime( true ) )
	{
		deltaTime = 1;
	}
	else
	{
		deltaTime = inTime - this->GetLastSuccessConnectTime( true );
		if ( deltaTime == 0 )
		{
			deltaTime = 1;
		}
	}

	CUtils::FormatElapsedTimeFromSecs( deltaTime, aTimeString );

	if (this->GetLastSuccessConnectTime(true) != 0)
	{
		fHostEntryData.fHostStatistics.totalConnectionTime += deltaTime;

		this->CalcNextETRNTime( inTime );
		this->SetDirty();
	}
}

uTime_t CHostEntry::NextConnectAt( void )
{
	return ( fHostEntryData.fHostStatistics.nextConnectionAttempt );
}


void CHostEntry::StatAddMessage(uInt32 inMessageSize)
{
	fHostEntryData.fHostStatistics.numMessages++;
	fHostEntryData.fHostStatistics.numMessagesThisHour++;
	if (inMessageSize > fHostEntryData.fHostStatistics.maxMessageSize)
		fHostEntryData.fHostStatistics.maxMessageSize = inMessageSize;

	this->SetDirty();
}

void CHostEntry::StatAddRecipients(uInt32 inRecipients)
{
	fHostEntryData.fHostStatistics.numRecipients += inRecipients;
	this->SetDirty();
}

void CHostEntry::StatAddBadRecipient(void)
{
	fHostEntryData.fHostStatistics.numBadRecipients++;
	this->SetDirty();
}

void CHostEntry::StatAddBytes (uInt32 inBytes)
{
	fHostEntryData.fHostStatistics.numBytesTransferred += inBytes;
	fHostEntryData.fHostStatistics.numBytesThisConnection += inBytes;
	this->SetDirty();
}


//--------------------------------------------------------------------------------------------------
//	* SetShortFlags
//	
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetShortFlags ( uInt16 inShortFlags )
{
	if ( fHostEntryData.fHostStatistics.fShortFlags != inShortFlags )
	{
		fHostEntryData.fHostStatistics.fShortFlags = inShortFlags;
		this->SetDirty();
	}
} // SetShortFlags


//--------------------------------------------------------------------------------------------------
//	* GetShortFlags
//	
//--------------------------------------------------------------------------------------------------

uInt16 CHostEntry::GetShortFlags ( void )
{
	return ( fHostEntryData.fHostStatistics.fShortFlags );
} // GetShortFlags


//--------------------------------------------------------------------------------------------------
//	* SetLongFlags
//	
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetLongFlags ( uInt32 inLongFlags )
{
	if ( fHostEntryData.fHostStatistics.fLongFlags != inLongFlags )
	{
		fHostEntryData.fHostStatistics.fLongFlags = inLongFlags;
		this->SetDirty();
	}
} // SetLongFlags


//--------------------------------------------------------------------------------------------------
//	* GetLongFlags
//	
//--------------------------------------------------------------------------------------------------

uInt32 CHostEntry::GetLongFlags ( void )
{
	return ( fHostEntryData.fHostStatistics.fLongFlags );
} // GetLongFlags


//--------------------------------------------------------------------------------------------------
//	* GetLastSuccessConnectTime
//	
//--------------------------------------------------------------------------------------------------

uTime_t CHostEntry::GetLastSuccessConnectTime ( const Boolean inAllowFakeNeverConnectTimeFlag )
{
	if ( inAllowFakeNeverConnectTimeFlag == true )
	{
		if ( fHostEntryData.fHostStatistics.lastConnectionSuccess == this->GetLastFailedConnectTime( false ) )
		{
			return( 0 );	// pretend we've never had a connection...
		}
	}
	
	return ( fHostEntryData.fHostStatistics.lastConnectionSuccess );

} // GetLastSuccessConnectTime


//--------------------------------------------------------------------------------------------------
//	* SetLastSuccessConnectTime
//	
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetLastSuccessConnectTime ( const uTime_t inTime )
{
	if (inTime != fHostEntryData.fHostStatistics.lastConnectionSuccess)
	{
		fHostEntryData.fHostStatistics.lastConnectionSuccess = inTime;
		this->SetDirty();
	}
} // SetLastSuccessConnectTime


//--------------------------------------------------------------------------------------------------
//	* GetLastFailedConnectTime
//	
//--------------------------------------------------------------------------------------------------

uTime_t CHostEntry::GetLastFailedConnectTime ( const Boolean inAllowFakeNeverConnectTimeFlag )
{
	if (inAllowFakeNeverConnectTimeFlag == true)
	{
		if (fHostEntryData.fHostStatistics.lastConnectionFailure == this->GetLastSuccessConnectTime(false))
		{
			return (0);	// pretend we've never had a connection...
		}
	}

	return ( fHostEntryData.fHostStatistics.lastConnectionFailure );
} // GetLastFailedConnectTime


//--------------------------------------------------------------------------------------------------
//	* SetLastFailedConnectTime
//	
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetLastFailedConnectTime ( const uTime_t inTime )
{
	if (inTime != fHostEntryData.fHostStatistics.lastConnectionFailure)
	{
		fHostEntryData.fHostStatistics.lastConnectionFailure = inTime;
		this->SetDirty();
	}
} // SetLastFailedConnectTime


//--------------------------------------------------------------------------------------------------
//	* GetIgnoreSchedule
//	
//--------------------------------------------------------------------------------------------------

Boolean CHostEntry::GetIgnoreSchedule ( void )
{
	return ( fHostEntryData.fIgnoreSchedule );
} // GetIgnoreSchedule



//--------------------------------------------------------------------------------------------------
//	* SetIgnoreSchedule
//	
//--------------------------------------------------------------------------------------------------

void CHostEntry::SetIgnoreSchedule ( Boolean inIgnoreIt )
{
	// Until we have an HI it is alwasy false

	inIgnoreIt = false;

	if ( fHostEntryData.fIgnoreSchedule != inIgnoreIt )
	{
		fHostEntryData.fIgnoreSchedule = inIgnoreIt;
		this->SetDirty();
	}
} // SetIgnoreSchedule

